<?php

/**
 * Template helpers fucntions
 *
 * @package Flicker
 * @since 0.0.1
 */

if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly
}

if (!function_exists('flicker_get_body_schema_itemtype')) :

	/**
	 * Get itemtype value for body tag
	 *
	 * @since 0.0.1
	 */
	function flicker_get_body_schema_itemtype()
	{

		//Get default itemtype
		$flicker_itemtype = (is_page()) ? 'WebPage' : 'Blog';

		//Change itemtype if is search page
		if (is_search()) {
			$flicker_itemtype = 'SearchResultsPage';
		}

		return $flicker_itemtype;
	}
endif;

//get theme template part
if (!function_exists('flicker_template_part')) :
	function flicker_template_part($flicker_template_part_name, $flicker_default_value = '1')
	{
		$flicker_return = flicker_option($flicker_template_part_name, $flicker_default_value);

		//for demo
		if (!empty($_GET[$flicker_template_part_name])) {
			$flicker_return = absint($_GET[$flicker_template_part_name]);
		}

		return $flicker_return;
	}
endif;

//get proper CSS classes for #main section based on page template
if (!function_exists('flicker_get_page_main_section_css_classes')) :
	function flicker_get_page_main_section_css_classes()
	{
		$return = 'width-inherit';
		if (is_page_template('page-templates/no-sidebar-720.php')) {
			$return = 'container-720';
		}
		if (is_page_template('page-templates/no-sidebar-960.php')) {
			$return = 'container-960';
		}
		if (is_page_template('page-templates/no-sidebar-1170.php')) {
			$return = 'container-1170';
		}

		return $return;
	}
endif;

//get proper CSS classes for main column, aside column and body
if (!function_exists('flicker_get_layout_css_classes')) :
	function flicker_get_layout_css_classes()
	{

		//default - sidebar
		$flicker_return = array(
			'body'  => 'with-sidebar',
			'main'  => 'column-main',
			'aside' => 'column-aside',
		);

		//check for shop
		if (function_exists('is_woocommerce') && is_woocommerce()) :
			if (is_product()) {
				$flicker_shop_sidebar_position_option = flicker_option('product_sidebar_position', 'right');
			}
			if (is_shop() || is_product_taxonomy()) {
				$flicker_shop_sidebar_position_option = flicker_option('shop_sidebar_position', 'right');
			}
			//if empty sidebar or disabled in customizer - removing aside
			if (!is_active_sidebar('shop') || 'no' === $flicker_shop_sidebar_position_option) {
				$flicker_return['body']  = 'no-sidebar';
				$flicker_return['aside'] = false;

				return $flicker_return;
			} //is_active_sidebar( 'shop' )
			//left sidebar
			if ('left' === $flicker_shop_sidebar_position_option) {
				$flicker_return['body'] .= ' sidebar-left';

				return $flicker_return;
				//default - right sidebar
			} else {
				return $flicker_return;
			}
		endif; //is_woocommerce()

		//if category has meta - overriding default customizer option option
		if (is_category()) {
			$flicker_sidebar_position_option = flicker_get_category_sidebar_position();
		} else {
			if (!is_single()) {
				$flicker_sidebar_position_option = flicker_option('blog_sidebar_position', 'right');
			} else {
				$flicker_sidebar_position_option = flicker_option('blog_single_sidebar_position', 'right');
			}
		} //is_category

		if (!is_page_template('page-templates/home.php')) {
			$sidebar = 'sidebar-1';
			if (is_front_page() && is_active_sidebar('sidebar-home-main')) {
				$sidebar = 'sidebar-home-main';
			}
			//if empty sidebar - removing aside
			if (!is_active_sidebar($sidebar)) {
				$flicker_return['body']  = 'no-sidebar';
				$flicker_return['aside'] = false;

				return $flicker_return;
			} //sidebar-1
		} else {
			//if empty sidebar on home.php page template - removing aside
			if (!is_active_sidebar('sidebar-home-main')) {
				$flicker_return['body']  = 'no-sidebar';
				$flicker_return['aside'] = false;

				return $flicker_return;
			} //sidebar-nome-main
		} //! is_page_template( 'page-templates/home.php'

		//various cases with sidebar
		//single post without sidebars
		if (is_single()) {

			//no sidebar for posts layouts
			if (
				is_page_template('page-templates/post-full-width-no-meta-no-thumbnail.php')
				||
				is_page_template('page-templates/post-full-width-no-meta.php')
				||
				is_page_template('page-templates/post-full-width.php')
			) {
				$flicker_return['body']  = 'no-sidebar';
				$flicker_return['aside'] = false;

				return $flicker_return;
			}
		} //is_single

		//pages
		if (is_page()) :

			//no sidebar
			if (
				is_page_template('page-templates/full-width.php')
				||
				is_page_template('page-templates/empty-page.php')
				||
				is_page_template('page-templates/empty-page-container.php')
				||
				is_page_template('page-templates/no-sidebar-720.php')
				||
				is_page_template('page-templates/no-sidebar-960.php')
				||
				is_page_template('page-templates/no-sidebar-1170.php')
				||
				is_page_template('page-templates/no-sidebar-no-title.php')
				||
				is_page_template('page-templates/no-sidebar-no-padding.php')
				||
				!is_page_template()
			) {
				$flicker_return['body']  = 'no-sidebar';
				$flicker_return['aside'] = false;

				return $flicker_return;
			}

			//left sidebar for page
			if (
				is_page_template('page-templates/sidebar-left.php')
				||
				('left' === $flicker_sidebar_position_option)
			) {
				$flicker_return['body'] .= ' sidebar-left';

				return $flicker_return;
			}

		//right sidebar is default
		endif; // is_page

		//if no sidebar option - removing aside
		if ('no' === $flicker_sidebar_position_option && !(is_page_template('page-templates/home.php'))) {
			$flicker_return['body']  = 'no-sidebar';
			$flicker_return['aside'] = false;

			return $flicker_return;
		}

		//left sidebar
		if ('left' === $flicker_sidebar_position_option) {
			$flicker_return['body'] .= ' sidebar-left';
		}

		return $flicker_return;
	}
endif;

//get category layout based on category meta with global blog option as fallback
if (!function_exists('flicker_get_category_layout')) :
	function flicker_get_category_layout()
	{
		$flicker_layout = '';

		$flicker_queried_object = get_queried_object();
		$flicker_term_id        = $flicker_queried_object->term_id;

		//if layout is overriden for category in admin panel
		$flicker_term_metas_layout = get_term_meta($flicker_term_id, 'layout', true);
		if (!empty($flicker_term_metas_layout)) {
			$flicker_layout = $flicker_term_metas_layout;
		}

		//if category layout not specified - getting default layout
		if (empty($flicker_layout)) {
			$flicker_layout = flicker_option('blog_layout', '') ? flicker_option('blog_layout', '') : 'default';
		}

		return $flicker_layout;
	}
endif;

//get category layout gap based on category meta with global blog option as fallback
if (!function_exists('flicker_get_category_layout_gap')) :
	function flicker_get_category_layout_gap()
	{
		$flicker_layout_gap = '';

		$flicker_queried_object = get_queried_object();
		$flicker_term_id        = $flicker_queried_object->term_id;

		//if layout is overriden for category in admin panel
		$flicker_term_metas_layout = get_term_meta($flicker_term_id, 'gap', true);
		if (!empty($flicker_term_metas_layout)) {
			$flicker_layout_gap = $flicker_term_metas_layout;
		}

		//if category layout not specified - getting default layout
		if (empty($flicker_layout_gap)) {
			$flicker_layout_gap = flicker_option('blog_layout_gap', '') ? flicker_option('blog_layout_gap', '') : '';
		}

		return $flicker_layout_gap;
	}
endif;

//get feed shot_title
if (!function_exists('flicker_get_feed_shot_title')) :
	function flicker_get_feed_shot_title()
	{
		if (is_category()) {
			$flicker_show_title = !flicker_option('title_show_title', '');
		} else {
			$flicker_show_title = !flicker_option('title_show_title', '') && !is_front_page();
		}

		return $flicker_show_title;
	}
endif;

//get feed layout
if (!function_exists('flicker_get_feed_layout')) :
	function flicker_get_feed_layout()
	{
		if (is_category()) {
			$flicker_layout = flicker_get_category_layout();
		} else {
			$flicker_layout = flicker_option('blog_layout', '') ? flicker_option('blog_layout', '') : 'default';
		}

		//override option for demo purposes
		if (isset($_GET['blog_layout'])) {
			$flicker_layout_id = absint($_GET['blog_layout']);
			$flicker_layouts   = array_keys(flicker_get_feed_layout_options());
			$flicker_layout    = !empty($flicker_layouts[$flicker_layout_id]) ? $flicker_layouts[$flicker_layout_id] : $flicker_layout;
		}

		return $flicker_layout;
	}
endif;

//get feed gap
if (!function_exists('flicker_get_feed_gap')) :
	function flicker_get_feed_gap()
	{
		if (is_category()) {
			$flicker_layout_gap = flicker_get_category_layout_gap();
		} else {
			$flicker_layout_gap = flicker_option('blog_layout_gap', '') ? flicker_option('blog_layout_gap', '') : '';
		}

		//override option for demo purposes
		if (isset($_GET['blog_layout_gap'])) {
			$flicker_layout_gap_id = absint($_GET['blog_layout_gap']);
			$flicker_layout_gaps   = array_keys(flicker_get_feed_layout_gap_options());
			$flicker_layout_gap    = !empty($flicker_layout_gaps[$flicker_layout_gap_id]) ? $flicker_layout_gaps[$flicker_layout_gap_id] : $flicker_layout_gap;
		}

		return $flicker_layout_gap;
	}
endif;

//get category sidebar_position based on category meta with global blog option as fallback
if (!function_exists('flicker_get_category_sidebar_position')) :
	function flicker_get_category_sidebar_position()
	{
		$flicker_sidebar_position = '';

		$flicker_queried_object = get_queried_object();
		$flicker_term_id        = $flicker_queried_object->term_id;

		//term metas from category options has higher priority than customizer option for special categories
		$flicker_term_metas = get_term_meta($flicker_term_id, 'sidebar_position', true);
		if (!empty($flicker_term_metas)) {
			$flicker_sidebar_position = $flicker_term_metas;
		}

		//if category sidebar_position not specified - getting default sidebar_position
		if (empty($flicker_sidebar_position)) {
			$flicker_sidebar_position = flicker_option('blog_sidebar_position', '') ? flicker_option('blog_sidebar_position', '') : 'right';
		}

		return $flicker_sidebar_position;
	}
endif;

//get single post layout based on blog post option
if (!function_exists('flicker_get_post_layout')) :
	function flicker_get_post_layout()
	{

		$flicker_layout = flicker_option('blog_single_layout', '') ? flicker_option('blog_single_layout', '') : 'default';

		//override option for demo purposes
		if (isset($_GET['blog_single_layout'])) {
			$flicker_layout_id = absint($_GET['blog_single_layout']);
			$flicker_layouts   = array_keys(flicker_get_post_layout_options());
			$flicker_layout    = !empty($flicker_layouts[$flicker_layout_id]) ? $flicker_layouts[$flicker_layout_id] : $flicker_layout;
		}

		return $flicker_layout;
	}
endif;

if (!function_exists('flicker_body_classes')) :
	/**
	 * Adds custom classes to the array of body classes.
	 *
	 * @param array $flicker_classes Classes for the body element.
	 *
	 * @return array
	 */
	function flicker_body_classes($flicker_classes)
	{
		//header-empty
		if (is_page_template('page-templates/empty-page.php')) {
			$flicker_classes[] = 'header-empty';
		}

		//header sticky
		if (has_nav_menu('side') || is_active_sidebar('sidebar-side')) {
			$flicker_classes[] = 'has-side-nav';
			$flicker_classes[] = flicker_option('side_nav_position', '') ? 'side-nav-right' : 'side-nav-left';
			$flicker_classes[] = flicker_option('side_nav_sticked', '') ? 'side-nav-sticked' : '';
			$flicker_classes[] = flicker_option('side_nav_header_overlap', '') ? 'side-nav-header-overlap' : '';
		}

		// Adds a class of hfeed to non-singular pages.
		if (!is_singular()) {
			$flicker_classes[] = 'hfeed';
		} else {
			//add 'singular' class for single post or page or any other post type
			$flicker_classes[] = 'singular';
		}

		//Adds a sidebar classes
		$flicker_css_classes = flicker_get_layout_css_classes();

		$flicker_classes[] = $flicker_css_classes['body'];

		//Add icons in meta classes
		//single post
		if (is_singular()) {

			$flicker_hide_meta_icons = flicker_option('blog_single_hide_meta_icons', false);

			//blog loop
		} else {

			$flicker_hide_meta_icons = flicker_option('blog_hide_meta_icons', false);
		}
		if ($flicker_hide_meta_icons) {
			$flicker_classes[] = 'meta-icons-hidden';
		}

		//container width
		$flicker_container_width      = flicker_option('main_container_width', '1170');
		$flicker_container_post_width = flicker_option('blog_single_container_width', '');
		$flicker_container_blog_width = flicker_option('blog_container_width', '');
		if (is_singular('post') && !empty($flicker_container_post_width)) {
			$flicker_container_width = $flicker_container_post_width;
		}
		if ((is_home() || is_category() || is_tag() || is_date() || is_author()) && !empty($flicker_container_blog_width)) {
			$flicker_container_width = $flicker_container_blog_width;
		}
		if ('1520' === $flicker_container_width) {
			$flicker_classes[] = 'container-1520';
		}
		if ('1170' === $flicker_container_width) {
			$flicker_classes[] = 'container-1170';
		}
		if ('960' === $flicker_container_width) {
			$flicker_classes[] = 'container-960';
		}
		if ('720' === $flicker_container_width) {
			$flicker_classes[] = 'container-720';
		}

		//meta icons color class
		$flicker_meta_icons_color = flicker_option('color_meta_icons', '');
		if ($flicker_meta_icons_color) {
			$flicker_classes[] = esc_attr($flicker_meta_icons_color);
		}

		//shop class
		if (class_exists('WooCommerce')) {
			$flicker_classes[] = 'woo';
		}

		//header class
		if ('always-sticky' === flicker_option('header_sticky', '')) {
			$flicker_classes[] = 'header-sticky';
		}

		//animation enabled
		$flicker_animation = flicker_option('animation_enabled', '');
		if (!empty($flicker_animation) && !is_customize_preview()) {
			$flicker_classes[] = 'animation-enabled';
		}

		//title section enabled
		$flicker_title = flicker_is_title_section_is_shown();
		if (empty($flicker_title)) {
			$flicker_classes[] = 'title-hidden';
		}

		return $flicker_classes;
	}
endif;
add_filter('body_class', 'flicker_body_classes');

//markup for animated page elements
if (!function_exists('flicker_animated_elements_markup')) :
	function flicker_animated_elements_markup()
	{
		$flicker_animation = flicker_option('animation_enabled', '');
		if (empty($flicker_animation)) {
			return;
		}

		//get animations array from customizer. Keys - selectors
		$flicker_animations = array(
			'.column-aside .widget'            => flicker_option('animation_sidebar_widgets', ''),
			'.footer-top-widgets .widget'      => flicker_option('animation_footer_top_widgets', ''),
			'.footer-widgets .widget'          => flicker_option('animation_footer_widgets', ''),
			'.hfeed article.post'              => flicker_option('animation_feed_posts', ''),
			'.hfeed .post .post-thumbnail img' => flicker_option('animation_feed_posts_thumbnail', ''),
		);

		$flicker_animations = array_filter($flicker_animations);
		if (!empty($flicker_animations) && !is_customize_preview()) :
?>
			data-animate='<?php echo esc_attr(str_replace('&quot;', '"', json_encode($flicker_animations))); ?>'
		<?php
		endif;
	}
endif;

//markup for sticky post label
if (!function_exists('flicker_sticky_post_label')) :
	function flicker_sticky_post_label()
	{
		if (is_sticky() && is_home() && !is_paged()) :
		?>
			<span class="icon-inline sticky-post">
				<span><?php echo esc_html_x('Featured', 'post', 'flicker'); ?></span>
			</span><!-- .sticky-post -->
			<?php
		endif; //is_sticky()
	}
endif;

//arguments for link pages
if (!function_exists('flicker_get_wp_link_pages_atts')) :
	function flicker_get_wp_link_pages_atts()
	{
		return apply_filters(
			'flicker_link_pages_atts',
			array(
				'before'      => '<div class="page-links"><span class="screen-reader-text">' . esc_html__('Pages: ', 'flicker') . '</span>',
				'after'       => '</div>',
				'link_before' => '<span>',
				'link_after'  => '</span>',
			)
		);
	}
endif;

//arguments for link pages
if (!function_exists('flicker_get_the_posts_pagination_atts')) :
	function flicker_get_the_posts_pagination_atts()
	{
		return array(
			'mid_size'  => 5,
			'prev_text' => '<span class="screen-reader-text">' . esc_html__('Previous page', 'flicker') . '</span><span class="icon-inline">' . flicker_icon('chevron-left', true) . '</span>',
			'next_text' => '<span class="screen-reader-text">' . esc_html__('Next page', 'flicker') . '</span><span class="icon-inline">' . flicker_icon('chevron-right', true) . '</span>',
		);
	}
endif;

//get top level menu items count
if (!function_exists('flicker_get_menu_top_level_items_count')) :
	function flicker_get_menu_top_level_items_count($flicker_menu_name)
	{

		$flicker_locations   = get_nav_menu_locations();
		$flicker_menu_id     = $flicker_locations[$flicker_menu_name];
		$flicker_menu_object = wp_get_nav_menu_object($flicker_menu_id);
		if (empty($flicker_menu_object)) {
			return '-1';
		}

		$flicker_menu_items       = wp_get_nav_menu_items($flicker_menu_object->term_id);
		$flicker_menu_items_count = 0;

		foreach ((array) $flicker_menu_items as $flicker_key => $flicker_menu_item) {
			if ('0' === $flicker_menu_item->menu_item_parent) {
				$flicker_menu_items_count++;
			}
		}

		return $flicker_menu_items_count;
	}
endif;

//get menu class depending on menu top level items count
if (!function_exists('flicker_get_menu_class_based_on_top_items_count')) :
	function flicker_get_menu_class_based_on_top_items_count($flicker_menu_name)
	{

		$flicker_menu_items_count = flicker_get_menu_top_level_items_count($flicker_menu_name);
		if ('-1' === $flicker_menu_items_count) {
			return 'menu-empty';
		}

		$flicker_css_class = 'menu-low-items';

		if ($flicker_menu_items_count > 5) {
			$flicker_css_class = 'menu-many-items';
		}

		return $flicker_css_class;
	}
endif;


//print svg icon
if (!function_exists('flicker_icon')) :
	function flicker_icon($flicker_name, $flicker_return = false, $flicker_container_css_class = 'svg-icon')
	{
		//in the future we'll add option for this
		$flicker_icons_pack = 'google';

		if ($flicker_return) {
			ob_start();
		}

		echo '<span class="' . esc_attr($flicker_container_css_class) . ' icon-' . esc_attr($flicker_name) . '">';
		get_template_part('/template-parts/svg/' . $flicker_icons_pack . '/' . $flicker_name);
		echo '</span>';

		if ($flicker_return) {
			return ob_get_clean();
		}
	}
endif;

//print social link
if (!function_exists('flicker_social_link')) :
	function flicker_social_link($flicker_name, $flicker_url)
	{
		echo '<a href="' . esc_url($flicker_url) . '" target="_blank" class="social-icon social-icon-' . esc_attr($flicker_name) . '">';
		flicker_icon($flicker_name);
		echo '</a>';
	}
endif;

//meta
//get meta array
if (!function_exists('flicker_get_theme_meta')) :
	function flicker_get_theme_meta($flicker_meta_names = array())
	{
		/*
		customizer options with meta are:
			'meta_email'
			'meta_email_label'
			'meta_phone'
			'meta_phone_label'
			'meta_phone_2'
			'meta_phone_label_2'
			'meta_address'
			'meta_address_label'
			'meta_opening_hours'
			'meta_opening_hours_label'
		*/

		//if no names specified - using all meta
		if (empty($flicker_meta_names)) :
			$flicker_meta_names = array(
				'email',
				'phone',
				'address',
				'website',
				'opening_hours',
			);
		endif;

		$flicker_theme_meta = array();

		//meta values
		foreach ($flicker_meta_names as $flicker_meta_name) {
			$flicker_value = flicker_option('meta_' . $flicker_meta_name);
			if (!empty($flicker_value)) {
				$flicker_theme_meta[$flicker_meta_name] = $flicker_value;
			}
		}

		//labels for meta if it is not empty
		if (!empty($flicker_theme_meta)) {
			foreach ($flicker_theme_meta as $flicker_meta_name => $flicker_meta_value) {
				$flicker_label = flicker_option('meta_' . $flicker_meta_name . '_label');
				if (!empty($flicker_label)) {
					$flicker_theme_meta[$flicker_meta_name . '_label'] = $flicker_label;
				}
			}
		}

		return $flicker_theme_meta;
	}
endif;

//print all social links based on theme_meta from Customizzer
if (!function_exists('flicker_social_links')) :
	function flicker_social_links($wrapper_class = '')
	{

		$flicker_facebook  = flicker_option('meta_facebook');
		$flicker_twitter   = flicker_option('meta_twitter');
		$flicker_linkedin  = flicker_option('meta_linkedin');
		$flicker_youtube   = flicker_option('meta_youtube');
		$flicker_instagram = flicker_option('meta_instagram');
		$flicker_pinterest = flicker_option('meta_pinterest');
		$flicker_github    = flicker_option('meta_github');

		if (
			!empty($flicker_facebook)
			||
			!empty($flicker_twitter)
			||
			!empty($flicker_youtube)
			||
			!empty($flicker_instagram)
			||
			!empty($flicker_pinterest)
			||
			!empty($flicker_linkedin)
			||
			!empty($flicker_github)
		) :

			if ($wrapper_class) {
				echo '<div class="' . esc_attr($wrapper_class) . '">';
			}
			echo '<span class="social-links">';

			if (!empty($flicker_facebook)) :
				flicker_social_link('facebook', $flicker_facebook);
			endif;

			if (!empty($flicker_twitter)) :
				flicker_social_link('twitter', $flicker_twitter);
			endif;

			if (!empty($flicker_linkedin)) :
				flicker_social_link('linkedin', $flicker_linkedin);
			endif;

			if (!empty($flicker_youtube)) :
				flicker_social_link('youtube', $flicker_youtube);
			endif;

			if (!empty($flicker_instagram)) :
				flicker_social_link('instagram', $flicker_instagram);
			endif;

			if (!empty($flicker_pinterest)) :
				flicker_social_link('pinterest', $flicker_pinterest);
			endif;

			if (!empty($flicker_github)) :
				flicker_social_link('github-circle', $flicker_github);
			endif;

			echo '</span><!--.social-links-->';
			if ($wrapper_class) {
				echo '</div><!--' . esc_html($wrapper_class) . '-->';
			}

		endif;
	}
endif;

//print copyright social link
if (!function_exists('flicker_social_link_2')) :
	function flicker_social_link_2($flicker_name, $flicker_url)
	{
		echo '<a href="' . esc_url($flicker_url) . '" target="_blank" class="social-icon color-bg-icon rounded social-icon-' . esc_attr($flicker_name) . '">';
		flicker_icon($flicker_name);
		echo '</a>';
	}
endif;

//print all social links based on theme_meta from Customizzer 
if (!function_exists('flicker_social_links_copyright')) :
	function flicker_social_links_copyright($wrapper_class = '')
	{

		$flicker_twitter   = flicker_option('meta_twitter');
		$flicker_facebook  = flicker_option('meta_facebook');
		$flicker_linkedin  = flicker_option('meta_linkedin');
		$flicker_youtube   = flicker_option('meta_youtube');
		$flicker_instagram = flicker_option('meta_instagram');
		$flicker_pinterest = flicker_option('meta_pinterest');
		$flicker_github    = flicker_option('meta_github');

		if (
			!empty($flicker_facebook)
			||
			!empty($flicker_twitter)
			||
			!empty($flicker_youtube)
			||
			!empty($flicker_instagram)
			||
			!empty($flicker_pinterest)
			||
			!empty($flicker_linkedin)
			||
			!empty($flicker_github)
		) :

			if ($wrapper_class) {
				echo '<div class="' . esc_attr($wrapper_class) . '">';
			}
			echo '<span class="social-links copyright-social-links">';

			if (!empty($flicker_twitter)) :
				flicker_social_link_2('twitter', $flicker_twitter);
			endif;

			if (!empty($flicker_facebook)) :
				flicker_social_link_2('facebook', $flicker_facebook);
			endif;

			if (!empty($flicker_instagram)) :
				flicker_social_link_2('instagram', $flicker_instagram);
			endif;

			if (!empty($flicker_youtube)) :
				flicker_social_link_2('youtube', $flicker_youtube);
			endif;

			if (!empty($flicker_pinterest)) :
				flicker_social_link_2('pinterest', $flicker_pinterest);
			endif;

			if (!empty($flicker_linkedin)) :
				flicker_social_link_2('linkedin', $flicker_linkedin);
			endif;

			if (!empty($flicker_github)) :
				flicker_social_link_2('github-circle', $flicker_github);
			endif;

			echo '</span><!--.social-links-->';
			if ($wrapper_class) {
				echo '</div><!--' . esc_html($wrapper_class) . '-->';
			}

		endif;
	}
endif;

//print social link
if (!function_exists('flicker_social_link_theme_meta_widget')) :
	function flicker_social_link_theme_meta_widget($flicker_name, $flicker_url)
	{
		echo '<a href="' . esc_url($flicker_url) . '" target="_blank" class="social-icon social-icon-' . esc_attr($flicker_name) . '">';
		flicker_icon($flicker_name);
		echo esc_html($flicker_name);
		echo '</a>';
	}
endif;

//print all social links based on theme_meta from Customizzer
if (!function_exists('flicker_social_links_theme_meta_widget')) :
	function flicker_social_links_theme_meta_widget($wrapper_class = '')
	{

		$flicker_twitter   = flicker_option('meta_twitter');
		$flicker_facebook  = flicker_option('meta_facebook');
		$flicker_linkedin  = flicker_option('meta_linkedin');
		$flicker_youtube   = flicker_option('meta_youtube');
		$flicker_instagram = flicker_option('meta_instagram');
		$flicker_pinterest = flicker_option('meta_pinterest');
		$flicker_github    = flicker_option('meta_github');

		if (
			!empty($flicker_facebook)
			||
			!empty($flicker_twitter)
			||
			!empty($flicker_youtube)
			||
			!empty($flicker_instagram)
			||
			!empty($flicker_pinterest)
			||
			!empty($flicker_linkedin)
			||
			!empty($flicker_github)
		) :

			if ($wrapper_class) {
				echo '<div class="' . esc_attr($wrapper_class) . '">';
			}
			echo '<span class="social-links">';

			if (!empty($flicker_twitter)) :
				flicker_social_link_theme_meta_widget('twitter', $flicker_twitter);
			endif;

			if (!empty($flicker_facebook)) :
				flicker_social_link_theme_meta_widget('facebook', $flicker_facebook);
			endif;

			if (!empty($flicker_instagram)) :
				flicker_social_link_theme_meta_widget('instagram', $flicker_instagram);
			endif;

			if (!empty($flicker_youtube)) :
				flicker_social_link_theme_meta_widget('youtube', $flicker_youtube);
			endif;

			if (!empty($flicker_pinterest)) :
				flicker_social_link_theme_meta_widget('pinterest', $flicker_pinterest);
			endif;

			if (!empty($flicker_linkedin)) :
				flicker_social_link_theme_meta_widget('linkedin', $flicker_linkedin);
			endif;

			if (!empty($flicker_github)) :
				flicker_social_link_theme_meta_widget('github-circle', $flicker_github);
			endif;

			echo '</span><!--.social-links-->';
			if ($wrapper_class) {
				echo '</div><!--' . esc_html($wrapper_class) . '-->';
			}

		endif;
	}
endif;

if (!function_exists('flicker_has_post_thumbnail')) :
	/**
	 * Check if has post thumbnail and thumbnail file exists
	 */
	function flicker_has_post_thumbnail($flicker_id = false)
	{
		if (empty($flicker_id)) {
			$flicker_id = get_the_ID();
		}

		return !(post_password_required($flicker_id)
			||
			is_attachment()
			||
			!has_post_thumbnail($flicker_id)
			||
			!file_exists(get_attached_file(get_post_thumbnail_id($flicker_id))));
	}
endif;

if (!function_exists('flicker_post_thumbnail')) :
	/**
	 * Displays an optional post thumbnail.
	 *
	 * Wraps the post thumbnail in an anchor element on index views, or a div
	 * element when on single views.
	 */
	function flicker_post_thumbnail($flicker_size = 'flicker-default-post', $flicker_css_class = '', $show_categories = false)
	{

		if (
			!flicker_has_post_thumbnail()
		) {
			return;
		}

		//detect video
		$oembed_url = false;
		$oembed_post_thumbnail = false;
		//only video post format
		if ('video' === get_post_format()) {
			$oembed_post_thumbnail = true;
		}

		if ($oembed_post_thumbnail) {
			$post_content = get_the_content();
			//get oEmbed URL
			$reg = preg_match('|^\s*(https?://[^\s"]+)\s*$|im', $post_content, $matches);

			$oembed_url = !empty($reg) ? trim($matches[0]) : false;
			//if no youtube, trying to find self hosted

			$first_self_hosted = '';
			$embeds = array();
			if (empty($oembed_url)) {
				$post_content = apply_filters('the_content', $post_content);
				$embeds = get_media_embedded_in_content($post_content);
			}
		}

		if (is_singular()) :
			if (($oembed_url || !empty($embeds[0]))) :
				//if youtube
				if ($oembed_url) :
					add_filter('the_content', function ($content) use ($oembed_url) {
						//remove embed
						$content = str_replace($oembed_url, '', $content);
						//hide embed wrapper
						$pos = strpos($content, 'class="wp-block-embed');
						if ($pos !== false) {
							$content = substr_replace($content, 'class="d-none wp-block-embed', $pos, strlen('class="wp-block-embed'));
						}
						return $content;
						//1 - to run early
					}, 1);
			?>
					<figure class="wp-block-embed wp-embed-aspect-16-9 post-thumbnail mb-0">
						<div class="wp-block-embed__wrapper" itemprop="video" itemscope="itemscope" itemtype="https://schema.org/VideoObject">
							<?php

							echo wp_oembed_get($oembed_url);

							$thumbnail_url = get_the_post_thumbnail_url(get_the_ID(), 'full');

							if ($thumbnail_url) :
							?>
								<meta itemprop="thumbnailUrl" content="<?php echo esc_url($thumbnail_url); ?>">
							<?php
							endif; //thumbnail_url
							?>
							<meta itemprop="uploadDate" content="<?php echo esc_attr(the_time(get_option('date_format'))); ?>">
							<meta itemprop="contentUrl" content="<?php echo esc_url($oembed_url); ?>">
							<?php
							the_title('<h3 class="d-none" itemprop="name">', '</h3>');
							?>
							<p class="d-none" itemprop="description">
								<?php echo wp_kses(get_the_excerpt(), false); ?>
							</p>
						</div>
					</figure><!-- .post-thumbnail -->
				<?php
				//self hosted
				else :
					$embed = (!empty($embeds[0])) ? $embeds[0] : false;
					$url = preg_match('`src="(.*)"`', $embed, $founds);
					$hosted_video_url = !empty($founds['1']) ? $founds['1'] : '';
					add_filter('the_content', function ($content) use ($embed) {
						//remove embed
						$content = str_replace($embed, '', $content);
						//hide embed wrapper
						$pos = strpos($content, 'class="wp-block-video');
						if ($pos !== false) {
							$content = substr_replace($content, 'class="d-none wp-block-video', $pos, strlen('class="wp-block-embed'));
						}
						return $content;
						//1 - to run early
					}, 1);
				?>
					<figure class="post-thumbnail mb-0">
						<div class="wp-block-video" itemprop="video" itemscope="itemscope" itemtype="https://schema.org/VideoObject">
							<?php
							echo wp_kses_post($embed);
							$thumbnail_url = get_the_post_thumbnail_url(get_the_ID(), 'full');
							if ($thumbnail_url) :
							?>
								<meta itemprop="thumbnailUrl" content="<?php echo esc_url($thumbnail_url); ?>">
							<?php
							endif; //thumbnail_url
							?>
							<meta itemprop="uploadDate" content="<?php echo esc_attr(the_time(get_option('date_format'))); ?>">
							<meta itemprop="contentUrl" content="<?php echo esc_url($hosted_video_url); ?>">
							<?php
							the_title('<h3 class="d-none" itemprop="name">', '</h3>');
							?>
							<p class="d-none" itemprop="description">
								<?php echo wp_kses(get_the_excerpt(), false); ?>
							</p>
						</div>
					</figure><!-- .post-thumbnail -->
				<?php
				endif; //$oembed_url
			//not video
			else :
				?>
				<figure class="<?php echo esc_attr('post-thumbnail ' . $flicker_css_class); ?>">
					<?php
					if ($show_categories) {
						flicker_entry_meta(false, false, true, false, false);
					}
					the_post_thumbnail(
						$flicker_size,
						array(
							'itemprop' => 'image',
							'alt'      => get_the_title(),
						)
					);
					?>
				</figure><!-- .post-thumbnail -->
			<?php
			//oembed
			endif;
		//not is_singular
		//archive
		else :

			//detecting gallery
			$is_gallery = false;
			$gallery_css_class = '';
			$image_size = (flicker_option('blog_layout') !== '') ? 'flicker-square' : 'flicker-default-post';
			if ('gallery' === get_post_format()) {
				$galleries_images = get_post_galleries_images();
				//gutenberg block parse
				if (!empty($galleries_images)) {
					global $post;
					if (has_block('gallery', $post->post_content)) {
						$post_blocks = parse_blocks($post->post_content);
						foreach ($post_blocks as $post_block) {
							if ('core/gallery' === $post_block['blockName']) {
								$src_array = array();
								$gallery_imgs_ids = [];
								foreach ($post_block['innerBlocks'] as $subel) {
									$gallery_imgs_ids[] = $subel['attrs']['id'];
								}
								foreach ($gallery_imgs_ids as $id) {
									$src_array[] = wp_get_attachment_image_url($id, $image_size);
								}
								$galleries_images = $src_array;
								break;
							}
						}
					}
				}

				$galleries_images_count = count($galleries_images);
				if ($galleries_images_count) {
					$is_gallery = true;
					$gallery_css_class = 'item-media-gallery';
				}
			} //gallery post format

			?>
			<div class="post-thumbnail-wrap">
				<?php
				if ($show_categories) {
					flicker_entry_meta(false, false, true, false, false, false, false);
				}
				?>
				<figure class="<?php echo esc_attr('post-thumbnail ' . $flicker_css_class); ?>">
					<a href="<?php

								if (empty($oembed_url)) {
									the_permalink();
								} else {
									echo esc_url($oembed_url);
								}

								?>">
						<?php

						if (empty($is_gallery)) {


							the_post_thumbnail(
								$flicker_size,
								array(
									'itemprop' => 'image',
									'alt'      => get_the_title(),
								)
							);
						} else {
							//gallery

							echo '<div class="flexslider"><ul class="slides">';

							//adding featured image as a first element in carousel

							//featured image url
							$post_featured_image_src = wp_get_attachment_image_url(get_post_thumbnail_id(), $image_size);

							if ($post_featured_image_src) : ?>
								<li><img src="<?php echo esc_url($post_featured_image_src); ?>" alt="<?php echo esc_attr(get_the_title()); ?>">
								</li>
							<?php endif;
							$count = 1;

							foreach ($galleries_images as $gallerie) :
								//foreach ($gallerie as $src) :
								//showing only 3 images from gallery
								if ($count > 9) {
									break;
								}
							?>
								<li><img src="<?php echo esc_url($gallerie); ?>" alt="<?php echo esc_attr(get_the_title()); ?>">
								</li>
						<?php
								$count++;
							//endforeach;
							endforeach;
							echo '</ul></div><!--.flexslider-->';
						}
						flicker_post_format_icon(get_post_format());
						?>
					</a>
				</figure>
			</div>
			<!-- post-thumbnail-wrap -->
		<?php
		endif; // End is_singular().
	}
endif; //flicker_post_thumbnail

//print post format icon
if (!function_exists('flicker_post_format_icon')) :
	function flicker_post_format_icon($flicker_post_format = '')
	{
		// 'video', 'audio', 'image', 'gallery', 'quote'
		switch ($flicker_post_format):
			case 'video':
				flicker_icon('video');
				break;
			case 'audio':
				flicker_icon('volume-high');
				break;
			case 'image':
				//flicker_icon('image');
				break;
			case 'gallery':
				flicker_icon('camera');
				break;
			case 'quote':
				flicker_icon('format-quote-close');
				break;

			default:
		endswitch;
	}
endif;


if (!function_exists('flicker_the_author')) :
	/*
		** Prints author HTML with with link on author archive.
		 */
	function flicker_the_author()
	{

		//options
		//single post
		if (is_singular()) {

			$flicker_show_author   = flicker_option('blog_single_show_author', true);
			$flicker_author_avatar = flicker_option('blog_single_show_author_avatar', '');
			$flicker_author_word   = flicker_option('blog_single_before_author_word', '');
			$flicker_show_icons    = !flicker_option('blog_single_hide_meta_icons', false);

			//blog loop
		} else {

			$flicker_show_author   = flicker_option('blog_show_author', true);
			$flicker_author_avatar = flicker_option('blog_show_author_avatar', '');
			$flicker_author_word   = flicker_option('blog_before_author_word', '');
			$flicker_show_icons    = !flicker_option('blog_hide_meta_icons', false);
		}

		if (!empty($flicker_show_author)) :
			//author-wrapper
			if (!empty($flicker_author_avatar)) :
				echo '<span class="author-avatar">';
				$flicker_author_id        = get_the_author_meta('ID');
				$flicker_custom_image_url = get_the_author_meta('custom_profile_image', $flicker_author_id);
				if (!empty($flicker_custom_image_url)) {
					echo '<img src="' . esc_url($flicker_custom_image_url) . '" alt="' . esc_attr(get_the_author_meta('display_name', $flicker_author_id)) . '">';
				} else {
					echo get_avatar($flicker_author_id, 60);
				}
				echo '</span><!-- .author-avatar-->';
			endif; //$flicker_author_avatar
		?>
			<span class="entry-author-wrap icon-inline">
				<?php
				//icon
				if (!empty($flicker_show_icons)) {
					flicker_icon('user');
				}
				//word
				if (!empty($flicker_author_word)) :
				?>
					<span class="entry-author-word meta-word">
						<?php echo esc_html($flicker_author_word); ?>
					</span>
					<!--.entry-author-word-->
				<?php
				endif;
				//value
				?>
				<span class="vcard author" itemtype="https://schema.org/Person" itemscope="itemscope" itemprop="author">
					<?php
					the_author_posts_link();
					?>
				</span><!-- .author -->
			</span>
			<!--.entry-author-wrap-->
			<!-- publisher -->
			<div class="hidden" itemprop="publisher" itemtype="http://schema.org/Organization" itemscope="itemscope">
				<span itemprop="name"><?php the_author(); ?></span>
				<?php
				$flicker_custom_logo = flicker_option('custom_logo');
				if (!empty($flicker_custom_logo)) :
				?>
					<span itemprop="logo" itemscope="itemscope" itemtype="https://schema.org/ImageObject">
						<?php
						$flicker_custom_logo_metadata = !empty($flicker_custom_logo) ? wp_get_attachment_metadata($flicker_custom_logo) : array(
							'width'  => '0',
							'height' => '0',
						);
						echo wp_get_attachment_image($flicker_custom_logo, 'full');
						?>
						<meta itemprop="url" content="<?php echo esc_url(home_url('/')); ?>" />
						<meta itemprop="width" content="<?php echo esc_attr($flicker_custom_logo_metadata['width']); ?>" />
						<meta itemprop="height" content="<?php echo esc_attr($flicker_custom_logo_metadata['height']); ?>" />
					</span>
				<?php endif; //custom_logo 
				?>
			</div>
			<!-- publisher -->
		<?php
		endif; //author
	}
endif; //flicker_the_author

add_filter('the_author_posts_link', 'flicker_the_author_link_itemprop');
if (!function_exists('flicker_the_author_link_itemprop')) :
	/**
	 * Add 'itemprop' attribute to the author link.
	 */
	function flicker_the_author_link_itemprop($flicker_link)
	{
		$flicker_link = str_replace('rel="author">', 'rel="author" itemprop="url"><span itemprop="name">', $flicker_link);
		$flicker_link = str_replace('</a>', '</span></a>', $flicker_link);

		return $flicker_link;
	}
endif;

if (!function_exists('flicker_the_date')) :
	/**
	 * Prints date HTML with the post link on blog.
	 */
	function flicker_the_date()
	{

		//options
		//single post
		if (is_singular()) {

			$flicker_show_date  = flicker_option('blog_single_show_date', true);
			$flicker_date_word  = flicker_option('blog_single_before_date_word', '');
			$flicker_show_icons = !flicker_option('blog_single_hide_meta_icons', false);

			//blog loop
		} else {

			$flicker_show_date  = flicker_option('blog_show_date', true);
			$flicker_date_word  = flicker_option('blog_before_date_word', '');
			$flicker_show_icons = !flicker_option('blog_hide_meta_icons', false);
		}

		if (!empty($flicker_show_date)) :
			//date-wrapper
		?>
			<span class="entry-date-wrap icon-inline">
				<?php

				//icon
				if (!empty($flicker_show_icons)) {
					flicker_icon('clock-outline');
				}

				//word
				if (!empty($flicker_date_word)) :
				?>
					<span class="date-word meta-word">
						<?php echo esc_html($flicker_date_word); ?>
					</span>
					<!--.date-word-->
				<?php
				endif;
				//value
				//link date to post on archive
				if (!is_singular()) :
				?>
					<a href="<?php echo esc_url(get_permalink()); ?>" rel="bookmark" itemprop="mainEntityOfPage">
					<?php endif; //is_singular 
					?>
					<span itemprop="datePublished"><?php the_time(get_option('date_format')); ?></span>
					<span class="hidden" itemprop="dateModified"><?php the_modified_time(get_option('date_format')); ?></span>
					<?php if (!is_singular()) : ?>
					</a>
				<?php endif; //is_singular 
				?>
			</span>
			<!--.entry-date-wrap-->
			<?php
		endif; //date
	}
endif; //flicker_the_date

if (!function_exists('flicker_the_categories')) :
	/**
	 * Prints categories HTML for the current post.
	 */

	function flicker_the_categories()
	{

		//options
		//single post
		if (is_singular()) {

			$flicker_show_categories = flicker_option('blog_single_show_categories', true);
			$flicker_categories_word = flicker_option('blog_single_before_categories_word', '');
			$flicker_show_icons      = !flicker_option('blog_single_hide_meta_icons', false);

			//blog loop
		} else {

			$flicker_show_categories = flicker_option('blog_show_categories', true);
			$flicker_categories_word = flicker_option('blog_before_categories_word', '');
			$flicker_show_icons      = !flicker_option('blog_hide_meta_icons', false);
		}

		if (!empty($flicker_show_categories)) :
			$flicker_c = wp_get_post_categories(get_the_ID());

			//only if categories exists
			if (!empty($flicker_c)) :

				//categories-wrapper
			?>
				<span class="entry-categories-wrap icon-inline">
					<?php

					//icon
					if (!empty($flicker_show_icons)) {
						flicker_icon('folder-outline');
					}

					//word
					if (!empty($flicker_categories_word)) :
					?>
						<span class="categories-word meta-word">
							<?php echo esc_html($flicker_categories_word); ?>
						</span>
						<!--.categories-word-->
					<?php
					endif;

					//value
					?>
					<span class="categories-list">
						<?php
						echo wp_kses_post(get_the_category_list('<span class="entry-categories-separator"> </span>'));
						?>
					</span>
					<!--.categories-list-->
				</span>
				<!--.entry-categories-wrap-->
			<?php
			endif; //$flicker_c
		endif; //categories
	}
endif; //flicker_the_categories

if (!function_exists('flicker_the_tags')) :
	/**
	 * Prints tags HTML for the current post.
	 */
	function flicker_the_tags()
	{

		//options
		//single post
		if (is_singular()) {

			$flicker_show_tags  = flicker_option('blog_single_show_tags', true);
			$flicker_tags_word  = flicker_option('blog_single_before_tags_word', '');
			$flicker_show_icons = !flicker_option('blog_single_hide_meta_icons', false);

			//blog loop
		} else {

			$flicker_show_tags  = flicker_option('blog_show_tags', true);
			$flicker_tags_word  = flicker_option('blog_before_tags_word', '');
			$flicker_show_icons = !flicker_option('blog_hide_meta_icons', false);
		}

		if (!empty($flicker_show_tags)) :

			$flicker_t = wp_get_post_tags(get_the_ID());

			//only if tags exists
			if (!empty($flicker_t)) :

				//tags-wrapper
			?>
				<span class="entry-tags-wrap icon-inline">
					<?php

					//icon
					if (!empty($flicker_show_icons)) {
						flicker_icon('tag');
					}

					//word
					if (!empty($flicker_tags_word)) :
					?>
						<span class="tags-word meta-word">
							<?php echo esc_html($flicker_tags_word); ?>
						</span>
						<!--.tags-word-->
					<?php
					endif; //tags_word

					//value
					?>
					<span class="tags-list">
						<?php
						echo wp_kses_post(get_the_tag_list('<span class="entry-tags">', '<span class="entry-tags-separator"> </span>', '</span>'));
						?>
					</span>
					<!--.tags-list-->
				</span>
				<!--.entry-tags-wrap-->
				<?php
			endif; //$flicker_t
		endif; //tags
	}
endif; //flicker_the_tags

if (!function_exists('flicker_comment_count')) :
	/**
	 * Prints HTML with the comment count for the current post.
	 */
	function flicker_comment_count()
	{

		//options
		//single post
		if (is_singular()) {

			$flicker_show_comments = flicker_option('blog_single_show_comments_link', 'number');
			$flicker_show_icons    = !flicker_option('blog_single_hide_meta_icons', false);

			//blog loop
		} else {

			$flicker_show_comments = flicker_option('blog_show_comments_link', 'number');
			$flicker_show_icons    = !flicker_option('blog_hide_meta_icons', false);
		}

		if (!post_password_required() && (comments_open() || get_comments_number()) && $flicker_show_comments) :
			switch ($flicker_show_comments):
				case 'number':
				?>
					<span class="comments-link icon-inline">
						<?php
						if (!empty($flicker_show_icons)) {
							flicker_icon('comment-outline');
						}
						$comments_count = get_comments_number();
						comments_popup_link($comments_count, $comments_count, $comments_count);
						?>
					</span><!-- .comments-link -->
				<?php
					break;
					//text
				default:
				?>
					<span class="comments-link icon-inline">
						<?php

						if (!empty($flicker_show_icons)) {
							flicker_icon('comment-outline');
						}
						?>

						<?php

						comments_popup_link(
							sprintf(
								wp_kses(
									/* translators: %s: Name of current post. Only visible to screen readers. */
									__(' Leave a comment<span class="screen-reader-text"> on %s</span>', 'flicker'),
									array(
										'span' => array(
											'class' => array(),
										),
									)
								),
								get_the_title()
							)
						);
						?>
					</span><!-- .comments-link -->
				<?php
			endswitch;
		endif; //show_comments
	}
endif;

if (!function_exists('flicker_set_post_likes')) :
	/**
	 * Likes incrementor
	 *
	 * @param int $postID ID of the post.
	 *
	 * @return bool No success if cookies are disabled
	 */
	function flicker_set_post_likes($postID)
	{
		if (empty($_COOKIE["$postID"])) {

			$count_key = 'flicker_post_likes_count';
			$count     = get_post_meta($postID, $count_key, true);
			if ($count == '') {
				$count = 0;
				delete_post_meta($postID, $count_key);
				add_post_meta($postID, $count_key, '1');
			} else {
				$count++;
				update_post_meta($postID, $count_key, $count);
			}
			setcookie("$postID", "voted", strtotime('+1 day'), COOKIEPATH, COOKIE_DOMAIN, false); // 86400 = 1 day
			return true;
		}

		return false;
	} //flicker_set_post_likes()
endif;

if (!function_exists('flicker_get_post_likes')) :
	/**
	 * Get likes value
	 *
	 * @param int $postID ID of the post.
	 */
	function flicker_get_post_likes($postID)
	{
		$count_key = 'flicker_post_likes_count';
		$count     = get_post_meta($postID, $count_key, true);
		if ($count == '') {
			delete_post_meta($postID, $count_key);
			add_post_meta($postID, $count_key, '0');

			return '0';
		}

		return $count;
	} //flicker_get_post_likes()
endif;

if (!function_exists('flicker_print_post_likes')) :
	/**
	 * Get likes value
	 *
	 * @param int $count of likes of the post.
	 */
	function flicker_print_post_likes($count)
	{
		$html = '';
		if (!$count) {
			$html = '<span class="item-likes-count">0</span> <span class="item-likes-word">' . esc_html__('Likes', 'flicker') . '</span>';
		}

		if ($count == 1) {
			$html = '<span class="item-likes-count">1</span> <span class="item-likes-word">' . esc_html__('Like', 'flicker') . '</span>';
		}

		if ($count > 1) {
			$html = '<span class="item-likes-count">' . $count . '</span> <span class="item-likes-word">' . esc_html__('Likes', 'flicker') . '</span>';
		}

		return $html;
	} //flicker_print_post_likes()
endif;

if (!function_exists('flicker_post_likes_scripts')) :
	// Add the JS
	function flicker_post_likes_scripts()
	{
		wp_enqueue_script('post-likes', get_template_directory_uri() . '/assets/js/mod-post-likes.js', array('jquery'), '1.0.0', true);
		wp_localize_script('post-likes', 'MyAjax', array(
			// URL to wp-admin/admin-ajax.php to process the request
			'ajaxurl'  => admin_url('admin-ajax.php'),
			// generate a nonce with a unique ID "myajax-post-comment-nonce"
			// so that you can check it later when an AJAX request is sent
			'security' => wp_create_nonce('increment-post-likes') //,
			//'post_id' => get_the_ID()
		));
	} //flicker_post_likes_scripts()
endif;
add_action('wp_enqueue_scripts', 'flicker_post_likes_scripts');

if (!function_exists('flicker_inc_post_like_callback')) :
	// The function that handles the AJAX request
	function flicker_inc_post_like_callback()
	{
		check_ajax_referer('increment-post-likes', 'security');
		$pID = intval($_POST['pID']);
		flicker_set_post_likes($pID);
		echo flicker_print_post_likes(flicker_get_post_likes($pID));

		die(); // this is required to return a proper result
	} //flicker_inc_post_like_callback()
endif;
add_action('wp_ajax_add_like', 'flicker_inc_post_like_callback');
add_action('wp_ajax_nopriv_add_like', 'flicker_inc_post_like_callback');

if (!function_exists('flicker_post_like_button')) :
	/**
	 * Print like button
	 */
	function flicker_post_like_button($postID)
	{
		$output = '';
		if (empty($_COOKIE["$postID"])) {
			$output = '<span data-id="' . $postID . '"><a href="" class="like_button like_active_button"><i class="fa fa-heart" aria-hidden="true"></i></a></span>';
		} else {
			$output = '<span data-id="' . $postID . '"><span class="like_button"><i class="fa fa-check" aria-hidden="true"></i></span></span>';
		}
		echo apply_filters('flicker_like_button', $output);
	} //flicker_post_like_button()
endif;
add_action('flicker_post_meta', 'flicker_post_like_button', 10, 1);

if (!function_exists('flicker_post_like_count')) :
	/**
	 * Print like counter value
	 */
	function flicker_post_like_count($postID)
	{
		echo apply_filters('flicker_like_count', '<span class="item-likes votes_count_' . $postID . '">' . flicker_print_post_likes(flicker_get_post_likes($postID)) . '</span>');
	} //flicker_post_like_count()
endif;
add_action('flicker_post_meta', 'flicker_post_like_count', 20, 1);

if (!function_exists('flicker_the_likes')) :
	/**
	 * Prints date HTML with the post link on blog.
	 */
	function flicker_the_likes()
	{

		//options
		//single post
		if (is_singular()) {

			$blog_show_likes  = flicker_option('blog_single_show_likes', 'number');
			$blog_before_likes_word  = flicker_option('blog_single_before_likes_word', '');

			//blog loop
		} else {

			$blog_show_likes  = flicker_option('blog_show_likes', 'number');
			$blog_before_likes_word  = flicker_option('blog_before_likes_word', '');
		}

		if (!empty($blog_show_likes)) :
			//likes-wrapper
			switch ($blog_show_likes):
				case 'number':
				?>
					<span class="entry-like-wrap icon-inline number-only">
						<?php

						//icon
						flicker_post_like_button(get_the_ID());

						//word
						if (!empty($blog_before_likes_word)) :
						?>
							<span class="date-word meta-word">
								<?php echo esc_html($blog_before_likes_word); ?>
							</span>
							<!--.likes-word-->
						<?php
						endif;
						flicker_post_like_count(get_the_ID());
						?>
					</span>
				<?php
					break;
					//text
				default:
				?>
					<span class="entry-view-wrap icon-inline">
						<?php

						//icon
						flicker_post_like_button(get_the_ID());

						//word
						if (!empty($blog_before_likes_word)) :
						?>
							<span class="date-word meta-word">
								<?php echo esc_html($blog_before_likes_word); ?>
							</span>
							<!--.likes-word-->
						<?php
						endif;
						flicker_post_like_count(get_the_ID());
						?>
					</span>
			<?php
			endswitch;
		endif; //likes
	}
endif; //flicker_show_likes

if (!function_exists('flicker_the_view')) :
	/**
	 * Prints date HTML with the post link on blog.
	 */
	function flicker_the_view()
	{

		//options
		//single post
		if (is_singular()) {

			$blog_show_views  = flicker_option('blog_single_show_views', true);
			$blog_before_views_word  = flicker_option('blog_single_before_views_word', '');
			$flicker_show_icons = !flicker_option('blog_single_hide_meta_icons', false);

			//blog loop
		} else {

			$blog_show_views  = flicker_option('blog_show_views', true);
			$blog_before_views_word  = flicker_option('blog_before_views_word', '');
			$flicker_show_icons = !flicker_option('blog_hide_meta_icons', false);
		}

		if (!empty($blog_show_views) && function_exists('flicker_show_post_views_count')) :
			//date-wrapper
			?>
			<span class="entry-view-wrap icon-inline">
				<?php

				//icon
				if (!empty($flicker_show_icons)) {
					flicker_icon('eye');
				}

				//word
				if (!empty($blog_before_views_word)) :
				?>
					<span class="date-word meta-word">
						<?php echo esc_html($blog_before_views_word); ?>
					</span>
					<!--.date-word-->
			<?php
				endif;
				flicker_show_post_views_count();
				echo '</span>';
			endif; //date
		}
	endif; //flicker_the_date

	if (!function_exists('flicker_entry_meta')) :
		/**
		 * Prints HTML with the comment count for the current post.
		 */

		function flicker_entry_meta($flicker_show_date = true, $flicker_show_author = true, $flicker_show_categories = true, $flicker_show_tags = true, $flicker_show_comments = true, $flicker_show_likes = true, $flicker_the_view = true)
		{
			if (!empty($flicker_show_date)) :
				flicker_the_date();
			endif; //date

			if (!empty($flicker_show_author)) :
				flicker_the_author();
			endif; //author

			if (!empty($flicker_show_categories)) :
				flicker_the_categories();
			endif; //categories

			if (!empty($flicker_show_tags)) :
				flicker_the_tags();
			endif; //tags

			if (!empty($flicker_show_comments)) :
				flicker_comment_count();
			endif; //comments

			if (!empty($flicker_show_likes)) :
				flicker_the_likes();
			endif; //likes

			if (!empty($flicker_the_view)) :
				flicker_the_view();
			endif; //view
		}
	endif;


	if (!function_exists('flicker_post_nav')) :
		/**
		 * Display navigation to next/previous post when applicable.
		 */
		function flicker_post_nav()
		{

			$flicker_blog_single_post_nav = flicker_option('blog_single_post_nav', '');

			if (empty($flicker_blog_single_post_nav)) {
				return;
			}

			// Don't print empty markup if there's nowhere to navigate.
			$flicker_previous = (is_attachment()) ? get_post(get_post()->post_parent) : get_adjacent_post(false, '', true);
			$flicker_next     = get_adjacent_post(false, '', false);

			if (!$flicker_next && !$flicker_previous) {
				return;
			}

			$flicker_word_prev = flicker_option('blog_single_post_nav_word_prev', esc_html__('Prev', 'flicker'));
			$flicker_word_next = flicker_option('blog_single_post_nav_word_next', esc_html__('Next', 'flicker'));
			?>
			<nav class="post-nav post-nav-layout-<?php echo esc_attr($flicker_blog_single_post_nav); ?>">
				<?php

				if (is_attachment() && 'attachment' === $flicker_previous->post_type) {
					return;
				}

				if ($flicker_previous && has_post_thumbnail($flicker_previous->ID)) {
					$flicker_prevthumb = wp_get_attachment_image_src(get_post_thumbnail_id($flicker_previous->ID), 'post-thumbnail');
					if ($flicker_prevthumb) {
						$flicker_prevthumb_sm         = wp_get_attachment_image_src(get_post_thumbnail_id($flicker_previous->ID), 'thumbnail');
						$flicker_prev_thumbnail_style = ' style="background-image: url(' . esc_url($flicker_prevthumb[0]) . '); "';
						$flicker_prev_thumbnail_class = 'has-image background-cover background-overlay';
						$flicker_prev_thumbnail_img   = '<span class="post-nav-thumb"><img src="' . esc_url($flicker_prevthumb_sm[0]) . '" alt="' . $flicker_previous->post_title . '"></span>';
					} else {
						$flicker_prev_thumbnail_style = '';
						$flicker_prev_thumbnail_class = 'no-image';
						$flicker_prev_thumbnail_img   = '';
					}
				} else {
					$flicker_prev_thumbnail_style = '';
					$flicker_prev_thumbnail_class = 'no-image';
					$flicker_prev_thumbnail_img   = '';
				}

				if ($flicker_next && has_post_thumbnail($flicker_next->ID)) {
					$flicker_nextthumb = wp_get_attachment_image_src(get_post_thumbnail_id($flicker_next->ID), 'post-thumbnail');
					if ($flicker_nextthumb) {
						$flicker_nextthumb_sm         = wp_get_attachment_image_src(get_post_thumbnail_id($flicker_next->ID), 'thumbnail');
						$flicker_next_thumbnail_style = ' style="background-image: url(' . esc_url($flicker_nextthumb[0]) . '); "';
						$flicker_next_thumbnail_class = 'has-image background-cover background-overlay';
						$flicker_next_thumbnail_img   = '<span class="post-nav-thumb"><img src="' . esc_url($flicker_nextthumb_sm[0]) . '" alt="' . $flicker_next->post_title . '"></span>';
					} else {
						$flicker_next_thumbnail_style = '';
						$flicker_next_thumbnail_class = 'no-image';
						$flicker_next_thumbnail_img   = '';
					}
				} else {
					$flicker_next_thumbnail_style = '';
					$flicker_next_thumbnail_class = 'no-image';
					$flicker_next_thumbnail_img   = '';
				}

				//layouts
				switch ($flicker_blog_single_post_nav):
					case 'bg':
						echo '<div>';
						previous_post_link(
							'%link',
							'<div class="post-nav-item bg-item prev-item ' . esc_attr($flicker_prev_thumbnail_class) . '"' . $flicker_prev_thumbnail_style . '>
							<span class="post-nav-words-wrap small-text">
								<span class="post-nav-word">' . esc_html($flicker_word_prev) . '</span>
								<span class="post-nav-title">%title</span>
							</span>
						</div>',
							false,
							''
						);
						echo '</div>';

						echo '<div>';
						next_post_link(
							'%link',
							'<div class="post-nav-item bg-item next-item ' . esc_attr($flicker_next_thumbnail_class) . '"' . $flicker_next_thumbnail_style . '>
							<span class="post-nav-words-wrap small-text">
								<span class="post-nav-word">' . esc_html($flicker_word_next) . '</span>
								<span class="post-nav-title">%title</span>
							</span>
						</div>',
							false,
							''
						);
						echo '</div>';
						break;
					case 'thumbnail':
						echo '<div>';
						previous_post_link(
							'%link',
							'<div class="post-nav-item prev-item">
							<span class="post-nav-arrow">' . flicker_icon('chevron-left', true) . '</span>
							' . $flicker_prev_thumbnail_img . '
							<span class="post-nav-words-wrap">
								<span class="post-nav-word">' . esc_html($flicker_word_prev) . '</span>
								<span class="post-nav-title">%title</span>
							</span>
						</div>',
							false,
							''
						);
						echo '</div>';

						echo '<div>';
						next_post_link(
							'%link',
							'<div class="post-nav-item next-item">
							<span class="post-nav-words-wrap">
								<span class="post-nav-word">' . esc_html($flicker_word_next) . '</span> 
								<span class="post-nav-title">%title</span>
							</span>
							' . $flicker_next_thumbnail_img . '
							<span class="post-nav-arrow">' . flicker_icon('chevron-right', true) . '</span>
						</div>',
							false,
							''
						);
						echo '</div>';
						break;
					case 'arrow':
						echo '<div>';
						previous_post_link(
							'%link',
							'<div class="post-nav-item prev-item">
								<span class="post-nav-word small-text">' . esc_html($flicker_word_prev) . '</span>
							</div>',
							false,
							''
						);
						echo '</div>';

						echo '<div>';
						next_post_link(
							'%link',
							'<div class="post-nav-item next-item">
								<span class="post-nav-word small-text">' . esc_html($flicker_word_next) . '</span> 
							</div>',
							false,
							''
						);
						echo '</div>';
						break;
						//title
					default:
						echo '<div>';
						previous_post_link(
							'%link',
							'<div class="post-nav-item prev-item">
							<span class="post-nav-words-wrap small-text">
								<span class="post-nav-word">' . esc_html($flicker_word_prev) . '</span>
								<span class="post-nav-title">%title</span>
							</span>
						</div>',
							false,
							''
						);
						echo '</div>';

						echo '<div>';
						next_post_link(
							'%link',
							'<div class="post-nav-item next-item">
							<span class="post-nav-words-wrap small-text">
								<span class="post-nav-word">' . esc_html($flicker_word_next) . '</span> 
								<span class="post-nav-title">%title</span>
							</span>
						</div>',
							false,
							''
						);
						echo '</div>';
				endswitch;

				?>
			</nav><!-- .navigation -->
			<?php
		} //flicker_post_nav
	endif;


	if (!function_exists('flicker_section_background_image_array')) :
		/**
		 * Get array of section attributes to display background image.
		 */
		function flicker_section_background_image_array($flicker_section, $flicker_empty_image = false)
		{

			//processing title section background for simple single post 'title-section-image' layout
			if (is_single() && 'title' === $flicker_section) :
				if (flicker_get_post_layout() === 'title-section-image') :
					//if has post thumbnail
					if (!post_password_required() && !is_attachment() && has_post_thumbnail()) {
						return array(
							'url'   => get_the_post_thumbnail_url(get_the_ID(), 'full'),
							'class' => 'i post-thumbnail-background background-cover cover-center background-fixed background-overlay overlay-dark',
						);
					}
				endif;
			endif; //is_single

			//for page with feature image - override default header_image
			if ('header_image' === $flicker_section) {
				$flicker_image = get_header_image();
				//for page with feature image - override default image
				if (is_page()) {
					if (has_post_thumbnail()) {
						$flicker_image = get_the_post_thumbnail_url();
					}
				}
			} else {
				$flicker_image = flicker_option($flicker_section . '_background_image', '');
				// override title background if page featured image is set
				if ('title' === $flicker_section && $flicker_image) {
					//for page with feature image - override default image
					if (is_page()) {
						if (has_post_thumbnail()) {
							$flicker_image = get_the_post_thumbnail_url();
						}
					}
				}
			}

			$flicker_return = array(
				'url'   => $flicker_image,
				'class' => '',
			);

			if (empty($flicker_image) && empty($flicker_empty_image)) {
				return $flicker_return;
			}

			$flicker_cover   = flicker_option($flicker_section . '_background_image_cover', '');
			$flicker_fixed   = flicker_option($flicker_section . '_background_image_fixed', '');
			$flicker_overlay = flicker_option($flicker_section . '_background_image_overlay', '');

			if (!empty($flicker_cover)) {
				$flicker_return['class'] .= 'background-cover cover-center';
			}

			if (!empty($flicker_fixed)) {
				$flicker_return['class'] .= ' background-fixed';
			}

			if (!empty($flicker_overlay)) {
				$flicker_return['class'] .= ' background-overlay ' . $flicker_overlay;
			}

			return $flicker_return;
		}
	endif;

	/////////////
	//Read More//
	/////////////

	// Read more markup inside link for excertp and the_content
	if (!function_exists('flicker_read_more_inside_link_markup')) :
		function flicker_read_more_inside_link_markup($flicker_read_more_text = '')
		{

			if (empty($flicker_read_more_text)) {
				$flicker_read_more_text = flicker_option('blog_read_more_text', '');
			}

			if (empty($flicker_read_more_text)) {
				return '';
			}

			return sprintf(
				wp_kses(
					$flicker_read_more_text . '<span class="screen-reader-text"> "%s"</span>',
					array(
						'span' => array(
							'class' => array(),
						),
					)
				),
				get_the_title()
			);
		}
	endif;

	//generated excerpt ending...
	if (!function_exists('flicker_excerpt_more')) :
		function flicker_excerpt_more($flicker_more)
		{
			return '<span class="more-dots">...</span>';
		}
	endif;
	add_filter('excerpt_more', 'flicker_excerpt_more', 21);

	//read more for excerpt
	if (!function_exists('flicker_read_more_markup_excerpt')) :
		function flicker_read_more_markup_excerpt()
		{
			global $post;
			$flicker_markup = ' <span class="more-tag"><a class="more-link theme_button color1" href="' .
				esc_url(get_permalink($post->ID)) . '">' .
				flicker_read_more_inside_link_markup() .
				'</a></span><!-- .more-tag -->';

			return $flicker_markup;
		}
	endif;

	//putting read more text inside excerpt if text is not empty
	if (!function_exists('flicker_read_more_in_excerpt')) :
		function flicker_read_more_in_excerpt($flicker_excerpt)
		{

			$flicker_read_more_text = flicker_option('blog_read_more_text', '');

			if (empty($flicker_read_more_text)) {
				return $flicker_excerpt;
			}

			$flicker_excerpt = str_replace('</p>', flicker_read_more_markup_excerpt($flicker_read_more_text) . '</p>', $flicker_excerpt);

			return $flicker_excerpt;
		}
	endif;
	add_filter('the_excerpt', 'flicker_read_more_in_excerpt', 21);

	//Filter the except length
	if (!function_exists('flicker_excerpt_custom_length')) :
		function flicker_excerpt_custom_length($flicker_length)
		{

			return absint(flicker_option('blog_excerpt_length', 40));
		}
	endif;
	add_filter('excerpt_length', 'flicker_excerpt_custom_length', 999);

	//home page intro teasers
	if (!function_exists('flicker_get_intro_teasers')) :
		function flicker_get_intro_teasers()
		{

			$flicker_teasers = array();

			for ($flicker_i = 1; $flicker_i < 5; $flicker_i++) {
				/*
			reeatable options:
				intro_teaser_image_
				intro_teaser_title_
				intro_teaser_text_
				intro_teaser_link_
				intro_teaser_button_text_
			*/
				$flicker_teasers[$flicker_i] = array_filter(
					array(
						'image'  => flicker_option('intro_teaser_image_' . $flicker_i, ''),
						'title'  => flicker_option('intro_teaser_title_' . $flicker_i, ''),
						'text'   => flicker_option('intro_teaser_text_' . $flicker_i, ''),
						'link'   => flicker_option('intro_teaser_link_' . $flicker_i, ''),
						'button' => flicker_option('intro_teaser_button_text_' . $flicker_i, ''),
					)
				);
			}

			return array_filter($flicker_teasers);
		}
	endif;

	//related posts
	if (!function_exists('flicker_related_posts')) :
		function flicker_related_posts($flicker_id)
		{

			$flicker_layout = flicker_option('blog_single_related_posts', '');
			if (empty($flicker_layout)) {
				return;
			}
			$flicker_tags   = wp_get_post_tags($flicker_id, array('fields' => 'ids'));
			if (!empty($flicker_tags)) :
				//list
				//list-thumbnails
				//grid
				//num of posts
				$flicker_posts_number = absint(flicker_option('blog_single_related_posts_number', 3));
				if (empty($flicker_posts_number)) {
					$flicker_posts_number = 3;
				}

				$flicker_args  = array(
					'tag__in'        => $flicker_tags,
					'post__not_in'   => array($flicker_id),
					'posts_per_page' => $flicker_posts_number,
				);
				$flicker_query = new WP_Query($flicker_args);
				if ($flicker_query->have_posts()) :
					$flicker_related_title = flicker_option('blog_single_related_posts_title', esc_html__('Related Posts', 'flicker'));
			?>
					<div class="related-posts content-padding with-background">
						<?php if (!empty($flicker_related_title)) : ?>
							<h4 class="related-posts-heading"><?php echo wp_kses_post($flicker_related_title); ?></h4>
							<?php
						endif; //related_title
						switch ($flicker_layout):
							case 'grid':
								switch ($flicker_posts_number):
									case 3:
										$flicker_wrapper_class = 'layout-cols-3';
										break;
									case 4:
										$flicker_wrapper_class = 'layout-cols-4';
										break;
									default:
										$flicker_wrapper_class = '';
								endswitch;
								if ($flicker_query->post_count < 3) {
									$flicker_wrapper_class = 'layout-cols-' . $flicker_query->post_count;
								}
							?>
								<div class="layout-gap-30">
									<div class="grid-wrapper <?php echo esc_attr($flicker_wrapper_class); ?>">
										<?php
										while ($flicker_query->have_posts()) :
											$flicker_query->the_post();
										?>
											<div class=grid-item>
												<article <?php post_class('vertical-item'); ?>>
													<?php if (has_post_thumbnail()) : ?>
														<figure class="post-thumbnail">
															<a href="<?php the_permalink(); ?>">
																<?php the_post_thumbnail('flicker-default-post'); ?>
															</a>
														</figure>
													<?php endif; ?>
													<div class="item-content">
														<h5 class="entry-title">
															<a rel="bookmark" href="<?php the_permalink(); ?>"><?php the_title(); ?></a>
														</h5>
														<div class="entry-meta post-meta small-text highlight">
															<div class="meta-left"><?php flicker_entry_meta(true, false, false, false, false, false, false); ?></div>
														</div>
													</div>
												</article>
											</div>
										<?php endwhile; ?>
									</div><!-- .grid-wrapper -->
								</div><!-- .layout-gap-* -->
							<?php
								break;

							case 'list-thumbnails':
							?>
								<ul class="posts-list">
									<?php
									while ($flicker_query->have_posts()) :
										$flicker_query->the_post();
									?>
										<li class="list-has-post-thumbnail">
											<?php if (has_post_thumbnail()) : ?>
												<a class="posts-list-thumbnail" href="<?php the_permalink(); ?>">
													<?php the_post_thumbnail('thumbnail'); ?>
												</a>
											<?php endif; ?>
											<div class="item-content">
												<h4 class="post-title">
													<a rel="bookmark" href="<?php the_permalink(); ?>"><?php the_title(); ?></a>
												</h4>
												<div class="icon-inline-wrap">
													<span class="icon-inline post-date">
														<?php flicker_icon('clock-outline'); ?>
														<span><?php echo get_the_date('', get_the_ID()); ?></span>
													</span>
												</div>
											</div>
										</li>
									<?php endwhile; ?>
								</ul>
							<?php
								break;

							default:
							?>
								<ul class="list-styled">
									<?php
									while ($flicker_query->have_posts()) :
										$flicker_query->the_post();
									?>
										<li>
											<h6>
												<a href="<?php the_permalink(); ?>" rel="bookmark"><?php the_title(); ?></a>
											</h6>
										</li>
									<?php endwhile; ?>
								</ul>
						<?php
						endswitch; //layout
						?>
					</div><!-- .related-posts -->
			<?php
				endif; //have_posts
				wp_reset_postdata();
			endif; //tags
		}
	endif;

	//get any widget HTML markup
	if (!function_exists('flicker_get_the_widget')) :
		function flicker_get_the_widget($flicker_widget_class, $flicker_instance = array())
		{

			if (!class_exists($flicker_widget_class)) {
				return '';
			}
			//same as in inc/setup.php file
			$flicker_args = array(
				'before_title' => '<h3 class="widget-title"><span>',
				'after_title'  => '</span></h3>',
			);

			ob_start();

			the_widget($flicker_widget_class, $flicker_instance, $flicker_args);

			return ob_get_clean();
		}
	endif;

	//count widgets in sidebar
	if (!function_exists('flicker_get_sidebar_widgets_count')) :
		function flicker_get_sidebar_widgets_count($sidebar_id)
		{
			$widgets = get_option('sidebars_widgets');

			return count($widgets[$sidebar_id]);
		}
	endif;

	//detect shop - handy for sidebar and breadcrumbs
	if (!function_exists('flicker_is_shop')) :
		function flicker_is_shop()
		{
			$flicker_return = false;
			if (function_exists('is_woocommerce')) {
				if (is_woocommerce() || is_cart() || is_checkout() || is_account_page()) {
					$flicker_return = true;
				}
			}
			if (function_exists('yith_wcwl_is_wishlist_page')) {
				if (yith_wcwl_is_wishlist_page()) {
					$flicker_return = true;
				}
			}

			return $flicker_return;
		}
	endif;

	//echo breadcrumbs markup
	if (!function_exists('flicker_breadcrumbs')) :
		function flicker_breadcrumbs()
		{
			$flicker_args              = array(
				'before' => '<nav class="breadcrumbs">',
				'after'  => '</nav>',
			);
			$flicker_seo_options       = get_option('wpseo_titles');
			$flicker_args['delimiter'] = !empty($flicker_seo_options['breadcrumbs-sep']) ? $flicker_seo_options['breadcrumbs-sep'] : '/';
			if (flicker_is_shop()) :
				woocommerce_breadcrumb(
					array(
						'wrap_before' => $flicker_args['before'] . '<span>',
						'wrap_after'  => '</span>' . $flicker_args['after'],
						'before'      => '<span class="breadcrumbs_item">',
						'after'       => '</span>',
						'delimiter'   => ' ' . $flicker_args['delimiter'] . ' ',
					)
				);
			elseif (function_exists('yoast_breadcrumb')) :
				yoast_breadcrumb('<nav class="breadcrumbs">', '</nav>');
			elseif (function_exists('rank_math_the_breadcrumbs')) :
				$args = array(
					'delimiter'   => '&nbsp;&#47;&nbsp;',
					'wrap_before' => '<nav class="breadcrumbs">',
					'wrap_after'  => '</nav>',
					'before'      => '',
					'after'       => '',
				);
				rank_math_the_breadcrumbs($args);
			endif;
		}
	endif;

	//check if breadcrumbs are enabled and plugins to show them are active
	if (!function_exists('flicker_breadcrumbs_enabled')) :
		function flicker_breadcrumbs_enabled()
		{
			$flicker_return = flicker_option('title_show_breadcrumbs', true);
			if (flicker_is_shop() && $flicker_return) {
				return $flicker_return;
			} elseif (function_exists('yoast_breadcrumb') && $flicker_return) {
				return $flicker_return;
			} elseif (function_exists('rank_math_the_breadcrumbs') && $flicker_return) {
				return $flicker_return;
			} else {
				return false;
			}
		}
	endif;

	//copyright text - year
	if (!function_exists('flicker_get_copyright_text')) :
		function flicker_get_copyright_text($flicker_text = '')
		{
			$flicker_text = str_replace('[year]', '<span class="copyright-year">' . date('Y') . '</span>', $flicker_text);
			return $flicker_text;
		}
	endif;

	//detect is_front_page and not is paged
	if (!function_exists('flicker_is_front_page')) :
		function flicker_is_front_page()
		{
			return is_front_page();
		}
	endif;

	//detect for displaying title section
	if (!function_exists('flicker_is_title_section_is_shown')) :
		function flicker_is_title_section_is_shown()
		{
			if (is_page_template('page-templates/no-sidebar-no-title.php')) {
				return false;
			}
			$flicker_show_title       = flicker_option('title_show_title', '');
			$flicker_show_search      = flicker_option('title_show_search', '');
			$flicker_show_breadcrumbs = flicker_breadcrumbs_enabled();
			$flicker_is_front_page    = flicker_is_front_page();

			if (!empty($flicker_is_front_page) && empty($flicker_show_search)) {
				return false;
			}

			if (empty($flicker_show_title) && empty($flicker_show_breadcrumbs) && empty($flicker_show_search)) {
				return false;
			} else {
				return true;
			}
		}
	endif;

	/**
	 *  Display info corner / date
	 */

	if (!function_exists('flicker_corner_date')) :
		function flicker_corner_date()
		{ ?>
			<div class="entry-meta-corner">
				<?php
				// Set up and print post meta information.
				printf(
					'<span class="date"><time class="entry-date" datetime="%1$s">%3$s<strong>%2$s</strong></time></span>',
					esc_attr(get_the_date('c')),
					esc_html(get_the_date('j')),
					esc_html(get_the_date('M'))
				); ?>
			</div>
	<?php
		}
	endif; // flicker_corner_date