<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

if ( ! defined( 'UB' ) ) {
	define( 'UB', true );
}


/**
 * Alias
 *
 * @param string $extension_name
 *
 * @return UB_Extension|null
 */
if ( ! function_exists( 'ub_ext' ) ) :
	function ub_ext( $extension_name ) {

		if( 'backups' === $extension_name ) {
			global $ub_global_backup;
			return $ub_global_backup;
		}
		return $extension_name;
	}
endif;
/**
 * Check for command line interface
 *
 * @return bool
 * @since 2.6.16
 */
if ( ! function_exists( 'ub_is_cli' ) ) :
	function ub_is_cli() {
		return ( php_sapi_name() === 'cli' ) && defined( 'WP_CLI' );
	}
endif;



/**
 * @return string Current url
 */
if ( ! function_exists( 'ub_current_url' ) ) :
function ub_current_url() {
	static $url = null;
	if ( $url === null ) {
		if ( is_multisite() && ! ( defined( 'SUBDOMAIN_INSTALL' ) && SUBDOMAIN_INSTALL ) ) {
			switch_to_blog( 1 );
			$url = get_option( 'home' );
			restore_current_blog();
		} else {
			$url = get_option( 'home' );
		}

		//Remove the "//" before the domain name
		$url = ltrim( ub_get_url_without_scheme( $url ), '/' );

		//Remove the ulr subdirectory in case it has one
		$split = explode( '/', $url );

		//Remove end slash
		$url = rtrim( $split[0], '/' );

		$url .= '/' . ltrim( ub_akg( 'REQUEST_URI', $_SERVER, '' ), '/' );
		$url = set_url_scheme( '//' . $url ); // https fix
	}

	return $url;
}
endif;

/*
 * Return URI without scheme
 */
if ( ! function_exists( 'ub_get_url_without_scheme' ) ) :
function ub_get_url_without_scheme( $url ) {
	return preg_replace( '/^[^:]+:\/\//', '//', $url );
}
endif;

/**
 * Check is the current value is instance of UB_Callback class
 *
 * @param mixed $value
 *
 * @return bool
 */
if ( ! function_exists( 'ub_is_callback' ) ) :
	function ub_is_callback( $value ) {
		return $value instanceof UB_Callback || ( is_object( $value ) && get_class( $value ) == 'Closure' );
	}
endif;


/**
 * In the value is instance of UB_Callback class then it is executed and returns the callback value
 * In other case function returns the provided value
 *
 * @param mixed|UB_Callback $value
 *
 * @return mixed
 *
 * @since 2.6.14
 */
if ( ! function_exists( 'ub_call' ) ) :
	function ub_call( $value ) {
		if ( ! ub_is_callback( $value ) ) {
			return $value;
		}

		return ( is_object( $value ) && get_class( $value ) == 'Closure' )
			? $value()
			: $value->execute();
	}
endif;

/**
 * Recursively find a key's value in array
 *
 * @param string $keys 'a/b/c'
 * @param array|object $array_or_object
 * @param null|mixed $default_value
 * @param string $keys_delimiter
 *
 * @return null|mixed
 */
if ( ! function_exists( 'ub_akg' ) ) :
	function ub_akg( $keys, $array_or_object, $default_value = null, $keys_delimiter = '/' ) {
		if ( ! is_array( $keys ) ) {
			$keys = explode( $keys_delimiter, (string) $keys );
		}

		$array_or_object = ub_call( $array_or_object );

		$key_or_property = array_shift( $keys );
		if ( $key_or_property === null ) {
			return ub_call( $default_value );
		}

		$is_object = is_object( $array_or_object );

		if ( $is_object ) {
			if ( ! property_exists( $array_or_object, $key_or_property ) ) {
				return ub_call( $default_value );
			}
		} else {
			if ( ! is_array( $array_or_object ) || ! array_key_exists( $key_or_property, $array_or_object ) ) {
				return ub_call( $default_value );
			}
		}

		if ( isset( $keys[0] ) ) { // not used count() for performance reasons
			if ( $is_object ) {
				return ub_akg( $keys, $array_or_object->{$key_or_property}, $default_value );
			} else {
				return ub_akg( $keys, $array_or_object[ $key_or_property ], $default_value );
			}
		} else {
			if ( $is_object ) {
				return $array_or_object->{$key_or_property};
			} else {
				return $array_or_object[ $key_or_property ];
			}
		}
	}
endif;

/**
 * Set (or create if not exists) value for specified key in some array level
 *
 * @param string $keys 'a/b/c', or 'a/b/c/' equivalent to: $arr['a']['b']['c'][] = $val;
 * @param mixed $value
 * @param array|object $array_or_object
 * @param string $keys_delimiter
 *
 * @return array|object
 */
if ( ! function_exists( 'ub_aks' ) ) :
	function ub_aks( $keys, $value, &$array_or_object, $keys_delimiter = '/' ) {
		if ( ! is_array( $keys ) ) {
			$keys = explode( $keys_delimiter, (string) $keys );
		}

		$key_or_property = array_shift( $keys );
		if ( $key_or_property === null ) {
			return $array_or_object;
		}

		$is_object = is_object( $array_or_object );

		if ( $is_object ) {
			if ( ! property_exists( $array_or_object, $key_or_property )
			     || ! ( is_array( $array_or_object->{$key_or_property} ) || is_object( $array_or_object->{$key_or_property} ) )
			) {
				if ( $key_or_property === '' ) {
					// this happens when use 'empty keys' like: abc/d/e////i/j//foo/
					trigger_error( 'Cannot push value to object like in array ($arr[] = $val)', E_USER_WARNING );
				} else {
					$array_or_object->{$key_or_property} = array();
				}
			}
		} else {
			if ( ! is_array( $array_or_object ) ) {
				$array_or_object = array();
			}

			if ( ! array_key_exists( $key_or_property,
					$array_or_object ) || ! is_array( $array_or_object[ $key_or_property ] )
			) {
				if ( $key_or_property === '' ) {
					// this happens when use 'empty keys' like: abc.d.e....i.j..foo.
					$array_or_object[] = array();

					// get auto created key (last)
					end( $array_or_object );
					$key_or_property = key( $array_or_object );
				} else {
					$array_or_object[ $key_or_property ] = array();
				}
			}
		}

		if ( isset( $keys[0] ) ) { // not used count() for performance reasons
			if ( $is_object ) {
				ub_aks( $keys, $value, $array_or_object->{$key_or_property} );
			} else {
				ub_aks( $keys, $value, $array_or_object[ $key_or_property ] );
			}
		} else {
			if ( $is_object ) {
				$array_or_object->{$key_or_property} = $value;
			} else {
				$array_or_object[ $key_or_property ] = $value;
			}
		}

		return $array_or_object;
	}
endif;

/**
 * Safe load variables from an file
 * Use this function to not include files directly and to not give access to current context variables (like $this)
 *
 * @param string $file_path
 * @param array $_extract_variables Extract these from file array('variable_name' => 'default_value')
 * @param array $_set_variables Set these to be available in file (like variables in view)
 *
 * @return array
 */
if ( ! function_exists( 'ub_get_variables_from_file' ) ) :
	function ub_get_variables_from_file( $file_path, array $_extract_variables, array $_set_variables = array() ) {
		extract( $_set_variables, EXTR_REFS );
		unset( $_set_variables );

		require $file_path;

		foreach ( $_extract_variables as $variable_name => $default_value ) {
			if ( isset( $$variable_name ) ) {
				$_extract_variables[ $variable_name ] = $$variable_name;
			}
		}

		return $_extract_variables;
	}
endif;

/**
 * @param string|array $callback Callback function
 * @param array $args Callback arguments
 * @param bool $cache Whenever you want to cache the function value after it's first call or not
 * Recommend when the function call may require many resources or time (database requests) , or the value is small
 * Not recommended using on very large values
 *
 * @return UB_Callback
 *
 * @since 2.6.14
 */
if ( ! function_exists( 'ub_callback' ) ) :
	function ub_callback( $callback, array $args = array(), $cache = true ) {
		return new UB_Callback( $callback, $args, $cache );
	}
endif;

/**
 * There is a theme which does: if (!defined('UB')): function ub_render_view() { ... } endif;
 * It works fine, except in this case
 * https://github.com/ThemeFuse/Unyson/commit/07be8b1f4b50eaf0f1f7e85ea1c6912a0415d241#diff-cf866bf08b8f747e3120221a6b1b07cfR48
 * it throws fatal error because this function here is defined after that
 */
if ( ! function_exists( 'ub_render_view' ) ):
	/**
	 * Safe render a view and return html
	 * In view will be accessible only passed variables
	 * Use this function to not include files directly and to not give access to current context variables (like $this)
	 *
	 * @param string $file_path
	 * @param array $view_variables
	 * @param bool $return In some cases, for memory saving reasons, you can disable the use of output buffering
	 *
	 * @return string HTML
	 */
	function ub_render_view( $file_path, $view_variables = array(), $return = true ) {

		if ( ! is_file( $file_path ) ) {
			return '';
		}

		extract( $view_variables, EXTR_REFS );
		unset( $view_variables );

		if ( $return ) {
			ob_start();
			require $file_path;

			return ob_get_clean();
		} else {
			require $file_path;
		}

		return '';
	}
endif;


/**
 * Use this function to not include files directly and to not give access to current context variables (like $this)
 *
 * @param string $file_path
 * @param bool $once
 *
 * @return bool If was included or not
 */
if ( ! function_exists( 'ub_include_file_isolated' ) ) :
	function ub_include_file_isolated( $file_path, $once = false ) {
		if ( file_exists( $file_path ) ) {
			if ( (bool) $once ) {
				include_once $file_path;
			} else {
				include $file_path;
			}

			return true;
		} else {
			return false;
		}
	}
endif;
/**
 * Use this id do not want to enter every time same last two parameters
 * Info: Cannot use default parameters because in php 5.2 encoding is not UTF-8 by default
 *
 * @param string $string
 *
 * @return string
 */
if ( ! function_exists( 'ub_htmlspecialchars' ) ) :
	function ub_htmlspecialchars( $string ) {
		return htmlspecialchars( $string, ENT_QUOTES, 'UTF-8' );
	}
endif;

/**
 * Generate attributes string for html tag
 *
 * @param array $attr_array array('href' => '/', 'title' => 'Test')
 *
 * @return string 'href="/" title="Test"'
 */
if ( ! function_exists( 'ub_attr_to_html' ) ) :
	function ub_attr_to_html( array $attr_array ) {
		$html_attr = '';

		foreach ( $attr_array as $attr_name => $attr_val ) {
			if ( $attr_val === false ) {
				continue;
			}

			$html_attr .= $attr_name . '="' . ub_htmlspecialchars( $attr_val ) . '" ';
		}

		return $html_attr;
	}
endif;
/**
 * Generate html tag
 *
 * @param string $tag Tag name
 * @param array $attr Tag attributes
 * @param bool|string $end Append closing tag. Also accepts body content
 *
 * @return string The tag's html
 */
if ( ! function_exists( 'ub_html_tag' ) ) :
	function ub_html_tag( $tag, $attr = array(), $end = false ) {
		$html = '<' . $tag . ' ' . ub_attr_to_html( $attr );

		if ( $end === true ) {
			# <script></script>
			$html .= '></' . $tag . '>';
		} else if ( $end === false ) {
			# <br/>
			$html .= '/>';
		} else {
			# <div>content</div>
			$html .= '>' . $end . '</' . $tag . '>';
		}

		return $html;
	}
endif;

/**
 * Try to make user friendly title from an id
 *
 * @param string $id 'hello-world'
 *
 * @return string 'Hello world'
 */
if ( ! function_exists( 'ub_id_to_title' ) ) :
	function ub_id_to_title( $id ) {
		// mb_ucfirst()
		if ( function_exists( 'mb_strtoupper' ) && function_exists( 'mb_substr' ) && function_exists( 'mb_strlen' ) ) {
			$id = mb_strtoupper( mb_substr( $id, 0, 1, 'UTF-8' ), 'UTF-8' ) . mb_substr( $id,
					1,
					mb_strlen( $id, 'UTF-8' ),
					'UTF-8' );
		} else {
			$id = strtoupper( substr( $id, 0, 1 ) ) . substr( $id, 1, strlen( $id ) );
		}

		return str_replace( array( '_', '-' ), ' ', $id );
	}
endif;

/** Framework related functions */
{
	/**
	 * Full path to the parent-theme/framework directory
	 *
	 * @param string $rel_path
	 *
	 * @return string
	 */
	if ( ! function_exists( 'ub_get_framework_directory' ) ) :
	function ub_get_framework_directory( $rel_path = '' ) {
		try {
			$dir = UB_Cache::get( $cache_key = 'ub_framework_dir' );
		} catch ( UB_Cache_Not_Found_Exception $e ) {
			UB_Cache::set(
				$cache_key,
				$dir = apply_filters(
					'ub_framework_directory',
					ub_fix_path( dirname( dirname( __FILE__ ) ) ) // double dirname() to remove '/helpers', use parent dir
				)
			);
		}

		return $dir . $rel_path;
	}
	endif;
	/**
	 * URI to the parent-theme/framework directory
	 *
	 * @param string $rel_path
	 *
	 * @return string
	 */
	if ( ! function_exists( 'ub_get_framework_directory_uri' ) ) :
		function ub_get_framework_directory_uri( $rel_path = '' ) {
			try {
				$uri = UB_Cache::get( $cache_key = 'ub_framework_dir_uri' );
			} catch ( UB_Cache_Not_Found_Exception $e ) {
				UB_Cache::set(
					$cache_key,
					$uri = apply_filters(
						'ub_framework_directory_uri',
						( $uri = ub_get_path_url( ub_get_framework_directory() ) )
							? $uri
							: get_template_directory_uri() . '/framework'
					)
				);
			}

			return $uri . $rel_path;
		}
	endif;
	
	 /**
     * @return string|null
     * @since 2.4.10
     */
    if ( ! function_exists( 'ub_get_json_last_error_message' ) ) :
        function ub_get_json_last_error_message() {
            switch ( function_exists( 'json_last_error' ) ? json_last_error() : - 1 ) {
                case JSON_ERROR_NONE:
                    return null; // __('No errors', 'fw');
                    break;
                case JSON_ERROR_DEPTH:
                    return __( 'Maximum stack depth exceeded', 'fw' );
                    break;
                case JSON_ERROR_STATE_MISMATCH:
                    return __( 'Underflow or the modes mismatch', 'fw' );
                    break;
                case JSON_ERROR_CTRL_CHAR:
                    return __( 'Unexpected control character found', 'fw' );
                    break;
                case JSON_ERROR_SYNTAX:
                    return __( 'Syntax error, malformed JSON', 'fw' );
                    break;
                case JSON_ERROR_UTF8:
                    return __( 'Malformed UTF-8 characters, possibly incorrectly encoded', 'fw' );
                    break;
                default:
                    return __( 'Unknown error', 'fw' );
                    break;
            }
        }
    endif;
}