<?php
/**
 * Widget API: MWT_Widget_Theme_Category class
 *
 * @package Flicker
 * @since 0.0.1
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}


if ( ! class_exists( 'MWT_Widget_Theme_Category' ) ) :

	class MWT_Widget_Theme_Category extends WP_Widget {

		/**
		 * Sets up a new Theme Category widget instance.
		 *
		 * @since 0.0.1
		 */
		public function __construct() {
			$mwt_widget_ops = array(
				'classname'                   => 'widget_theme_category',
				'description'                 => esc_html__( 'A category with articles counter.', 'mwt-widgets' ),
				'customize_selective_refresh' => true,
			);
			parent::__construct( 'theme-category', esc_html__( 'Theme Category', 'mwt-widgets' ), $mwt_widget_ops );
			$this->alt_option_name = 'widget_theme_category';
		}

		/**
		 * Outputs the content for the current Categories widget instance.
		 *
		 * @param array $mwt_args Display arguments including 'before_title', 'after_title',
		 *                        'before_widget', and 'after_widget'.
		 * @param array $mwt_instance Settings for the current Categories widget instance.
		 *
		 * @since 0.0.1
		 *
		 */
		public function widget( $mwt_args, $mwt_instance ) {
			if ( ! isset( $mwt_args['widget_id'] ) ) {
				$mwt_args['widget_id'] = $this->id;
			}

			$mwt_title = ! empty( $mwt_instance['title'] ) ? $mwt_instance['title'] : '';

			/** This filter is documented in wp-includes/widgets/class-wp-widget-pages.php */
			$mwt_title = apply_filters( 'widget_title', $mwt_title, $mwt_instance, $this->id_base );

			$mwt_cat         = ! empty( $mwt_instance['cat'] ) ? absint( $mwt_instance['cat'] ) : '0';
			$mwt_c           = ! empty( $mwt_instance['count'] ) ? '1' : '';
			$mwt_bordered    = ! empty( $mwt_instance['bordered'] ) ? '1' : '';
			$mwt_muted       = ! empty( $mwt_instance['muted'] ) ? '1' : '';
			$mwt_center      = ! empty( $mwt_instance['center'] ) ? '1' : '';
			$mwt_count_label = ! empty( $mwt_instance['count_label'] ) ? $mwt_instance['count_label'] : '';

			echo wp_kses_post( $mwt_args['before_widget'] );

			if ( $mwt_title ) {
				echo wp_kses_post( $mwt_args['before_title'] . $mwt_title . $mwt_args['after_title'] );
			}

			$mwt_category = get_category( $mwt_cat );

			if ( $mwt_category ) :
				$mwt_css_class = '';
				if ( $mwt_bordered ) {
					$mwt_css_class .= ' bordered';
				}
				if ( $mwt_muted ) {
					$mwt_css_class .= ' muted';
				}
				if ( $mwt_center ) {
					$mwt_css_class .= ' text-center';
				}
				?>
				<a href="<?php echo esc_url( get_category_link( $mwt_category ) ); ?>">
					<div class="category-block<?php echo esc_attr( $mwt_css_class ); ?>">
						<?php if ( empty( $mwt_count_label ) ) : ?>
							<h5>
								<?php
								echo esc_html( $mwt_category->name );

								if ( ! empty( $mwt_c ) ) {
									echo ' <span class="items-count"><span class="items-count-open">(</span>' . esc_html( $mwt_category->count ) . '<span class="items-count-close">)</span></span>';
								}
								?>
							</h5>
							<?php
						else :
							?>
							<h5><?php echo esc_html( $mwt_category->name ); ?></h5>
							<p>
								<?php
								echo '<span class="items-count-label">' . wp_kses_post( $mwt_count_label ) . '</span>';
								if ( ! empty( $mwt_c ) ) {
									echo ' <span class="items-count">' . esc_html( $mwt_category->count ) . '</span>';
								}
								?>
							</p>
							<?php
						endif; //$mwt_count_label
						?>
					</div><!-- .category-block -->
				</a>
				<?php
			endif; //category

			echo wp_kses_post( $mwt_args['after_widget'] );
		}

		/**
		 * Handles updating settings for the current Categories widget instance.
		 *
		 * @param array $mwt_new_instance New settings for this instance as input by the user via
		 *                            WP_Widget::form().
		 * @param array $mwt_old_instance Old settings for this instance.
		 *
		 * @return array Updated settings to save.
		 * @since 0.0.1
		 *
		 */
		public function update( $mwt_new_instance, $mwt_old_instance ) {
			$mwt_instance                = $mwt_old_instance;
			$mwt_instance['title']       = sanitize_text_field( $mwt_new_instance['title'] );
			$mwt_instance['cat']         = ! empty( $mwt_new_instance['cat'] ) ? absint( $mwt_new_instance['cat'] ) : 0;
			$mwt_instance['count']       = ! empty( $mwt_new_instance['count'] ) ? 1 : 0;
			$mwt_instance['bordered']    = ! empty( $mwt_new_instance['bordered'] ) ? 1 : 0;
			$mwt_instance['muted']       = ! empty( $mwt_new_instance['muted'] ) ? 1 : 0;
			$mwt_instance['center']      = ! empty( $mwt_new_instance['center'] ) ? 1 : 0;
			$mwt_instance['count_label'] = sanitize_text_field( $mwt_new_instance['count_label'] );

			return $mwt_instance;
		}

		/**
		 * Outputs the settings form for the Categories widget.
		 *
		 * @param array $mwt_instance Current settings.
		 *
		 * @since 0.0.1
		 *
		 */
		public function form( $mwt_instance ) {
			//Defaults
			$mwt_instance    = wp_parse_args( (array) $mwt_instance, array( 'title' => '' ) );
			$mwt_cat         = isset( $mwt_instance['cat'] ) ? absint( $mwt_instance['cat'] ) : false;
			$mwt_count       = isset( $mwt_instance['count'] ) ? (bool) $mwt_instance['count'] : false;
			$mwt_bordered    = isset( $mwt_instance['bordered'] ) ? (bool) $mwt_instance['bordered'] : false;
			$mwt_muted       = isset( $mwt_instance['muted'] ) ? (bool) $mwt_instance['muted'] : false;
			$mwt_center      = isset( $mwt_instance['center'] ) ? (bool) $mwt_instance['center'] : false;
			$mwt_count_label = isset( $mwt_instance['count_label'] ) ? sanitize_text_field( $mwt_instance['count_label'] ) : '';
			?>
			<p>
				<label for="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>">
					<?php esc_html_e( 'Title:', 'mwt-widgets' ); ?>
				</label>
				<input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>"
						name="<?php echo esc_attr( $this->get_field_name( 'title' ) ); ?>" type="text"
						value="<?php echo esc_attr( $mwt_instance['title'] ); ?>"/>
			</p>

			<p>
				<label for="<?php echo esc_attr( $this->get_field_id( 'cat' ) ); ?>">
					<?php esc_html_e( 'Category:', 'mwt-widgets' ); ?>
				</label>
				<?php
				$mwt_dropdown_args = array(
					'taxonomy'          => 'category',
					'selected'          => $mwt_cat,
					'orderby'           => 'name',
					'order'             => 'ASC',
					'show_count'        => 1,
					'hide_empty'        => 0,
					'child_of'          => 0,
					'exclude'           => '',
					'hierarchical'      => 1,
					'depth'             => 0,
					'class'             => 'widefat',
					'tab_index'         => 0,
					'hide_if_empty'     => false,
					'option_none_value' => 0,
					'value_field'       => 'term_id',
					'id'                => $this->get_field_id( 'cat' ),
					'name'              => $this->get_field_name( 'cat' ),
				);
				wp_dropdown_categories( $mwt_dropdown_args );
				?>
			</p>

			<p>
				<input type="checkbox" class="checkbox" id="<?php echo esc_attr( $this->get_field_id( 'count' ) ); ?>"
						name="<?php echo esc_attr( $this->get_field_name( 'count' ) ); ?>"<?php checked( $mwt_count ); ?> />
				<label for="<?php echo esc_attr( $this->get_field_id( 'count' ) ); ?>">
					<?php esc_html_e( 'Show post counts', 'mwt-widgets' ); ?>
				</label>
			</p>


			<p>
				<label
					for="<?php echo esc_attr( $this->get_field_id( 'count_label' ) ); ?>"><?php esc_html_e( 'Count number label:', 'mwt-widgets' ); ?></label>
				<input
					class="widefat"
					id="<?php echo esc_attr( $this->get_field_id( 'count_label' ) ); ?>"
					name="<?php echo esc_attr( $this->get_field_name( 'count_label' ) ); ?>"
					type="text"
					value="<?php echo esc_attr( $mwt_count_label ); ?>"/>
			</p>

			<p>
				<input type="checkbox" class="checkbox"
						id="<?php echo esc_attr( $this->get_field_id( 'bordered' ) ); ?>"
						name="<?php echo esc_attr( $this->get_field_name( 'bordered' ) ); ?>"<?php checked( $mwt_bordered ); ?> />
				<label for="<?php echo esc_attr( $this->get_field_id( 'bordered' ) ); ?>">
					<?php esc_html_e( 'Bordered', 'mwt-widgets' ); ?>
				</label>
			</p>

			<p>
				<input type="checkbox" class="checkbox" id="<?php echo esc_attr( $this->get_field_id( 'muted' ) ); ?>"
						name="<?php echo esc_attr( $this->get_field_name( 'muted' ) ); ?>"<?php checked( $mwt_muted ); ?> />
				<label for="<?php echo esc_attr( $this->get_field_id( 'muted' ) ); ?>">
					<?php esc_html_e( 'Muted', 'mwt-widgets' ); ?>
				</label>
			</p>

			<p>
				<input type="checkbox" class="checkbox" id="<?php echo esc_attr( $this->get_field_id( 'center' ) ); ?>"
						name="<?php echo esc_attr( $this->get_field_name( 'center' ) ); ?>"<?php checked( $mwt_center ); ?> />
				<label for="<?php echo esc_attr( $this->get_field_id( 'center' ) ); ?>">
					<?php esc_html_e( 'Align center', 'mwt-widgets' ); ?>
				</label>
			</p>


			<?php
		}

	}

endif;

if ( ! function_exists( 'flicker_theme_register_widget_theme_category' ) ) :
	function flicker_theme_register_widget_theme_category() {
		register_widget( 'MWT_Widget_Theme_Category' );
	}
endif;
add_action( 'widgets_init', 'flicker_theme_register_widget_theme_category' );
