<?php
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

/**
 * Elementor MWT Testimonials Carousel Widget.
 *
 * Elementor widget that inserts an embbedable content into the page, from any given URL.
 *
 * @since 1.0.0
 */

class Elementor_MWT_Testimonials_Carousel extends \Elementor\Widget_Base

{

	/**
	 * Retrieve Widget Dependent JS.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array JS script handles.
	 */
	public function get_script_depends()
	{
		return ['testimonials-carousel-script'];
	}

	/**
	 * Retrieve Widget Dependent CSS.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array CSS style handles.
	 */
	public function get_style_depends()
	{
		return ['owl-carousel-style'];
		return ['testimonials-carousel-style'];
	}

	/**
	 * Get widget name.
	 *
	 * Retrieve list widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return string Widget name.
	 */
	public function get_name()
	{
		return 'testimonials-carousel';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve list widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return string Widget title.
	 */
	public function get_title()
	{
		return esc_html__('MWT Testimonials Carousel', 'mwt-addons-for-elementor');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the list widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array Widget categories.
	 */
	public function get_categories()
	{
		return ['mwt'];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve image carousel widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon()
	{
		return 'eicon-slider-push';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @since 2.1.0
	 * @access public
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords()
	{
		return ['mwt', 'image', 'photo', 'visual', 'testimonials', 'carousel', 'slider'];
	}

	/**
	 * Register list widget controls.
	 *
	 * Add input fields to allow the user to customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls()
	{

		$this->start_controls_section(
			'content_section',
			[
				'label' => esc_html__('Carousel Items', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'item_layout',
			[
				'label' => esc_html__('Testimonials Layout', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'single_carousel',
				'options' => [
					'single_carousel' => esc_html__('Testimonials Single Carousel', 'mwt-addons-for-elementor'),
					'single_carousel_2' => esc_html__('Testimonials Single Carousel 2', 'mwt-addons-for-elementor'),
					'carousel' => esc_html__('Testimonials Carousel', 'mwt-addons-for-elementor'),
					'carousel2' => esc_html__('Testimonials Carousel 2', 'mwt-addons-for-elementor'),
					'carousel3' => esc_html__('Testimonials Carousel 3', 'mwt-addons-for-elementor'),
					'carousel4' => esc_html__('Testimonials Carousel 4', 'mwt-addons-for-elementor'),
				],
				'frontend_available' => true,
			]
		);

		/* Start repeater */

		$repeater = new \Elementor\Repeater();

		$repeater->add_control(
			'image',
			[
				'label' => esc_html__('Choose Image', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::MEDIA,
				'dynamic' => [
					'active' => true,
				],
				'default' => [
					'url' => \Elementor\Utils::get_placeholder_image_src(),
				],
			]
		);

		$repeater->add_control(
			'item_name',
			[
				'label' => esc_html__('Name', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::TEXT,
				'placeholder' => esc_html__('Name', 'mwt-addons-for-elementor'),
				'default' => esc_html__('Name', 'mwt-addons-for-elementor'),
				'description' => __('Enter the Name of the Person to quote', 'mwt-addons-for-elementor'),
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
				'ai' => [
					'active' => false,
				],
			]
		);

		$repeater->add_control(
			'item_position',
			[
				'label'       => __('Position', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::TEXT,
				'placeholder' => esc_html__('Position', 'mwt-addons-for-elementor'),
				'default' => esc_html__('Position', 'mwt-addons-for-elementor'),
				'description' => __('Can be used for a job description', 'mwt-addons-for-elementor'),
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
				'ai' => [
					'active' => false,
				],
			]
		);

		$repeater->add_control(
			'item_quote',
			[
				'label'       => __('Quote', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__('Quote', 'mwt-addons-for-elementor'),
				'default' => esc_html__('Enter the testimonial here', 'mwt-addons-for-elementor'),
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
				'ai' => [
					'active' => false,
				],
			]
		);

		$repeater->add_control(
			'item_date',
			[
				'label'       => __('Date', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::TEXT,
				'placeholder' => esc_html__('Date', 'mwt-addons-for-elementor'),
				'description' => esc_html__('Works only when layout set to \'Testimonials Carousel 4\'', 'mwt-addons-for-elementor'),
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
				'ai' => [
					'active' => false,
				],
			]
		);

		/* End repeater */

		$this->add_control(
			'testimonials-list',
			[
				'label' => esc_html__('Items', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::REPEATER,
				'fields' => $repeater->get_controls(),
				'default' => [
					[
						'image' => '',
						'item_position' => esc_html__('Position 1', 'mwt-addons-for-elementor'),
						'item_name' => esc_html__('Name 1', 'mwt-addons-for-elementor'),
						'item_quote' => esc_html__('Enter the testimonial here...', 'mwt-addons-for-elementor'),
						'item_date' => '',
					],
					[
						'image' => '',
						'item_position' => esc_html__('Position 2', 'mwt-addons-for-elementor'),
						'item_name' => esc_html__('Name 2', 'mwt-addons-for-elementor'),
						'item_quote' => esc_html__('Enter the testimonial here...', 'mwt-addons-for-elementor'),
						'item_date' => '',
					],
					[
						'image' => '',
						'item_position' => esc_html__('Position 3', 'mwt-addons-for-elementor'),
						'item_name' => esc_html__('Name 3', 'mwt-addons-for-elementor'),
						'item_quote' => esc_html__('Enter the testimonial here...', 'mwt-addons-for-elementor'),
						'item_date' => '',
					],
				],
				'title_field' => '{{{ item_name }}}',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Image_Size::get_type(),
			[
				'name' => 'thumbnail', // Usage: `{name}_size` and `{name}_custom_dimension`, in this case `thumbnail_size` and `thumbnail_custom_dimension`.
				'separator' => 'none',
			]
		);

		$this->add_control(
			'quote_icon',
			[
				'label'            => __('Quote Icon', 'mwt-addons-for-elementor'),
				'type'             => \Elementor\Controls_Manager::ICONS,
				'fa4compatibility' => 'icon',
				'default'          => [
					'value'   => 'fas fa-quote-left',
					'library' => 'fa-solid',
				],
				'recommended'      => [
					'fa-solid'   => [
						'quote-left',
					],
				],
				'skin'             => 'inline',
				'label_block'      => false,
				'condition' => [
					'item_layout!' => 'carousel4',
				],
			]
		);

		$this->add_control(
			'date_icon',
			[
				'label'            => __('Date Icon', 'mwt-addons-for-elementor'),
				'type'             => \Elementor\Controls_Manager::ICONS,
				'fa4compatibility' => 'icon',
				'default'          => [
					'value'   => 'far fa-clock',
					'library' => 'fa-regular',
				],
				'recommended'      => [
					'fa-regular'   => [
						'quote-left',
					],
				],
				'skin'             => 'inline',
				'label_block'      => false,
				'condition' => [
					'item_layout' => 'carousel4',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'carousel_settings',
			[
				'label' => __('Carousel Settings', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'data_loop',
			[
				'label'   => __('Loop', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_autoplay',
			[
				'label'   => __('Autoplay', 'mwt-addons-for-elementor'),
				'description' => esc_html__('Slide will start automatically', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_autoplaytimeout',
			[
				'label' => esc_html__('Autoplay Speed (ms)', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'default' => 5000,
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_autoplay_hover_pause',
			[
				'label'   => __('Pause on Hover', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_center',
			[
				'label'   => __('Centered Carousel?', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
				'frontend_available' => true,
				'condition' => [
					'item_layout' => ['carousel', 'carousel2', 'carousel4'],
				],
			]
		);

		//Slides to Scroll

		$this->add_control(
			'data_responsive_lg',
			[
				'label' => esc_html__('Desktop Slides', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'default' => 3,
				'frontend_available' => true,
				'condition' => [
					'item_layout' => ['carousel', 'carousel2', 'carousel3', 'carousel4'],
				],
			]
		);

		$this->add_control(
			'data_responsive_sm',
			[
				'label' => esc_html__('Tabs Slides', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'default' => 2,
				'frontend_available' => true,
				'condition' => [
					'item_layout' => ['carousel', 'carousel2', 'carousel3', 'carousel4'],
				],
			]
		);

		$this->add_control(
			'data_responsive_xs',
			[
				'label' => esc_html__('Mobile Slides', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'default' => 1,
				'frontend_available' => true,
				'condition' => [
					'item_layout' => ['carousel', 'carousel2', 'carousel3', 'carousel4'],
				],
			]
		);

		$this->add_control(
			'data_margin',
			[
				'label' => esc_html__('Item Gap (px)', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'default' => 15,
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_nav',
			[
				'label'   => __('Navigation Arrows', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
				'separator' => 'before',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_dots',
			[
				'label'   => __('Navigation Dots', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
				'frontend_available' => true,
				'condition' => [
					'item_layout!' => ['carousel3'],
				],
			]
		);


		$this->end_controls_section();

		// style
		$this->start_controls_section(
			'style_item_section',
			[
				'label' => esc_html__('Item', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'item_bg_color',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-testimonials-carousel .owl-carousel-item' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'item_border',
				'selector' => '{{WRAPPER}} .testimonials-carousel4 .owl-carousel-item',
				'condition' => [
					'item_layout' => 'carousel4',
				],
			]
		);

		$this->add_control(
			'item_border_radius',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .testimonials-carousel4 .owl-carousel-item'  => 'border-radius: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'item_layout' => 'carousel4',
				],
			]
		);

		$this->add_responsive_control(
			'testimonials_item_align',
			[
				'label'     => __('Alignment', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::CHOOSE,
				'options'   => [
					'left' => [
						'title' => __('Left', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-left',
					],
					'center'     => [
						'title' => __('Center', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-center',
					],
					'right'   => [
						'title' => __('Right', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-right',
					],
				],
				'default'   => 'Left',
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-testimonials-carousel-image-wrap' => 'text-align: {{VALUE}}',
					'{{WRAPPER}} .mwt-elementor-testimonials-carousel-position' => 'text-align: {{VALUE}}',
					'{{WRAPPER}} .mwt-elementor-testimonials-carousel-name' => 'text-align: {{VALUE}}',
					'{{WRAPPER}} .mwt-elementor-testimonials-carousel-quote' => 'text-align: {{VALUE}}',
					'{{WRAPPER}} .mwt-elementor-testimonials-carousel-date' => 'text-align: {{VALUE}}',
				],
				'condition' => [
					'item_layout' => ['carousel', 'carousel4'],
				],
			]
		);

		$this->add_responsive_control(
			'item_padding',
			[
				'label'      => __('Padding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-testimonials-carousel .owl-carousel-item' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_image_section',
			[
				'label' => esc_html__('Image', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'item_layout!' => 'carousel2',
				],
			]
		);

		$this->add_responsive_control(
			'image_size',
			[
				'label'      => __('Image Size', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em'],
				'range'      => [
					'px' => [
						'min'  => 50,
						'max'  => 300,
						'step' => 1,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 270,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-testimonials-carousel-image' => 'max-width: {{SIZE}}{{UNIT}}; min-width: {{SIZE}}{{UNIT}}; max-height: {{SIZE}}{{UNIT}}; min-height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'image_border',
				'selector' => '{{WRAPPER}} .mwt-elementor-testimonials-carousel-image',
			]
		);

		$this->add_control(
			'image_border_radius',
			[
				'label'      => esc_html__('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-testimonials-carousel-image' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'image_bottom_spacing',
			[
				'label'      => __('Bottom Spacing', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-testimonials-carousel-image-wrap'  => 'margin-bottom: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_name_section',
			[
				'label' => esc_html__('Name', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'name_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-testimonials-carousel-name' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'name_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-elementor-testimonials-carousel-name, {{WRAPPER}}',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Text_Shadow::get_type(),
			[
				'label' => esc_html__('Text Shadow', 'mwt-addons-for-elementor'),
				'name' => 'name_shadow',
				'selector' => '{{WRAPPER}} .mwt-elementor-testimonials-carousel-name',
			]
		);

		$this->add_responsive_control(
			'name_margin',
			[
				'label'      => esc_html__('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-testimonials-carousel-name' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_position_section',
			[
				'label' => esc_html__('Position', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'position_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-testimonials-carousel-position' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'position_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-elementor-testimonials-carousel-position',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Text_Shadow::get_type(),
			[
				'label' => esc_html__('Text Shadow', 'mwt-addons-for-elementor'),
				'name' => 'position_shadow',
				'selector' => '{{WRAPPER}} .mwt-elementor-testimonials-carousel-position',
			]
		);

		$this->add_responsive_control(
			'position_margin',
			[
				'label'      => esc_html__('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-testimonials-carousel-position' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_quote_section',
			[
				'label' => esc_html__('Quote', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'quote_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-testimonials-carousel-quote' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'quote_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-elementor-testimonials-carousel-quote',
			]
		);

		$this->add_responsive_control(
			'quote_margin',
			[
				'label'      => esc_html__('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%', 'em', 'rem', 'vw', 'custom'],
				'allowed_dimensions' => 'vertical',
				'placeholder' => [
					'top' => '',
					'right' => 'auto',
					'bottom' => '',
					'left' => 'auto',
				],
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-testimonials-carousel-quote' => 'margin-top: {{TOP}}{{UNIT}}; margin-bottom: {{BOTTOM}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'quote_icon_color',
			[
				'label' => esc_html__('Icon Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'separator' => 'before',
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-testimonials-carousel-quote-icon i' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-elementor-testimonials-carousel-quote-icon svg' => 'fill: {{VALUE}};',
				],
				'condition' => [
					'item_layout!' => 'carousel4'
				]
			]
		);

		$this->add_responsive_control(
			'icon_size',
			[
				'label'      => __('Icon Size', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em'],
				'range'      => [
					'px' => [
						'min'  => 8,
						'max'  => 64,
						'step' => 1,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 32,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-testimonials-carousel-quote-icon i' => 'font-size: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .mwt-elementor-testimonials-carousel-quote-icon svg' => 'max-width: {{SIZE}}{{UNIT}}; max-height: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'item_layout!' => 'carousel4'
				]
			]
		);

		$this->add_responsive_control(
			'icon_rotate',
			[
				'label'      => __('Icon Rotate', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => '',
				'range'      => [
					'px' => [
						'min'  => -360,
						'max'  => 360,
						'step' => 1,
					],
				],
				'default'    => [
					'size' => 0,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-testimonials-carousel-quote-icon i' => 'transform: rotate({{SIZE}}deg);',
					'{{WRAPPER}} .mwt-elementor-testimonials-carousel-quote-icon svg' => 'transform: rotate({{SIZE}}deg);',
				],
				'condition' => [
					'item_layout!' => ['carousel4', 'single_carousel'],
				]
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_date_section',
			[
				'label' => esc_html__('Date', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'item_layout' => 'carousel4'
				]
			]
		);

		$this->add_control(
			'date_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-testimonials-carousel-date' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'date_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-elementor-testimonials-carousel-date, {{WRAPPER}}',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Text_Shadow::get_type(),
			[
				'label' => esc_html__('Text Shadow', 'mwt-addons-for-elementor'),
				'name' => 'date_shadow',
				'selector' => '{{WRAPPER}} .mwt-elementor-testimonials-carousel-date',
			]
		);

		$this->add_responsive_control(
			'date_margin',
			[
				'label'      => esc_html__('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-testimonials-carousel-date' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'date_icon_color',
			[
				'label' => esc_html__('Icon Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'separator' => 'before',
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-testimonials-carousel-date-icon i' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-elementor-testimonials-carousel-date-icon svg' => 'fill: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'date_icon_size',
			[
				'label'      => __('Icon Size', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em'],
				'range'      => [
					'px' => [
						'min'  => 8,
						'max'  => 64,
						'step' => 1,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 32,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-testimonials-carousel-date-icon i' => 'font-size: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .mwt-elementor-testimonials-carousel-date-icon svg' => 'max-width: {{SIZE}}{{UNIT}}; max-height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'date_icon_spacing',
			[
				'label'      => __('Spacing', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-testimonials-carousel-date-icon'  => 'margin-right: {{SIZE}}{{UNIT}}',
				],
			]
		);


		$this->end_controls_section();

		$this->start_controls_section(
			'style_arrows_section',
			[
				'label' => esc_html__('Arrows', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'data_nav' => 'yes',
				],
			]
		);


		$this->add_control(
			'arrows_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .testimonials-slider .owl-nav > button:after' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'arrows_color_hover',
			[
				'label' => esc_html__('Color Hover', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .testimonials-slider .owl-nav > button:hover:after' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'arrows_bg_color',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .testimonials-slider .owl-nav > button' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'arrows_bg_color_hover',
			[
				'label' => esc_html__('Background Color Hover', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .testimonials-slider .owl-nav > button:hover' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'nav_icon_size',
			[
				'label'      => __('Icon Size', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em'],
				'range'      => [
					'px' => [
						'min'  => 10,
						'max'  => 40,
						'step' => 1,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 10,
				],
				'selectors'  => [
					'{{WRAPPER}} .testimonials-slider .owl-nav > button:after' => 'font-size: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'arrows_border_radius',
			[
				'label'      => esc_html__('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .testimonials-slider .owl-nav > button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'arrows_position_x',
			[
				'label'      => __('Position X', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%'],
				'range'      => [
					'px' => [
						'min'  => -1000,
						'max'  => 1000,
						'step' => 1,
					],
					'%' => [
						'min'  => -100,
						'max'  => 100,
						'step' => 1,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 0,
				],
				'selectors'  => [
					'{{WRAPPER}} .testimonials-carousel3 .owl-theme .owl-nav' => 'left: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'item_layout' => 'carousel3'
				]
			]
		);

		$this->add_responsive_control(
			'arrows_position_y',
			[
				'label'      => __('Position Y', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%'],
				'range'      => [
					'px' => [
						'min'  => -1000,
						'max'  => 1000,
						'step' => 1,
					],
					'%' => [
						'min'  => -100,
						'max'  => 100,
						'step' => 1,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => -80,
				],
				'selectors'  => [
					'{{WRAPPER}} .testimonials-carousel3 .owl-theme .owl-nav' => 'bottom: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'item_layout' => 'carousel3'
				]
			]
		);

		$this->add_responsive_control(
			'arrows_padding',
			[
				'label'      => esc_html__('Pdding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .testimonials-slider .owl-nav > button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition' => [
					'item_layout!' => 'carousel3'
				]
			]
		);

		$this->add_responsive_control(
			'arrows_margin',
			[
				'label'      => esc_html__('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .testimonials-slider .owl-nav > button' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition' => [
					'item_layout!' => 'carousel3'
				]
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_dots_section',
			[
				'label' => esc_html__('Dots', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'data_dots' => 'yes',
				],
			]
		);

		$this->start_controls_tabs(
			'style_tabs'
		);

		$this->start_controls_tab(
			'style_normal_tab',
			[
				'label' => esc_html__('Normal', 'textdomain'),
			]
		);

		$this->add_control(
			'dots_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .testimonials-slider .owl-dots > .owl-dot span' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'style_hover_tab',
			[
				'label' => esc_html__('Hover', 'textdomain'),
			]
		);

		$this->add_control(
			'dots_color_hover',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .testimonials-slider .owl-dots > .owl-dot span:hover' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'dots_border_color_hover',
			[
				'label' => esc_html__('Border Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .testimonials-slider .owl-dots > .owl-dot span:hover' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'style_active_tab',
			[
				'label' => esc_html__('Active', 'textdomain'),
			]
		);

		$this->add_control(
			'dots_color_active',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .testimonials-slider .owl-dots > .owl-dot.active span' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'dots_border_color_active',
			[
				'label' => esc_html__('Border Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .testimonials-slider .owl-dots > .owl-dot.active span' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_responsive_control(
			'dots_size_width',
			[
				'label'      => __('Width', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em'],
				'range'      => [
					'px' => [
						'min'  => 5,
						'max'  => 50,
						'step' => 1,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 10,
				],
				'selectors'  => [
					'{{WRAPPER}} .testimonials-slider .owl-dots > .owl-dot span' => 'width: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'dots_size_height',
			[
				'label'      => __('Height', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em'],
				'range'      => [
					'px' => [
						'min'  => 5,
						'max'  => 50,
						'step' => 1,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 10,
				],
				'selectors'  => [
					'{{WRAPPER}} .testimonials-slider .owl-dots > .owl-dot span' => 'height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'dots_border',
				'selector' => '{{WRAPPER}} .testimonials-slider .owl-dots > .owl-dot span',
			]
		);

		$this->add_control(
			'dots_border_radius',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .testimonials-slider .owl-dots > .owl-dot span'  => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'dots_margin',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .testimonials-slider .owl-dots > .owl-dot span' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render list widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render()
	{
		$settings = $this->get_settings_for_display();

		$unique_id = uniqid();

		if (empty($settings['testimonials-list'])) {
			return;
		}

		$item_layout = $settings['item_layout'];
		switch ($item_layout) {
			case 'single_carousel': ?>
				<div class="owl-carousel-container mwt-testimonials-carousel testimonials-single-carousel">
					<?php
					$data_loop = ($settings['data_loop'] === 'yes') ? 'true' : 'false';
					$data_autoplay = ($settings['data_autoplay'] === 'yes') ? 'true' : 'false';
					$data_center = 'false';
					$data_nav = ($settings['data_nav'] === 'yes') ? 'true' : 'false';
					$data_dots = ($settings['data_dots'] === 'yes') ? 'true' : 'false';
					$data_autoplay_hover_pause = ($settings['data_autoplay_hover_pause'] === 'yes') ? 'true' : 'false';
					?>
					<div id="widget_portfolio_carousel_<?php echo esc_attr($unique_id); ?>" class="owl-carousel testimonials-slider" data-loop="<?php echo esc_attr($data_loop) ?>" data-autoplay="<?php echo esc_attr($data_autoplay) ?>" data-autoplaytimeout="<?php echo esc_attr($settings['data_autoplaytimeout']) ?>" data-autoplayhoverpause="<?php echo esc_attr($data_autoplay_hover_pause) ?>" data-center="<?php echo esc_attr($data_center) ?>" data-nav="<?php echo esc_attr($data_nav) ?>" data-dots="<?php echo esc_attr($data_dots) ?>" data-margin="<?php echo esc_attr($settings['data_margin']) ?>" data-responsive-lg="1" data-responsive-sm="1" data-responsive-xs="1">
						<?php
						foreach ($settings['testimonials-list'] as $index => $item) { ?>
							<div class="owl-carousel-item text-center">
								<?php
								$image_url = \Elementor\Group_Control_Image_Size::get_attachment_image_src($item['image']['id'], 'thumbnail', $settings);
								if (!$image_url && isset($item['image']['url'])) {
									$image_url = $item['image']['url'];
								}
								if (!empty($item['item_name'])) {
									$image_alt = $item['item_name'];
								} else {
									$image_alt = 'This is a name';
								}
								$image_html = '<img class="mwt-elementor-testimonials-carousel-image" src="' . esc_attr($image_url) . '" alt="' . esc_attr($image_alt) . '" />';
								if (!empty($item['image']['url'])) {
								?>
									<div class="mwt-elementor-testimonials-carousel-image-wrap">
										<?php
										echo wp_kses_post($image_html);
										?>
									</div>
								<?php
								}
								$name = $settings['testimonials-list'][$index]['item_name'];
								$position = $settings['testimonials-list'][$index]['item_position'];
								$quote = $settings['testimonials-list'][$index]['item_quote'];
								$quote_icon = isset($settings['quote_icon']);
								?>
								<div class="mwt-elementor-testimonials-carousel-quote">
									<?php if ($quote_icon) : ?>
										<span class="mwt-elementor-testimonials-carousel-quote-icon">
											<?php \Elementor\Icons_Manager::render_icon($settings['quote_icon'], ['aria-hidden' => 'true', 'class' => 'icon-left']); ?>
										</span>
									<?php endif; ?>
									<?php echo $quote; ?>
									<?php if ($quote_icon) : ?>
										<span class="mwt-elementor-testimonials-carousel-quote-icon">
											<?php \Elementor\Icons_Manager::render_icon($settings['quote_icon'], ['aria-hidden' => 'true', 'class' => 'icon-right']); ?>
										</span>
									<?php endif; ?>
								</div>
								<div class="mwt-elementor-testimonials-carousel-name">
									<?php echo $name; ?>
								</div>
								<div class="mwt-elementor-testimonials-carousel-position">
									<?php echo $position; ?>
								</div>
							</div>
						<?php } ?>
					</div>
				</div>
			<?php break;

			case 'single_carousel_2': ?>
				<div class="owl-carousel-container mwt-testimonials-carousel testimonials-single-carousel-2">
					<?php
					$data_loop = ($settings['data_loop'] === 'yes') ? 'true' : 'false';
					$data_autoplay = ($settings['data_autoplay'] === 'yes') ? 'true' : 'false';
					$data_center = 'false';
					$data_nav = ($settings['data_nav'] === 'yes') ? 'true' : 'false';
					$data_dots = ($settings['data_dots'] === 'yes') ? 'true' : 'false';
					?>
					<div id="widget_portfolio_carousel_<?php echo esc_attr($unique_id); ?>" class="owl-carousel testimonials-slider" data-loop="<?php echo esc_attr($data_loop) ?>" data-autoplay="<?php echo esc_attr($data_autoplay) ?>" data-autoplaytimeout="<?php echo esc_attr($settings['data_autoplaytimeout']) ?>" data-center="<?php echo esc_attr($data_center) ?>" data-nav="<?php echo esc_attr($data_nav) ?>" data-dots="<?php echo esc_attr($data_dots) ?>" data-margin="<?php echo esc_attr($settings['data_margin']) ?>" data-responsive-lg="1" data-responsive-sm="1" data-responsive-xs="1">
						<?php
						foreach ($settings['testimonials-list'] as $index => $item) { ?>
							<div class="owl-carousel-item text-center">
								<?php
								$image_url = \Elementor\Group_Control_Image_Size::get_attachment_image_src($item['image']['id'], 'thumbnail', $settings);
								if (!$image_url && isset($item['image']['url'])) {
									$image_url = $item['image']['url'];
								}
								if (!empty($item['item_name'])) {
									$image_alt = $item['item_name'];
								} else {
									$image_alt = 'This is a name';
								}
								$image_html = '<img class="mwt-elementor-testimonials-carousel-image" src="' . esc_attr($image_url) . '" alt="' . esc_attr($image_alt) . '" />';

								if (!empty($item['image']['url'])) {
								?>
									<div class="mwt-elementor-testimonials-carousel-image-wrap">
										<?php
										echo wp_kses_post($image_html);
										?>
									</div>
								<?php
								}
								$position = $settings['testimonials-list'][$index]['item_position'];
								$name = $settings['testimonials-list'][$index]['item_name'];
								$quote = $settings['testimonials-list'][$index]['item_quote'];
								$quote_icon = isset($settings['quote_icon']);
								?>
								<div class="mwt-elementor-testimonials-carousel-quote">
									<?php if ($quote_icon) : ?>
										<span class="mwt-elementor-testimonials-carousel-quote-icon icon-left">
											<?php \Elementor\Icons_Manager::render_icon($settings['quote_icon'], ['aria-hidden' => 'true', 'class' => 'icon-left']); ?>
										</span>
									<?php endif; ?>
									<?php echo $quote; ?>
									<?php if ($quote_icon) : ?>
										<span class="mwt-elementor-testimonials-carousel-quote-icon icon-right">
											<?php \Elementor\Icons_Manager::render_icon($settings['quote_icon'], ['aria-hidden' => 'true', 'class' => 'icon-right']); ?>
										</span>
									<?php endif; ?>
								</div>
								<div class="mwt-elementor-testimonials-carousel-name">
									<?php echo $name; ?>
								</div>
								<div class="mwt-elementor-testimonials-carousel-position">
									<?php echo $position; ?>
								</div>
							</div>
						<?php } ?>
					</div>
				</div>
			<?php break;

			case 'carousel': ?>
				<div class="owl-carousel-container mwt-testimonials-carousel testimonials-carousel">
					<?php
					$data_loop = ($settings['data_loop'] === 'yes') ? 'true' : 'false';
					$data_autoplay = ($settings['data_autoplay'] === 'yes') ? 'true' : 'false';
					$data_center = ($settings['data_center'] === 'yes') ? 'true' : 'false';
					$data_nav = ($settings['data_nav'] === 'yes') ? 'true' : 'false';
					$data_dots = ($settings['data_dots'] === 'yes') ? 'true' : 'false';
					?>
					<div id="widget_portfolio_carousel_<?php echo esc_attr($unique_id); ?>" class="owl-carousel testimonials-slider" data-loop="<?php echo esc_attr($data_loop) ?>" data-autoplay="<?php echo esc_attr($data_autoplay) ?>" data-autoplaytimeout="<?php echo esc_attr($settings['data_autoplaytimeout']) ?>" data-center="<?php echo esc_attr($data_center) ?>" data-nav="<?php echo esc_attr($data_nav) ?>" data-dots="<?php echo esc_attr($data_dots) ?>" data-margin="<?php echo esc_attr($settings['data_margin']) ?>" data-responsive-lg="<?php echo esc_attr($settings['data_responsive_lg']) ?>" data-responsive-sm="<?php echo esc_attr($settings['data_responsive_sm']) ?>" data-responsive-xs="<?php echo esc_attr($settings['data_responsive_xs']) ?>">
						<?php
						foreach ($settings['testimonials-list'] as $index => $item) { ?>
							<div class="owl-carousel-item">
								<?php
								$quote_icon = isset($settings['quote_icon']);
								$image_url = \Elementor\Group_Control_Image_Size::get_attachment_image_src($item['image']['id'], 'thumbnail', $settings);
								if (!$image_url && isset($item['image']['url'])) {
									$image_url = $item['image']['url'];
								}
								if (!empty($item['item_name'])) {
									$image_alt = $item['item_name'];
								} else {
									$image_alt = 'This is a name';
								}
								$image_html = '<img class="mwt-elementor-testimonials-carousel-image" src="' . esc_attr($image_url) . '" alt="' . esc_attr($image_alt) . '" />';
								if (!empty($item['image']['url'])) {
								?>
									<div class="mwt-elementor-testimonials-carousel-image-wrap">
										<span>
											<?php
											echo wp_kses_post($image_html);
											?>
											<?php if ($quote_icon) : ?>
												<span class="mwt-elementor-testimonials-carousel-quote-icon">
													<?php \Elementor\Icons_Manager::render_icon($settings['quote_icon'], ['aria-hidden' => 'true', 'class' => 'icon-left']); ?>
												</span>
											<?php endif; ?>
										</span>
									</div>
								<?php
								}
								$position = $settings['testimonials-list'][$index]['item_position'];
								$name = $settings['testimonials-list'][$index]['item_name'];
								$quote = $settings['testimonials-list'][$index]['item_quote'];
								?>
								<div class="mwt-elementor-testimonials-carousel-position">
									<?php echo $position; ?>
								</div>

								<div class="mwt-elementor-testimonials-carousel-name">
									<?php echo $name; ?>
								</div>

								<div class="mwt-elementor-testimonials-carousel-quote">
									<?php echo $quote; ?>
								</div>
							</div>
						<?php } ?>
					</div>
				</div>
			<?php break;

			case 'carousel2': ?>
				<div class="owl-carousel-container mwt-testimonials-carousel testimonials-carousel2">
					<?php
					$data_loop = ($settings['data_loop'] === 'yes') ? 'true' : 'false';
					$data_autoplay = ($settings['data_autoplay'] === 'yes') ? 'true' : 'false';
					$data_center = ($settings['data_center'] === 'yes') ? 'true' : 'false';
					$data_nav = ($settings['data_nav'] === 'yes') ? 'true' : 'false';
					$data_dots = ($settings['data_dots'] === 'yes') ? 'true' : 'false';
					?>
					<div id="widget_portfolio_carousel_<?php echo esc_attr($unique_id); ?>" class="owl-carousel testimonials-slider" data-loop="<?php echo esc_attr($data_loop) ?>" data-autoplay="<?php echo esc_attr($data_autoplay) ?>" data-autoplaytimeout="<?php echo esc_attr($settings['data_autoplaytimeout']) ?>" data-center="<?php echo esc_attr($data_center) ?>" data-nav="<?php echo esc_attr($data_nav) ?>" data-dots="<?php echo esc_attr($data_dots) ?>" data-margin="<?php echo esc_attr($settings['data_margin']) ?>" data-responsive-lg="<?php echo esc_attr($settings['data_responsive_lg']) ?>" data-responsive-sm="<?php echo esc_attr($settings['data_responsive_sm']) ?>" data-responsive-xs="<?php echo esc_attr($settings['data_responsive_xs']) ?>">
						<?php
						foreach ($settings['testimonials-list'] as $index => $item) { ?>
							<div class="owl-carousel-item text-center">
								<?php
								$quote_icon = isset($settings['quote_icon']);
								$position = $settings['testimonials-list'][$index]['item_position'];
								$name = $settings['testimonials-list'][$index]['item_name'];
								$quote = $settings['testimonials-list'][$index]['item_quote'];
								?>
								<div class="mwt-elementor-testimonials-carousel-position">
									<?php echo $position; ?>
								</div>

								<div class="mwt-elementor-testimonials-carousel-name">
									<?php if ($quote_icon) : ?>
										<span class="mwt-elementor-testimonials-carousel-quote-icon">
											<?php \Elementor\Icons_Manager::render_icon($settings['quote_icon'], ['aria-hidden' => 'true', 'class' => 'icon-left']); ?>
										</span>
									<?php endif; ?>
									<?php echo $name; ?>
									<?php if ($quote_icon) : ?>
										<span class="mwt-elementor-testimonials-carousel-quote-icon">
											<?php \Elementor\Icons_Manager::render_icon($settings['quote_icon'], ['aria-hidden' => 'true', 'class' => 'icon-right']); ?>
										</span>
									<?php endif; ?>
								</div>

								<div class="mwt-elementor-testimonials-carousel-quote">
									<?php echo $quote; ?>
								</div>
							</div>
						<?php } ?>
					</div>
				</div>
			<?php break;
			case 'carousel3': ?>
				<div class="owl-carousel-container mwt-testimonials-carousel testimonials-carousel3">
					<?php
					$data_loop = ($settings['data_loop'] === 'yes') ? 'true' : 'false';
					$data_autoplay = ($settings['data_autoplay'] === 'yes') ? 'true' : 'false';
					$data_center = ($settings['data_center'] === 'yes') ? 'true' : 'false';
					$data_nav = ($settings['data_nav'] === 'yes') ? 'true' : 'false';
					$data_dots = ($settings['data_dots'] === 'yes') ? 'true' : 'false';
					?>
					<div id="widget_portfolio_carousel_<?php echo esc_attr($unique_id); ?>" class="owl-carousel testimonials-slider" data-loop="<?php echo esc_attr($data_loop) ?>" data-autoplay="<?php echo esc_attr($data_autoplay) ?>" data-autoplaytimeout="<?php echo esc_attr($settings['data_autoplaytimeout']) ?>" data-center="<?php echo esc_attr($data_center) ?>" data-nav="<?php echo esc_attr($data_nav) ?>" data-dots="<?php echo esc_attr($data_dots) ?>" data-margin="<?php echo esc_attr($settings['data_margin']) ?>" data-responsive-lg="<?php echo esc_attr($settings['data_responsive_lg']) ?>" data-responsive-sm="<?php echo esc_attr($settings['data_responsive_sm']) ?>" data-responsive-xs="<?php echo esc_attr($settings['data_responsive_xs']) ?>">
						<?php
						foreach ($settings['testimonials-list'] as $index => $item) { ?>
							<div class="owl-carousel-item">
								<?php
								$quote_icon = isset($settings['quote_icon']);
								$quote = $settings['testimonials-list'][$index]['item_quote'];
								?>
								<div class="mwt-elementor-testimonials-carousel-quote">
									<?php if ($quote_icon) : ?>
										<span class="mwt-elementor-testimonials-carousel-quote-icon">
											<?php \Elementor\Icons_Manager::render_icon($settings['quote_icon'], ['aria-hidden' => 'true', 'class' => 'icon-left']); ?>
										</span>
									<?php endif; ?>
									<?php echo $quote; ?>
								</div>
								<?php
								$image_url = \Elementor\Group_Control_Image_Size::get_attachment_image_src($item['image']['id'], 'thumbnail', $settings);
								if (!$image_url && isset($item['image']['url'])) {
									$image_url = $item['image']['url'];
								}
								if (!empty($item['item_name'])) {
									$image_alt = $item['item_name'];
								} else {
									$image_alt = 'This is a name';
								}
								$image_html = '<img class="mwt-elementor-testimonials-carousel-image" src="' . esc_attr($image_url) . '" alt="' . esc_attr($image_alt) . '" />';
								if (!empty($item['image']['url'])) {
								?>
									<div class="mwt-elementor-testimonials-quote-bottom">
										<div class="mwt-elementor-testimonials-carousel-image-wrap">
											<span>
												<?php
												echo wp_kses_post($image_html);
												?>
											</span>
										</div>
									<?php
								}
								$position = $settings['testimonials-list'][$index]['item_position'];
								$name = $settings['testimonials-list'][$index]['item_name'];

									?>
									<div class="mwt-elementor-testimonials-quote-bottom-content">
										<div class="mwt-elementor-testimonials-carousel-name">
											<?php echo $name; ?>
										</div>
										<div class="mwt-elementor-testimonials-carousel-position">
											<?php echo $position; ?>
										</div>
									</div>
									</div>
							</div>
						<?php } ?>
					</div>
				</div>
			<?php break;
			case 'carousel4': ?>
				<div class="owl-carousel-container mwt-testimonials-carousel testimonials-carousel4">
					<?php
					$data_loop = ($settings['data_loop'] === 'yes') ? 'true' : 'false';
					$data_autoplay = ($settings['data_autoplay'] === 'yes') ? 'true' : 'false';
					$data_center = ($settings['data_center'] === 'yes') ? 'true' : 'false';
					$data_nav = ($settings['data_nav'] === 'yes') ? 'true' : 'false';
					$data_dots = ($settings['data_dots'] === 'yes') ? 'true' : 'false';
					?>
					<div id="widget_portfolio_carousel_<?php echo esc_attr($unique_id); ?>" class="owl-carousel testimonials-slider" data-loop="<?php echo esc_attr($data_loop) ?>" data-autoplay="<?php echo esc_attr($data_autoplay) ?>" data-autoplaytimeout="<?php echo esc_attr($settings['data_autoplaytimeout']) ?>" data-center="<?php echo esc_attr($data_center) ?>" data-nav="<?php echo esc_attr($data_nav) ?>" data-dots="<?php echo esc_attr($data_dots) ?>" data-margin="<?php echo esc_attr($settings['data_margin']) ?>" data-responsive-lg="<?php echo esc_attr($settings['data_responsive_lg']) ?>" data-responsive-sm="<?php echo esc_attr($settings['data_responsive_sm']) ?>" data-responsive-xs="<?php echo esc_attr($settings['data_responsive_xs']) ?>">
						<?php
						foreach ($settings['testimonials-list'] as $index => $item) { ?>
							<div class="owl-carousel-item">
								<?php
								$image_url = \Elementor\Group_Control_Image_Size::get_attachment_image_src($item['image']['id'], 'thumbnail', $settings);
								if (!$image_url && isset($item['image']['url'])) {
									$image_url = $item['image']['url'];
								}
								if (!empty($item['item_name'])) {
									$image_alt = $item['item_name'];
								} else {
									$image_alt = 'This is a name';
								}
								$image_html = '<img class="mwt-elementor-testimonials-carousel-image" src="' . esc_attr($image_url) . '" alt="' . esc_attr($image_alt) . '" />';
								if (!empty($item['image']['url'])) {
								?>
									<div class="mwt-elementor-testimonials-wrap">
										<div class="mwt-elementor-testimonials-carousel-image-wrap">
											<?php echo wp_kses_post($image_html); ?>
										</div>
									<?php
								}
								$name = $settings['testimonials-list'][$index]['item_name'];
								$position = $settings['testimonials-list'][$index]['item_position'];
								$quote = $settings['testimonials-list'][$index]['item_quote'];
								$date_icon = isset($settings['date_icon']);
								$date = $settings['testimonials-list'][$index]['item_date'];

									?>
									<div class="mwt-elementor-testimonials-quote-bottom-content">
										<div class="mwt-elementor-testimonials-carousel-name">
											<?php echo $name; ?>
										</div>
										<div class="mwt-elementor-testimonials-carousel-position">
											<?php echo $position; ?>
										</div>
										<div class="mwt-elementor-testimonials-carousel-quote">
											<?php echo $quote; ?>
										</div>
										<div class="mwt-elementor-testimonials-carousel-date">
											<?php if ($date_icon) : ?>
												<span class="mwt-elementor-testimonials-carousel-date-icon">
													<?php \Elementor\Icons_Manager::render_icon($settings['date_icon'], ['aria-hidden' => 'true', 'class' => 'icon-left']); ?>
												</span>
											<?php endif; ?>
											<?php echo $date; ?>
										</div>
									</div>
									</div>
							</div>
						<?php } ?>
					</div>
				</div>
		<?php break;
		} ?>
<?php }
}
