<?php
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

/**
 * Elementor MWT Team Slider Widget.
 *
 * Elementor widget that inserts an embbedable content into the page, from any given URL.
 *
 * @since 1.0.0
 */

class Elementor_MWT_Team_Slider_Widget extends \Elementor\Widget_Base

{

    /**
     * Retrieve Widget Dependent JS.
     *
     * @since 1.0.0
     * @access public
     * @return array JS script handles.
     */
    public function get_script_depends()
    {
        return ['team-slider-script'];
    }

    /**
     * Retrieve Widget Dependent CSS.
     *
     * @since 1.0.0
     * @access public
     * @return array CSS style handles.
     */
    public function get_style_depends()
    {
        return ['team-slider-style'];
    }

    /**
     * Get widget name.
     *
     * Retrieve list widget name.
     *
     * @since 1.0.0
     * @access public
     * @return string Widget name.
     */
    public function get_name()
    {
        return 'team-slider';
    }

    /**
     * Get widget title.
     *
     * Retrieve list widget title.
     *
     * @since 1.0.0
     * @access public
     * @return string Widget title.
     */
    public function get_title()
    {
        return esc_html__('MWT Team Slider', 'mwt-addons-for-elementor');
    }

    /**
     * Get widget categories.
     *
     * Retrieve the list of categories the list widget belongs to.
     *
     * @since 1.0.0
     * @access public
     * @return array Widget categories.
     */
    public function get_categories()
    {
        return ['mwt'];
    }

    /**
     * Get widget icon.
     *
     * Retrieve image carousel widget icon.
     *
     * @since 1.0.0
     * @access public
     *
     * @return string Widget icon.
     */
    public function get_icon()
    {
        return 'eicon-slides';
    }

    /**
     * Get widget keywords.
     *
     * Retrieve the list of keywords the widget belongs to.
     *
     * @since 2.1.0
     * @access public
     *
     * @return array Widget keywords.
     */
    public function get_keywords()
    {
        return ['mwt', 'image', 'photo', 'team', 'visual', 'carousel', 'slider'];
    }

    /**
     * Register list widget controls.
     *
     * Add input fields to allow the user to customize the widget settings.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function register_controls()
    {

        $this->start_controls_section(
            'content_section',
            [
                'label' => esc_html__('Members Settings', 'mwt-addons-for-elementor'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'title',
            [
                'label' => esc_html__('Title', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'placeholder' => esc_html__('Title', 'mwt-addons-for-elementor'),
                'default' => esc_html__('Title', 'mwt-addons-for-elementor'),
                'label_block' => true,
                'dynamic' => [
                    'active' => true,
                ],
            ]
        );

        /* Start repeater */

        $repeater = new \Elementor\Repeater();

        $repeater->add_control(
            'image',
            [
                'label' => esc_html__('Choose Image', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::MEDIA,
                'dynamic' => [
                    'active' => true,
                ],
                'default' => [
                    'url' => \Elementor\Utils::get_placeholder_image_src(),
                ],
            ]
        );

        $repeater->add_control(
            'name',
            [
                'label' => esc_html__('Name', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'placeholder' => esc_html__('Name', 'mwt-addons-for-elementor'),
                'default' => esc_html__('Name', 'mwt-addons-for-elementor'),
                'label_block' => true,
                'dynamic' => [
                    'active' => true,
                ],
            ]
        );

        $repeater->add_control(
            'position',
            [
                'label' => esc_html__('Position', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'placeholder' => esc_html__('Position', 'mwt-addons-for-elementor'),
                'default' => esc_html__('Position', 'mwt-addons-for-elementor'),
                'label_block' => true,
                'dynamic' => [
                    'active' => true,
                ],
            ]
        );

        $repeater->add_control(
            'member_social_enable',
            [
                'label'     => __('Enable Social Icons', 'mwt-addons-for-elementor'),
                'type'      => \Elementor\Controls_Manager::SWITCHER,
                'default'   => 'yes',
                'separator' => 'before',
            ]
        );

        $repeater->add_control(
            'member_facebook',
            [
                'label'       => __('Facebook', 'mwt-addons-for-elementor'),
                'type'        => \Elementor\Controls_Manager::TEXT,
                'dynamic'     => ['active' => true],
                'default'     => '#',
                'label_block' => true,
                'condition'   => [
                    'member_social_enable' => 'yes',
                ],
            ]
        );

        $repeater->add_control(
            'member_twitter',
            [
                'label'       => __('Twitter', 'mwt-addons-for-elementor'),
                'type'        => \Elementor\Controls_Manager::TEXT,
                'dynamic'     => ['active' => true],
                'default'     => '#',
                'label_block' => true,
                'condition'   => [
                    'member_social_enable' => 'yes',
                ],
            ]
        );

        $repeater->add_control(
            'member_linkedin',
            [
                'label'       => __('LinkedIn', 'mwt-addons-for-elementor'),
                'type'        => \Elementor\Controls_Manager::TEXT,
                'dynamic'     => ['active' => true],
                'label_block' => true,
                'condition'   => [
                    'member_social_enable' => 'yes',
                ],
            ]
        );

        $repeater->add_control(
            'member_youtube',
            [
                'label'       => __('YouTube', 'mwt-addons-for-elementor'),
                'type'        => \Elementor\Controls_Manager::TEXT,
                'dynamic'     => ['active' => true],
                'label_block' => true,
                'condition'   => [
                    'member_social_enable' => 'yes',
                ],
            ]
        );

        $repeater->add_control(
            'member_instagram',
            [
                'label'       => __('Instagram', 'mwt-addons-for-elementor'),
                'type'        => \Elementor\Controls_Manager::TEXT,
                'dynamic'     => ['active' => true],
                'default'     => '#',
                'label_block' => true,
                'condition'   => [
                    'member_social_enable' => 'yes',
                ],
            ]
        );

        $repeater->add_control(
            'member_skype',
            [
                'label'       => __('Skype', 'mwt-addons-for-elementor'),
                'type'        => \Elementor\Controls_Manager::TEXT,
                'dynamic'     => ['active' => true],
                'label_block' => true,
                'condition'   => [
                    'member_social_enable' => 'yes',
                ],
            ]
        );

        $repeater->add_control(
            'member_pinterest',
            [
                'label'       => __('Pinterest', 'mwt-addons-for-elementor'),
                'type'        => \Elementor\Controls_Manager::TEXT,
                'dynamic'     => ['active' => true],
                'label_block' => true,
                'condition'   => [
                    'member_social_enable' => 'yes',
                ],
            ]
        );

        $repeater->add_control(
            'member_dribbble',
            [
                'label'       => __('Dribbble', 'mwt-addons-for-elementor'),
                'type'        => \Elementor\Controls_Manager::TEXT,
                'dynamic'     => ['active' => true],
                'default'     => '#',
                'label_block' => true,
                'condition'   => [
                    'member_social_enable' => 'yes',
                ],
            ]
        );

        $repeater->add_control(
            'member_behance',
            [
                'label'       => __('Behance', 'mwt-addons-for-elementor'),
                'type'        => \Elementor\Controls_Manager::TEXT,
                'dynamic'     => ['active' => true],
                'label_block' => true,
                'condition'   => [
                    'member_social_enable' => 'yes',
                ],
            ]
        );

        $repeater->add_control(
            'member_whatsapp',
            [
                'label'       => __('WhatsApp', 'mwt-addons-for-elementor'),
                'type'        => \Elementor\Controls_Manager::TEXT,
                'dynamic'     => ['active' => true],
                'label_block' => true,
                'condition'   => [
                    'member_social_enable' => 'yes',
                ],
            ]
        );

        $repeater->add_control(
            'member_telegram',
            [
                'label'       => __('Telegram', 'mwt-addons-for-elementor'),
                'type'        => \Elementor\Controls_Manager::TEXT,
                'dynamic'     => ['active' => true],
                'label_block' => true,
                'condition'   => [
                    'member_social_enable' => 'yes',
                ],
            ]
        );

        /* End repeater */

        $this->add_control(
            'members-list',
            [
                'label' => esc_html__('Members', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),           /* Use our repeater */
                'default' => [
                    [
                        'image' => '',
                        'name' => esc_html__('Name #1', 'mwt-addons-for-elementor'),
                        'position' => esc_html__('Position #1', 'mwt-addons-for-elementor'),
                        'member_social_enable' => 'yes',
                        'member_facebook' => 'https://www.facebook.com/',
                        'member_twitter' => 'https://twitter.com/',
                        'member_linkedin' => '',
                        'member_google'    => '',
                        'member_youtube'   => '',
                        'member_instagram' => 'https://www.instagram.com/',
                        'member_skype'     => '',
                        'member_pinterest' => '',
                        'member_dribbble'  => '',
                        'member_behance'   => '',
                        'member_whatsapp'  => '',
                        'member_telegram'  => '',
                    ],
                    [
                        'image' => '',
                        'name' => esc_html__('Name #2', 'mwt-addons-for-elementor'),
                        'position' => esc_html__('Position #2', 'mwt-addons-for-elementor'),
                        'member_social_enable' => 'yes',
                        'member_facebook' => 'https://www.facebook.com/',
                        'member_twitter' => 'https://twitter.com/',
                        'member_linkedin' => '',
                        'member_google'    => '',
                        'member_youtube'   => '',
                        'member_instagram' => 'https://www.instagram.com/',
                        'member_skype'     => '',
                        'member_pinterest' => '',
                        'member_dribbble'  => '',
                        'member_behance'   => '',
                        'member_whatsapp'  => '',
                        'member_telegram'  => '',
                    ],
                    [
                        'image' => '',
                        'name' => esc_html__('Name #3', 'mwt-addons-for-elementor'),
                        'position' => esc_html__('Position #3', 'mwt-addons-for-elementor'),
                        'member_social_enable' => 'yes',
                        'member_facebook' => 'https://www.facebook.com/',
                        'member_twitter' => 'https://twitter.com/',
                        'member_linkedin' => '',
                        'member_google'    => '',
                        'member_youtube'   => '',
                        'member_instagram' => 'https://www.instagram.com/',
                        'member_skype'     => '',
                        'member_pinterest' => '',
                        'member_dribbble'  => '',
                        'member_behance'   => '',
                        'member_whatsapp'  => '',
                        'member_telegram'  => '',
                    ],
                ],
                'title_field' => '{{{ name }}}',
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Image_Size::get_type(),
            [
                'name' => 'thumbnail', // Usage: `{name}_size` and `{name}_custom_dimension`, in this case `thumbnail_size` and `thumbnail_custom_dimension`.
                'separator' => 'none',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'style_title_section',
            [
                'label' => esc_html__('Title', 'mwt-addons-for-elementor'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'title_color',
            [
                'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-elementor-team-slider-title' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
                'name' => 'title_typography',
                'selector' =>
                '{{WRAPPER}} .mwt-elementor-team-slider-title',
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Text_Shadow::get_type(),
            [
                'label' => esc_html__('Text Shadow', 'mwt-addons-for-elementor'),
                'name' => 'title_shadow',
                'selector' => '{{WRAPPER}} .mwt-elementor-team-slider-title',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'style_name_section',
            [
                'label' => esc_html__('Name/Position', 'mwt-addons-for-elementor'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'heading_style_name',
            [
                'label' => esc_html__('Name', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'name_color',
            [
                'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-elementor-team-slider-name' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'name_color_active',
            [
                'label' => esc_html__('Color Active', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .flex-active .mwt-elementor-team-slider-name' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
                'name' => 'name_typography',
                'selector' =>
                '{{WRAPPER}} .mwt-elementor-team-slider-name',
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Text_Shadow::get_type(),
            [
                'label' => esc_html__('Text Shadow', 'mwt-addons-for-elementor'),
                'name' => 'name_shadow',
                'selector' => '{{WRAPPER}} .mwt-elementor-team-slider-name',
            ]
        );

        $this->add_control(
            'heading_style_position',
            [
                'label' => esc_html__('Position', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'position_color',
            [
                'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-elementor-team-slider-position' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'position_color_active',
            [
                'label' => esc_html__('Color Active', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .flex-active .mwt-elementor-team-slider-position' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
                'name' => 'position_typography',
                'selector' =>
                '{{WRAPPER}} .mwt-elementor-team-slider-position',

            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Text_Shadow::get_type(),
            [
                'label' => esc_html__('Text Shadow', 'mwt-addons-for-elementor'),
                'name' => 'position_shadow',
                'selector' => '{{WRAPPER}} .mwt-elementor-team-slider-position',
            ]
        );

        $this->add_responsive_control(
            'nav_margin',
            [
                'label'      => __('Margin', 'mwt-addons-for-elementor'),
                'type'       => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'separator' => 'before',
                'selectors'  => [
                    '{{WRAPPER}} .flex-control-nav-1 .menu__item' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        //soc icons style
        $this->start_controls_section(
            'social_icon_style',
            [
                'label'     => __('Social Icons', 'mwt-addons-for-elementor'),
                'tab'       => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'social_size',
            [
                'label'       => __('Size', 'mwt-addons-for-elementor'),
                'type'        => \Elementor\Controls_Manager::SLIDER,
                'size_units'  => ['px', 'em', '%'],
                'label_block' => true,
                'selectors'   => [
                    '{{WRAPPER}} .mwt-team-slider-container .members-list-item i' => 'font-size: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'social_default_colors',
            [
                'label'        => __('Brands Default Colors', 'mwt-addons-for-elementor'),
                'type'         => \Elementor\Controls_Manager::SWITCHER,
                'prefix_class' => 'members-defaults-',
            ]
        );

        $this->add_control(
            'social_color',
            [
                'label'     => __('Color', 'mwt-addons-for-elementor'),
                'type'      => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-team-slider-container .members-list-item i'  => 'color: {{VALUE}};',
                ],
                'condition' => [
                    'social_default_colors!' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'social_default_hover_colors',
            [
                'label'        => __('Brands Default Hover Colors', 'mwt-addons-for-elementor'),
                'type'         => \Elementor\Controls_Manager::SWITCHER,
                'prefix_class' => 'members-defaults-hover-',
            ]
        );

        $this->add_control(
            'social_hover_color',
            [
                'label'     => __('Hover Color', 'mwt-addons-for-elementor'),
                'type'      => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-team-slider-container .members-list-item:hover i'  => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'social_default_hover_colors!'   => 'yes',
                ],
            ]
        );

        $this->add_control(
            'social_default_background',
            [
                'label'        => __('Brands Default Background Colors', 'mwt-addons-for-elementor'),
                'type'         => \Elementor\Controls_Manager::SWITCHER,
                'prefix_class' => 'members-defaults-background-',
            ]
        );

        $this->add_control(
            'social_background',
            [
                'label'     => __('Background Color', 'mwt-addons-for-elementor'),
                'type'      => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-team-slider-container .members-list-item a'  => 'background-color: {{VALUE}}',
                ],
                'condition' => [
                    'social_default_background!' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'social_default_hover_background',
            [
                'label'        => __('Brands Default Hover Background Colors', 'mwt-addons-for-elementor'),
                'type'         => \Elementor\Controls_Manager::SWITCHER,
                'prefix_class' => 'members-defaults-hover-background-',
            ]
        );

        $this->add_control(
            'social_hover_background',
            [
                'label'     => __('Hover Background Color', 'mwt-addons-for-elementor'),
                'type'      => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-team-slider-container li.members-list-item:hover a'  => 'background-color: {{VALUE}}',
                ],
                'condition' => [
                    'social_default_hover_background!'   => 'yes',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name'     => 'social_border',
                'selector' => '{{WRAPPER}} .mwt-team-slider-container .members-list-item a',
            ]
        );

        $this->add_responsive_control(
            'social_radius',
            [
                'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
                'type'       => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .mwt-team-slider-container .members-list-item a' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
                ],
            ]
        );

        $this->add_responsive_control(
            'social_margin',
            [
                'label'      => __('Margin', 'mwt-addons-for-elementor'),
                'type'       => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .mwt-team-slider-container .members-list-item a' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'social_padding',
            [
                'label'      => __('Padding', 'mwt-addons-for-elementor'),
                'type'       => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .mwt-team-slider-container .members-list-item a' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();
    }

    /**
     * Render list widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display();

        if (empty($settings['members-list'])) {
            return;
        }
?>
        <div class="flexslider-container mwt-team-slider-container">
            <?php
            $title = $settings['title']; ?>
            <h3 class="slider-title mwt-elementor-team-slider-title"><?php echo esc_html($title); ?></h3>
            <div class="flexslider team-slider" data-slideshow="1" data-slideshowspeed="3000" data-animation="fade">
                <ul class="slides">
                    <?php
                    foreach ($settings['members-list'] as $index => $item) { ?>
                        <li>
                            <?php
                            $image_url = \Elementor\Group_Control_Image_Size::get_attachment_image_src($item['image']['id'], 'thumbnail', $settings);

                            if (!$image_url && isset($item['image']['url'])) {
                                $image_url = $item['image']['url'];
                            }
                            $image_html = '<img class="swiper-slide-image" src="' . esc_attr($image_url) . '" alt="' . esc_attr(\Elementor\Control_Media::get_image_alt($item['image'])) . '" />';
                            echo wp_kses_post($image_html);
                            ?>
                            <?php
                            $social_sites = array(
                                'facebook'  => 'fa fa-facebook-f',
                                'twitter'   => 'fa fa-twitter',
                                'linkedin'  => 'fa fa-linkedin',
                                'youtube'   => 'fa fa-youtube',
                                'instagram' => 'fa fa-instagram',
                                'skype'     => 'fa fa-skype',
                                'pinterest' => 'fa fa-pinterest',
                                'dribbble'  => 'fa fa-dribbble',
                                'behance'   => 'fa fa-behance',
                                'whatsapp'  => 'fa fa-whatsapp',
                                'telegram'  => 'fa fa-telegram',
                            );
                            ?>
                            <ul class="member-social-list">
                                <?php
                                foreach ($social_sites as $site => $icon) {
                                    $value = $item['member_' . $site];

                                    if (!empty($value)) {
                                        $icon_class = sprintf('elementor-icon members-list-item members-%s', $site); ?>
                                        <li class="<?php echo esc_attr($icon_class); ?>">
                                            <a href="<?php echo esc_url($value); ?>" target="_blank">
                                                <i class="<?php echo esc_attr($icon); ?>"></i>
                                            </a>
                                        </li>
                                <?php
                                    }
                                } ?>
                            </ul>
                        </li>
                    <?php } ?>
                </ul>
            </div>
            <div class="flexslider-controls">
                <ul class="flex-control-nav-1">
                    <?php
                    foreach ($settings['members-list'] as $index => $item) { ?>
                        <li class="menu__item">
                            <?php
                            $name = $settings['members-list'][$index]['name'];
                            $position = $settings['members-list'][$index]['position'];
                            ?>
                            <span class="mwt-elementor-team-slider-name">
                                <?php echo esc_html($name); ?>
                            </span>
                            <span class="mwt-elementor-team-slider-position">
                                <?php echo esc_html($position); ?>
                            </span>
                        </li>
                    <?php } ?>
                </ul>
            </div>
        </div>
<?php }
}
