<?php
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

/**
 * Elementor MWT Team Carousel Widget.
 *
 * Elementor widget that inserts an embbedable content into the page, from any given URL.
 *
 * @since 1.0.0
 */

class Elementor_MWT_Team_Box extends \Elementor\Widget_Base

{

	/**
	 * Retrieve Widget Dependent CSS.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array CSS style handles.
	 */
	public function get_style_depends()
	{
		return ['team-box-style'];
	}

	/**
	 * Get widget name.
	 *
	 * Retrieve list widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return string Widget name.
	 */
	public function get_name()
	{
		return 'team-box';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve list widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return string Widget title.
	 */
	public function get_title()
	{
		return esc_html__('MWT Team Box', 'mwt-addons-for-elementor');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the list widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array Widget categories.
	 */
	public function get_categories()
	{
		return ['mwt'];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve image carousel widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon()
	{
		return 'eicon-person';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @since 2.1.0
	 * @access public
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords()
	{
		return ['mwt', 'image', 'photo', 'visual', 'team', 'box'];
	}

	/**
	 * Register list widget controls.
	 *
	 * Add input fields to allow the user to customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls()
	{

		$this->start_controls_section(
			'content_section',
			[
				'label' => esc_html__('Team Box', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'image',
			[
				'label' => esc_html__('Choose Image', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::MEDIA,
				'dynamic' => [
					'active' => true,
				],
				'default' => [
					'url' => \Elementor\Utils::get_placeholder_image_src(),
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Image_Size::get_type(),
			[
				'name' => 'thumbnail',
				'separator' => 'none',
			]
		);

		$this->add_control(
			'always_hovered',
			[
				'label' => esc_html__('Always Hovered', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'return_value' => 'yes',
				'default' => 'no',
			]
		);

		$this->add_control(
			'item_name',
			[
				'label' => esc_html__('Name', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::TEXT,
				'placeholder' => esc_html__('Name', 'mwt-addons-for-elementor'),
				'default' => esc_html__('Name', 'mwt-addons-for-elementor'),
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
			]
		);

		$this->add_control(
			'item_name_link',
			[
				'label'       => __('Title Link', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::URL,
				'placeholder' => 'https://example.com/',
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
			]
		);

		$this->add_control(
			'item_position',
			[
				'label'       => __('Position', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::TEXT,
				'placeholder' => esc_html__('Position', 'mwt-addons-for-elementor'),
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
			]
		);

		$this->add_control(
			'member_social_enable',
			[
				'label'     => __('Enable Social Icons', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::SWITCHER,
				'default'   => 'yes',
				'separator' => 'before',
			]
		);

		$this->add_control(
			'member_facebook',
			[
				'label'       => __('Facebook', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::TEXT,
				'dynamic'     => ['active' => true],
				'default'     => '#',
				'label_block' => true,
				'condition'   => [
					'member_social_enable' => 'yes',
				],
			]
		);

		$this->add_control(
			'member_twitter',
			[
				'label'       => __('Twitter', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::TEXT,
				'dynamic'     => ['active' => true],
				'default'     => '#',
				'label_block' => true,
				'condition'   => [
					'member_social_enable' => 'yes',
				],
			]
		);

		$this->add_control(
			'member_linkedin',
			[
				'label'       => __('LinkedIn', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::TEXT,
				'dynamic'     => ['active' => true],
				'label_block' => true,
				'condition'   => [
					'member_social_enable' => 'yes',
				],
			]
		);

		$this->add_control(
			'member_youtube',
			[
				'label'       => __('YouTube', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::TEXT,
				'dynamic'     => ['active' => true],
				'label_block' => true,
				'condition'   => [
					'member_social_enable' => 'yes',
				],
			]
		);

		$this->add_control(
			'member_instagram',
			[
				'label'       => __('Instagram', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::TEXT,
				'dynamic'     => ['active' => true],
				'default'     => '#',
				'label_block' => true,
				'condition'   => [
					'member_social_enable' => 'yes',
				],
			]
		);

		$this->add_control(
			'member_skype',
			[
				'label'       => __('Skype', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::TEXT,
				'dynamic'     => ['active' => true],
				'label_block' => true,
				'condition'   => [
					'member_social_enable' => 'yes',
				],
			]
		);

		$this->add_control(
			'member_pinterest',
			[
				'label'       => __('Pinterest', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::TEXT,
				'dynamic'     => ['active' => true],
				'label_block' => true,
				'condition'   => [
					'member_social_enable' => 'yes',
				],
			]
		);

		$this->add_control(
			'member_dribbble',
			[
				'label'       => __('Dribbble', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::TEXT,
				'dynamic'     => ['active' => true],
				'default'     => '#',
				'label_block' => true,
				'condition'   => [
					'member_social_enable' => 'yes',
				],
			]
		);

		$this->add_control(
			'member_behance',
			[
				'label'       => __('Behance', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::TEXT,
				'dynamic'     => ['active' => true],
				'label_block' => true,
				'condition'   => [
					'member_social_enable' => 'yes',
				],
			]
		);

		$this->add_control(
			'member_whatsapp',
			[
				'label'       => __('WhatsApp', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::TEXT,
				'dynamic'     => ['active' => true],
				'label_block' => true,
				'condition'   => [
					'member_social_enable' => 'yes',
				],
			]
		);

		$this->add_control(
			'member_telegram',
			[
				'label'       => __('Telegram', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::TEXT,
				'dynamic'     => ['active' => true],
				'label_block' => true,
				'condition'   => [
					'member_social_enable' => 'yes',
				],
			]
		);

		$this->end_controls_section();

		// style
		$this->start_controls_section(
			'style_item_section',
			[
				'label' => esc_html__('Items', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'items_border',
				'selector' => '{{WRAPPER}} .team-box-container .mwt-elementor-team-box-wrap',
			]
		);

		$this->add_responsive_control(
			'items_radius',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .team-box-container .mwt-elementor-team-box-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
				],
			]
		);

		$this->start_controls_tabs(
			'style_tabs'
		);

		$this->start_controls_tab(
			'style_normal_tab',
			[
				'label' => esc_html__('Normal', 'textdomain'),
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'member_box_shadow',
				'selector' => '{{WRAPPER}} .team-box-container .mwt-elementor-team-box-wrap',
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'style_hover_tab',
			[
				'label' => esc_html__('Hover', 'textdomain'),
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'member_box_shadow_hover',
				'selector' => '{{WRAPPER}} .team-box-container .mwt-elementor-team-box-wrap:hover',
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->end_controls_section();

		$this->start_controls_section(
			'style_image',
			[
				'label' => esc_html__('Image', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'image_hover_overlay_color',
			[
				'label' => esc_html__('Hover Overlay Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .team-box-container .mwt-elementor-team-box-wrap::before' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_name_section',
			[
				'label' => esc_html__('Name', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'name_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-team-box-name' => 'color: {{VALUE}};',
					'{{WRAPPER}} a.mwt-elementor-team-box-name' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'name_color_active',
			[
				'label' => esc_html__('Color Hover', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} a.mwt-elementor-team-box-name:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'name_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-elementor-team-box-name, {{WRAPPER}} .mwt-elementor-team-box-name > a',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Text_Shadow::get_type(),
			[
				'label' => esc_html__('Text Shadow', 'mwt-addons-for-elementor'),
				'name' => 'name_shadow',
				'selector' => '{{WRAPPER}} .mwt-elementor-team-box-name',
			]
		);

		$this->add_responsive_control(
			'name_left_position',
			[
				'label'      => __('Left Position', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100,
						'step' => 1,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 50,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-team-box-name'  => 'left: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->add_responsive_control(
			'name_bottom_position',
			[
				'label'      => __('Bottom Position', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100,
						'step' => 1,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 50,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-team-box-name'  => 'bottom: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_position_section',
			[
				'label' => esc_html__('Position', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'position_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-team-box-position' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'position_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-elementor-team-box-position',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Text_Shadow::get_type(),
			[
				'label' => esc_html__('Text Shadow', 'mwt-addons-for-elementor'),
				'name' => 'position_shadow',
				'selector' => '{{WRAPPER}} .mwt-elementor-team-box-position',
			]
		);

		$this->add_responsive_control(
			'position_right',
			[
				'label'      => __('Right Position', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100,
						'step' => 1,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 50,
				],
				'selectors'  => [
					'{{WRAPPER}} .team-box-container:hover .mwt-elementor-team-box-position'  => 'right: {{SIZE}}{{UNIT}}',
					'{{WRAPPER}} .team-box-container.always-hovered .mwt-elementor-team-box-position'  => 'right: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->add_responsive_control(
			'position_top',
			[
				'label'      => __('Top Position', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100,
						'step' => 1,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 50,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-team-box-position'  => 'top: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->end_controls_section();

		//soc icons style
		$this->start_controls_section(
			'social_icon_style',
			[
				'label'     => __('Social Icons', 'mwt-addons-for-elementor'),
				'tab'       => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'social_size',
			[
				'label'       => __('Size', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::SLIDER,
				'size_units'  => ['px', 'em', '%'],
				'label_block' => true,
				'selectors'   => [
					'{{WRAPPER}} .team-box-container .members-list-item i' => 'font-size: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->start_controls_tabs(
			'style_social_tabs'
		);

		$this->start_controls_tab(
			'style_social_normal_tab',
			[
				'label' => esc_html__('Normal', 'textdomain'),
			]
		);

		$this->add_control(
			'social_default_colors',
			[
				'label'        => __('Brands Default Colors', 'mwt-addons-for-elementor'),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'prefix_class' => 'members-defaults-',
			]
		);

		$this->add_control(
			'social_color',
			[
				'label'     => __('Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .team-box-container .members-list-item i'  => 'color: {{VALUE}};',
				],
				'condition' => [
					'social_default_colors!'   => 'yes',
				],
			]
		);

		$this->add_control(
			'social_default_background',
			[
				'label'        => __('Brands Default Background Colors', 'mwt-addons-for-elementor'),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'prefix_class' => 'members-defaults-background-',
			]
		);

		$this->add_control(
			'social_background',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .team-box-container .members-list-item a'  => 'background-color: {{VALUE}}',
				],
				'condition' => [
					'social_default_background!'   => 'yes',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'style_social_hover_tab',
			[
				'label' => esc_html__('Hover', 'textdomain'),
			]
		);

		$this->add_control(
			'social_default_hover_colors',
			[
				'label'        => __('Brands Default Hover Colors', 'mwt-addons-for-elementor'),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'prefix_class' => 'members-defaults-hover-',
			]
		);

		$this->add_control(
			'social_hover_color',
			[
				'label'     => __('Hover Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .team-box-container .members-list-item:hover i'  => 'color: {{VALUE}}',
				],
				'condition' => [
					'social_default_hover_colors!'   => 'yes',
				],
			]
		);

		$this->add_control(
			'social_default_hover_background',
			[
				'label'        => __('Brands Default Hover Background Colors', 'mwt-addons-for-elementor'),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'prefix_class' => 'members-defaults-hover-background-',
			]
		);

		$this->add_control(
			'social_hover_background',
			[
				'label'     => __('Hover Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .team-box-container li.members-list-item:hover a'  => 'background-color: {{VALUE}}',
				],
				'condition' => [
					'social_default_hover_background!'   => 'yes',
				],
			]
		);

		$this->add_control(
			'social_hover_border_color',
			[
				'label'     => __('Border Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}}  .team-box-container li.members-list-item:hover a'  => 'border-color: {{VALUE}}',
				],
				'condition' => [
					'social_default_hover_background!'   => 'yes',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'social_border',
				'selector' => '{{WRAPPER}} .team-box-container .members-list-item a',
			]
		);

		$this->add_responsive_control(
			'social_radius',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .team-box-container .members-list-item a' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
				],
			]
		);

		$this->add_responsive_control(
			'social_padding',
			[
				'label'      => __('Padding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .team-box-container .members-list-item a' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'social_margin',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .team-box-container .members-list-item a' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'social_right_position',
			[
				'label'      => __('Right Position', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100,
						'step' => 1,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 50,
				],
				'selectors'  => [
					'{{WRAPPER}} .team-box-container:hover .member-social-list'  => 'right: {{SIZE}}{{UNIT}}',
					'{{WRAPPER}} .team-box-container.always-hovered .member-social-list'  => 'right: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->add_responsive_control(
			'social_bottom_position',
			[
				'label'      => __('Bottom Position', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100,
						'step' => 1,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 50,
				],
				'selectors'  => [
					'{{WRAPPER}} .team-box-container .member-social-list'  => 'bottom: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render list widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render()
	{
		$settings = $this->get_settings_for_display();

		$always_hovered = ('yes' === ($settings['always_hovered'])) ? 'always-hovered' : ''; ?>

		<div class="team-box-container <?php echo esc_attr($always_hovered) ?>">
			<div class="mwt-elementor-team-box-wrap">
				<?php
				$image_url = \Elementor\Group_Control_Image_Size::get_attachment_image_src($settings['image']['id'], 'thumbnail', $settings);
				if (!$image_url && isset($settings['image']['url'])) {
					$image_url = $settings['image']['url'];
				}
				if (!empty($settings['item_name'])) {
					$image_alt = $settings['item_name'];
				} else {
					$image_alt = 'This is a title';
				}
				$image_html = '<img class="team-image" src="' . esc_attr($image_url) . '" alt="' . esc_attr($image_alt) . '" />';
				echo wp_kses_post($image_html);
				?>
				<?php
				$name_link = $settings['item_name_link']['url'];
				if (!empty($name_link)) { ?>
					<a href="<?php echo esc_attr($name_link) ?>" class="mwt-elementor-team-box-media-link"></a>
				<?php } ?>
				<?php
				$name = $settings['item_name'];
				$position = $settings['item_position'];
				?>
				<?php
				if (!empty($name_link)) { ?>
					<a href="<?php echo esc_attr($name_link) ?>" class="mwt-elementor-team-box-name">
						<?php echo $name; ?>
					</a>
				<?php } else { ?>
					<div class="mwt-elementor-team-box-name">
						<?php echo $name; ?>
					</div>
				<?php } ?>
				<?php if (!empty($position)) { ?>
					<div class="mwt-elementor-team-box-position">
						<?php echo $position; ?>
					</div>
				<?php } ?>
				<?php
				if ('yes' === $settings['member_social_enable']) {
					$social_sites = array(
						'facebook'  => 'fa fa-facebook-f',
						'twitter'   => 'ico-Twitter',
						'linkedin'  => 'fa fa-linkedin',
						'youtube'   => 'ico-Youtube',
						'instagram' => 'fa fa-instagram',
						'skype'     => 'fa fa-skype',
						'pinterest' => 'fa fa-pinterest',
						'dribbble'  => 'fa fa-dribbble',
						'behance'   => 'fa fa-behance',
						'whatsapp'  => 'fa fa-whatsapp',
						'telegram'  => 'fa fa-telegram',
					);
				?>
					<ul class="member-social-list">
						<?php
						foreach ($social_sites as $site => $icon) {
							$value = $settings['member_' . $site];

							if (!empty($value)) {
								$icon_class = sprintf('elementor-icon members-list-item members-%s', $site); ?>
								<li class="<?php echo esc_attr($icon_class); ?>">
									<a href="<?php echo esc_url($value); ?>" target="_blank">
										<i class="<?php echo esc_attr($icon); ?>"></i>
									</a>
								</li>
						<?php
							}
						} ?>
					</ul>
				<?php } ?>
			</div>
		</div>
<?php }
}
