<?php
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

/**
 * Elementor MWT Pricing Box Widget.
 *
 * Elementor widget that inserts an embbedable content into the page, from any given URL.
 *
 * @since 1.0.0
 */

class Elementor_MWT_Pricing_Box extends \Elementor\Widget_Base

{

	/**
	 * Retrieve Widget Dependent CSS.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array CSS style handles.
	 */
	public function get_style_depends()
	{
		return ['pricing-box-style'];
	}

	/**
	 * Get widget name.
	 *
	 * Retrieve list widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return string Widget name.
	 */
	public function get_name()
	{
		return 'pricing-box';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve list widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return string Widget title.
	 */
	public function get_title()
	{
		return esc_html__('MWT Pricing Box', 'mwt-addons-for-elementor');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the list widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array Widget categories.
	 */
	public function get_categories()
	{
		return ['mwt'];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve image carousel widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon()
	{
		return 'eicon-price-table';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @since 2.1.0
	 * @access public
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords()
	{
		return ['mwt', 'image', 'icon', 'pricing', 'box', 'table', 'visual'];
	}

	/**
	 * Register list widget controls.
	 *
	 * Add input fields to allow the user to customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls()
	{

		$this->start_controls_section(
			'icon_section',
			[
				'label' => esc_html__('Icon', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'pricing_box_type',
			[
				'label' => esc_html__('Type', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'icon',
				'options' => [
					'icon' => esc_html__('Icon', 'mwt-addons-for-elementor'),
					'image' => esc_html__('Image', 'mwt-addons-for-elementor'),
				],
			]
		);

		$this->add_control(
			'pricing_box_icon',
			[
				'label' => esc_html__('Icon', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::ICONS,
				'default' => [
					'value' => 'fas fa-circle',
					'library' => 'fa-solid',
				],
				'recommended' => [
					'fa-solid' => [
						'circle',
						'dot-circle',
						'square-full',
					],
					'fa-regular' => [
						'circle',
						'dot-circle',
						'square-full',
					],
				],
				'condition' => [
					'pricing_box_type' => 'icon'
				]
			]
		);

		$this->add_control(
			'pricing_box_image',
			[
				'label' => esc_html__('Choose Image', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::MEDIA,
				'dynamic' => [
					'active' => true,
				],
				'default' => [
					'url' => \Elementor\Utils::get_placeholder_image_src(),
				],
				'condition' => [
					'pricing_box_type' => 'image'
				],
				'ai' => [
					'active' => false,
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Image_Size::get_type(),
			[
				'name' => 'thumbnail',
				'separator' => 'none',
				'condition' => [
					'pricing_box_type' => 'image'
				]
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'pricing_title_section',
			[
				'label' => esc_html__('Title', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'pricing_title',
			[
				'label' => esc_html__('Title', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::TEXT,
				'placeholder' => esc_html__('Title', 'mwt-addons-for-elementor'),
				'default' => esc_html__('Title', 'mwt-addons-for-elementor'),
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
				'ai' => [
					'active' => false,
				],
			]
		);

		$this->add_control(
			'pricing_title_tag',
			[
				'label' => esc_html__('HTML Tag', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'h2',
				'options' => [
					'h1' => 'H1',
					'h2' => 'H2',
					'h3' => 'H3',
					'h4' => 'H4',
					'h5' => 'H5',
					'h6' => 'H6',
					'div' => 'div',
					'span' => 'span',
					'p' => 'p',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'price_section',
			[
				'label' => esc_html__('Price', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'price_currency',
			[
				'label' => esc_html__('Currency', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::TEXT,
				'placeholder' => esc_html__('$', 'mwt-addons-for-elementor'),
				'default' => esc_html__('$', 'mwt-addons-for-elementor'),
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
				'ai' => [
					'active' => false,
				],
			]
		);

		$this->add_control(
			'price',
			[
				'label' => esc_html__('Price', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::TEXT,
				'placeholder' => esc_html__('55', 'mwt-addons-for-elementor'),
				'default' => esc_html__('55', 'mwt-addons-for-elementor'),
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
				'ai' => [
					'active' => false,
				],
			]
		);

		$this->add_control(
			'price_divider',
			[
				'label' => esc_html__('Divider', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::TEXT,
				'placeholder' => esc_html__('/', 'mwt-addons-for-elementor'),
				'default' => esc_html__('/', 'mwt-addons-for-elementor'),
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
				'ai' => [
					'active' => false,
				],
			]
		);

		$this->add_control(
			'price_duration',
			[
				'label' => esc_html__('Duration', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::TEXT,
				'placeholder' => esc_html__('m', 'mwt-addons-for-elementor'),
				'default' => esc_html__('m', 'mwt-addons-for-elementor'),
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
				'ai' => [
					'active' => false,
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'pricing_list_section',
			[
				'label' => esc_html__('Feature List', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		/* Start repeater */
		$repeater = new \Elementor\Repeater();

		$repeater->add_control(
			'list_text',
			[
				'label' => esc_html__('Text', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::TEXT,
				'placeholder' => esc_html__('List Item', 'mwt-addons-for-elementor'),
				'default' => esc_html__('List Item', 'mwt-addons-for-elementor'),
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
				'ai' => [
					'active' => false,
				],
			]
		);

		$repeater->add_control(
			'list_icon_type',
			[
				'label' => esc_html__('Icon Type', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'icon',
				'options' => [
					'none' => esc_html__('None', 'mwt-addons-for-elementor'),
					'icon' => esc_html__('Icon', 'mwt-addons-for-elementor'),
				],
			]
		);

		$repeater->add_control(
			'list_icon',
			[
				'label' => esc_html__('Icon', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::ICONS,
				'default' => [
					'value' => 'fas fa-circle',
					'library' => 'fa-solid',
				],
				'recommended' => [
					'fa-solid' => [
						'circle',
						'dot-circle',
						'square-full',
					],
					'fa-regular' => [
						'circle',
						'dot-circle',
						'square-full',
					],
				],
				'condition' => [
					'list_icon_type' => 'icon'
				]
			]
		);

		/* End repeater */
		$this->add_control(
			'feature_list_pricing',
			[
				'label' => esc_html__('Features', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::REPEATER,
				'fields' => $repeater->get_controls(),
				'default' => [
					[
						'list_text' => esc_html__('List Item #1', 'mwt-addons-for-elementor'),
					],
					[
						'list_text' => esc_html__('List Item #2', 'mwt-addons-for-elementor'),
					],
				],
				'title_field' => '{{{ list_text }}}',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'pricing_button_section',
			[
				'label' => esc_html__('Button', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'pricing_button',
			[
				'label' => esc_html__('Text', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::TEXT,
				'placeholder' => esc_html__('Type your text here', 'mwt-addons-for-elementor'),
				'default' => 'Purchase',
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
				'ai' => [
					'active' => false,
				],
			]
		);

		$this->add_control(
			'pricing_button_link',
			[
				'label'       => __('Link', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::URL,
				'placeholder' => 'https://example.com/',
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'pricing_ribbon_section',
			[
				'label' => esc_html__('Ribbon', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'pricing_ribbon',
			[
				'label' => esc_html__('Text', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::TEXT,
				'placeholder' => esc_html__('Type your text here', 'mwt-addons-for-elementor'),
				'default' => '',
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
				'ai' => [
					'active' => false,
				],
			]
		);

		$this->add_control(
			'pricing_ribbon_position',
			array(
				'label'   => __('Position', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::CHOOSE,
				'toggle'  => false,
				'options' => array(
					'left'  => array(
						'title' => __('Left', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-h-align-left',
					),
					'right' => array(
						'title' => __('Right', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-h-align-right',
					),
				),
				'default' => 'right',
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'pricing_options_section',
			[
				'label' => esc_html__('Display Options', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'show_icon',
			[
				'label' => esc_html__('Show Icon', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => esc_html__('Show', 'mwt-addons-for-elementor'),
				'label_off' => esc_html__('Hide', 'mwt-addons-for-elementor'),
				'return_value' => 'yes',
				'default' => 'yes',
			]
		);

		$this->add_control(
			'show_title',
			[
				'label' => esc_html__('Show Title', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => esc_html__('Show', 'mwt-addons-for-elementor'),
				'label_off' => esc_html__('Hide', 'mwt-addons-for-elementor'),
				'return_value' => 'yes',
				'default' => 'yes',
			]
		);

		$this->add_control(
			'show_price',
			[
				'label' => esc_html__('Show Price', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => esc_html__('Show', 'mwt-addons-for-elementor'),
				'label_off' => esc_html__('Hide', 'mwt-addons-for-elementor'),
				'return_value' => 'yes',
				'default' => 'yes',
			]
		);

		$this->add_control(
			'show_list',
			[
				'label' => esc_html__('Show Feature List', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => esc_html__('Show', 'mwt-addons-for-elementor'),
				'label_off' => esc_html__('Hide', 'mwt-addons-for-elementor'),
				'return_value' => 'yes',
				'default' => 'yes',
			]
		);

		$this->add_control(
			'show_button',
			[
				'label' => esc_html__('Show Button', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => esc_html__('Show', 'mwt-addons-for-elementor'),
				'label_off' => esc_html__('Hide', 'mwt-addons-for-elementor'),
				'return_value' => 'yes',
				'default' => 'yes',
			]
		);

		$this->add_control(
			'show_ribbon',
			[
				'label' => esc_html__('Show Ribbon', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => esc_html__('Show', 'mwt-addons-for-elementor'),
				'label_off' => esc_html__('Hide', 'mwt-addons-for-elementor'),
				'return_value' => 'yes',
				'default' => 'yes',
			]
		);

		$this->end_controls_section();

		// style
		$this->start_controls_section(
			'section_ribbon_style',
			[
				'label' => esc_html__('Ribbon', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->start_controls_tabs(
			'ribbon_style_tabs'
		);

		$this->start_controls_tab(
			'ribbon_style_normal_tab',
			[
				'label' => esc_html__('Normal', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'ribbon_text_color',
			[
				'label' => esc_html__('Text Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-pricing-box-ribbon' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'ribbon_bg_color',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-pricing-box-ribbon' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'ribbon_style_hover_tab',
			[
				'label' => esc_html__('Hover', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'ribbon_text_color_hover',
			[
				'label' => esc_html__('Text Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .pricing-box-container:hover .mwt-elementor-pricing-box-ribbon' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'ribbon_bg_color_hover',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .pricing-box-container:hover .mwt-elementor-pricing-box-ribbon' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name' => 'typography_ribbon',
				'selector' => '{{WRAPPER}} .mwt-elementor-pricing-box-ribbon',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_box_style',
			[
				'label' => esc_html__('Box', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'box_padding',
			[
				'label'      => __('Padding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .pricing-box-container .pricing-box-contant-container' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'box_border',
				'selector' => '{{WRAPPER}} .pricing-box-container',
			]
		);

		$this->add_responsive_control(
			'box_radius',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .pricing-box-container' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
				],
			]
		);

		$this->start_controls_tabs(
			'box_style_tabs'
		);

		$this->start_controls_tab(
			'box_style_normal_tab',
			[
				'label' => esc_html__('Normal', 'textdomain'),
			]
		);

		$this->add_control(
			'box_bg_color',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .pricing-box-container' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'pricing_box_shadow',
				'selector' => '{{WRAPPER}} .team-box-container .mwt-elementor-team-box-wrap',
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'box_style_hover_tab',
			[
				'label' => esc_html__('Hover', 'textdomain'),
			]
		);

		$this->add_control(
			'box_bg_color_hover',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .pricing-box-container:hover' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'pricing_box_shadow_hover',
				'selector' => '{{WRAPPER}} .team-box-container .mwt-elementor-team-box-wrap:hover',
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->end_controls_section();

		$this->start_controls_section(
			'section_icon_style',
			[
				'label' => esc_html__('Icon', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'icon_size',
			[
				'label'      => __('Size', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'range' => [
					'px' => [
						'min' => 10,
						'max' => 100,
						'step' => 1,
					],
					'%' => [
						'min' => 1,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 90,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-pricing-box-icon i'  => 'font-size: {{SIZE}}{{UNIT}}',
					'{{WRAPPER}} .mwt-elementor-pricing-box-icon svg'  => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}}',
					'{{WRAPPER}} .mwt-elementor-pricing-box-image img'  => 'max-width: {{SIZE}}{{UNIT}}; max-height: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->start_controls_tabs(
			'icon_style_tabs'
		);

		$this->start_controls_tab(
			'icon_style_normal_tab',
			[
				'label' => esc_html__('Normal', 'textdomain'),
			]
		);

		$this->add_control(
			'icon_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-pricing-box-icon i' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-elementor-pricing-box-icon svg' => 'fill: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'icon_style_hover_tab',
			[
				'label' => esc_html__('Hover', 'textdomain'),
			]
		);

		$this->add_control(
			'icon_color_hover',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .pricing-box-container:hover .mwt-elementor-pricing-box-icon i' => 'color: {{VALUE}};',
					'{{WRAPPER}} .pricing-box-container:hover .mwt-elementor-pricing-box-icon svg' => 'fill: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_responsive_control(
			'icon_margin',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-pricing-box-icon' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					'{{WRAPPER}} .mwt-elementor-pricing-box-image' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_title_section',
			[
				'label' => esc_html__('Title', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'title_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-pricing-box-title' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'title_color_hover',
			[
				'label' => esc_html__('Hover Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .pricing-box-container:hover .mwt-elementor-pricing-box-title' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'title_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-elementor-pricing-box-title',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Text_Shadow::get_type(),
			[
				'label' => esc_html__('Text Shadow', 'mwt-addons-for-elementor'),
				'name' => 'position_shadow',
				'selector' => '{{WRAPPER}} .mwt-elementor-pricing-box-title',
			]
		);

		$this->add_responsive_control(
			'title_margin',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-pricing-box-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_price_section',
			[
				'label' => esc_html__('Price', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'currency_options',
			[
				'label' => esc_html__('Currency', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::HEADING,
			]
		);

		$this->add_control(
			'currency_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-pricing-box-price-currency' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'currency_color_hover',
			[
				'label' => esc_html__('Hover Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .pricing-box-container:hover .mwt-elementor-pricing-box-price-currency' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'currency_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-elementor-pricing-box-price-currency',
			]
		);

		$this->add_responsive_control(
			'currency_margin',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'allowed_dimensions' => 'horizontal',
				'placeholder' => [
					'top' => '0',
					'right' => '0',
					'bottom' => '0',
					'left' => '0',
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-pricing-box-price-currency' => 'margin-right: {{RIGHT}}{{UNIT}}; margin-left: {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'price_options',
			[
				'label' => esc_html__('Price', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'price_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-pricing-box-price' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'price_color_hover',
			[
				'label' => esc_html__('Hover Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .pricing-box-container:hover .mwt-elementor-pricing-box-price' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'price_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-elementor-pricing-box-price',
			]
		);

		$this->add_responsive_control(
			'price_margin',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'allowed_dimensions' => 'horizontal',
				'placeholder' => [
					'top' => '0',
					'right' => '0',
					'bottom' => '0',
					'left' => '0',
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-pricing-box-price' => 'margin-right: {{RIGHT}}{{UNIT}}; margin-left: {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'separator_options',
			[
				'label' => esc_html__('Separator', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'separator_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-pricing-box-price-separator' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'separator_color_hover',
			[
				'label' => esc_html__('Hover Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .pricing-box-container:hover .mwt-elementor-pricing-box-price-separator' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'separator_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-elementor-pricing-box-price-separator',
			]
		);

		$this->add_responsive_control(
			'separator_margin',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'allowed_dimensions' => 'horizontal',
				'placeholder' => [
					'top' => '0',
					'right' => '0',
					'bottom' => '0',
					'left' => '0',
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-pricing-box-price-separator' => 'margin-right: {{RIGHT}}{{UNIT}}; margin-left: {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'duration_options',
			[
				'label' => esc_html__('Duration', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'duration_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-pricing-box-price-duration' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'duration_color_hover',
			[
				'label' => esc_html__('Hover Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .pricing-box-container:hover .mwt-elementor-pricing-box-price-duration' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'duration_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-elementor-pricing-box-price-duration',
			]
		);

		$this->add_responsive_control(
			'duration_margin',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'allowed_dimensions' => 'horizontal',
				'placeholder' => [
					'top' => '0',
					'right' => '0',
					'bottom' => '0',
					'left' => '0',
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-pricing-box-price-duration' => 'margin-right: {{RIGHT}}{{UNIT}}; margin-left: {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'price_container_options',
			[
				'label' => esc_html__('Price Container', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_responsive_control(
			'price_container_margin',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-pricing-box-price-container' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_list_section',
			[
				'label' => esc_html__('Features', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'list_text_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-pricing-box-price-list-text' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'list_text_color_hover',
			[
				'label' => esc_html__('Hover Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .pricing-box-container:hover .mwt-elementor-pricing-box-price-list-text' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'list_text_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-elementor-pricing-box-price-list-text',
			]
		);

		$this->add_responsive_control(
			'list_padding',
			[
				'label'      => __('Padding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-pricing-box-list li' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'list_icon_options',
			[
				'label' => esc_html__('Icon', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'list_icon_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-pricing-box-list-icon i' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-elementor-pricing-box-list-icon svg' => 'fill: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'list_icon_color_hover',
			[
				'label' => esc_html__('Hover Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .pricing-box-container:hover .mwt-elementor-pricing-box-list-icon i' => 'color: {{VALUE}};',
					'{{WRAPPER}} .pricing-box-container:hover .mwt-elementor-pricing-box-list-icon svg' => 'fill: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'list_icon_size',
			[
				'label'      => __('Size', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'range' => [
					'px' => [
						'min' => 6,
						'max' => 100,
						'step' => 1,
					],
					'%' => [
						'min' => 1,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 16,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-pricing-box-list-icon i'  => 'font-size: {{SIZE}}{{UNIT}}',
					'{{WRAPPER}} .mwt-elementor-pricing-box-list-icon svg'  => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->add_responsive_control(
			'list_icon_spacing',
			[
				'label'      => __('Spacing', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100,
						'step' => 1,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 10,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-pricing-box-list-icon'  => 'margin-right: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_button_section',
			[
				'label' => esc_html__('Button', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'button_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-elementor-pricing-box-button',
			]
		);

		$this->start_controls_tabs(
			'style_tabs'
		);

		$this->start_controls_tab(
			'button_style_normal_tab',
			[
				'label' => esc_html__('Normal', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'button_text_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-pricing-box-button' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'button_bg_color',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-pricing-box-button' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'button_style_hover_tab',
			[
				'label' => esc_html__('Hover', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'button_text_color_hover',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-pricing-box-button:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'button_bg_color_hover',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-pricing-box-button:hover' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'button_border_color_hover',
			[
				'label' => esc_html__('Border Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-pricing-box-button:hover' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'button_border',
				'selector' => '{{WRAPPER}} .mwt-elementor-pricing-box-button',
			]
		);

		$this->add_responsive_control(
			'button_radius',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-pricing-box-button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
				],
			]
		);

		$this->add_responsive_control(
			'button_padding',
			[
				'label'      => __('Padding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-pricing-box-button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'button_margin',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-pricing-box-button' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render list widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render()
	{
		$settings = $this->get_settings_for_display(); ?>

		<div class="pricing-box-container">
			<div class="pricing-box-contant-container">
				<?php
				//ribbon
				if ('yes' === $settings['show_ribbon']) { ?>
					<div class="mwt-elementor-pricing-box-ribbon-container <?php echo esc_attr($settings['pricing_ribbon_position']) ?>">
						<span class="mwt-elementor-pricing-box-ribbon"><?php echo esc_html($settings['pricing_ribbon']) ?></span>
					</div>
					<?php
				}
				//icon
				if ('yes' === $settings['show_icon']) {
					$pricing_box_icon = isset($settings['pricing_box_icon']);
					if ($pricing_box_icon) { ?>
						<div class="mwt-elementor-pricing-box-icon">
							<?php \Elementor\Icons_Manager::render_icon($settings['pricing_box_icon'], ['aria-hidden' => 'true', 'class' => 'pricing-box-icon']); ?>
						</div>
				<?php } else {
						$pricing_box_image_url = \Elementor\Group_Control_Image_Size::get_attachment_image_src($settings['pricing_box_image']['id'], 'thumbnail', $settings);
						if (!$pricing_box_image_url && isset($settings['pricing_box_image']['url'])) {
							$pricing_box_image_url = $settings['pricing_box_image']['url'];
						}
						if (!empty($settings['pricing_title'])) {
							$image_alt = $settings['pricing_title'];
						} else {
							$image_alt = 'This is a title';
						}
						$pricing_box_image_html = '<div class="mwt-elementor-pricing-box-image"><img src="' . esc_attr($pricing_box_image_url) . '" alt="' . esc_attr($image_alt) . '" /></div>';
						echo wp_kses_post($pricing_box_image_html);
					}
				}
				?>
				<?php
				//title
				if ('yes' === $settings['show_title']) {
					$pricing_title = $settings['pricing_title'];
					$lpricing_title_tag = $settings['pricing_title_tag'];
					echo '<' . \Elementor\Utils::validate_html_tag($lpricing_title_tag) . ' ' . 'class="mwt-elementor-pricing-box-title">' . wp_kses_post($pricing_title) . '</' . \Elementor\Utils::validate_html_tag($lpricing_title_tag) . '>';
				}
				//price
				if ('yes' === $settings['show_price']) {
					$priprice_currencyce = $settings['price_currency'];
					$price = $settings['price'];
					$price_divider = $settings['price_divider'];
					$price_duration = $settings['price_duration'];
				?>
					<div class="mwt-elementor-pricing-box-price-container">
						<?php if (!empty($priprice_currencyce)) : ?>
							<span class="mwt-elementor-pricing-box-price-currency"><?php echo wp_kses_post($priprice_currencyce); ?></span>
						<?php endif; ?>
						<?php if (!empty($price)) : ?>
							<span class="mwt-elementor-pricing-box-price"><?php echo wp_kses_post($price); ?></span>
						<?php endif; ?>
						<?php if (!empty($price_divider)) : ?>
							<span class="mwt-elementor-pricing-box-price-separator"><?php echo wp_kses_post($price_divider); ?></span>
						<?php endif; ?>
						<?php if (!empty($price_duration)) : ?>
							<span class="mwt-elementor-pricing-box-price-duration"><?php echo wp_kses_post($price_duration); ?></span>
						<?php endif; ?>
					</div>
				<?php
				}
				//list
				if ('yes' === $settings['show_list']) { ?>
					<ul class="mwt-elementor-pricing-box-list">
						<?php
						foreach ($settings['feature_list_pricing'] as $index => $item) {
						?>
							<li>
								<div class="mwt-elementor-pricing-box-list-flex-wrap">
									<?php
									//icon
									$list_icon = isset($item['list_icon']);
									if (('none' !== $item['list_icon_type']) && $list_icon) : ?>
										<div class="mwt-elementor-pricing-box-list-icon">
											<?php \Elementor\Icons_Manager::render_icon($item['list_icon'], ['aria-hidden' => 'true', 'class' => 'pricing-box-list-icon']); ?>
										</div>
									<?php endif;
									//text
									$list_text = $item['list_text'];
									if (!empty($list_text)) : ?>
										<span class="mwt-elementor-pricing-box-price-list-text">
											<?php echo wp_kses_post($list_text); ?>
										</span>
									<?php endif; ?>
								</div>
							</li>
						<?php } ?>
					</ul>
				<?php } ?>
			</div>
			<?php
			//button
			if ('yes' === $settings['show_button']) { ?>
				<a class="mwt-elementor-pricing-box-button" href="<?php echo esc_url($settings['pricing_button_link']['url']); ?>">
					<?php echo esc_html($settings['pricing_button']) ?>
				</a>
			<?php } ?>
		</div>
<?php
	}
}
