<?php
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

/**
 * Elementor MWT Posts Widget.
 *
 * Elementor widget that inserts an embbedable content into the page, from any given URL.
 *
 * @since 1.0.0
 */

class Elementor_MWT_Posts extends \Elementor\Widget_Base

{

	/**
	 * Retrieve Widget Dependent JS.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array JS script handles.
	 */
	public function get_script_depends()
	{
		return ['posts-script'];
	}

	/**
	 * Retrieve Widget Dependent CSS.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array CSS style handles.
	 */
	public function get_style_depends()
	{
		return ['owl-carousel-style'];
		return ['posts-style'];
	}

	/**
	 * Get widget name.
	 *
	 * Retrieve list widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return string Widget name.
	 */
	public function get_name()
	{
		return 'mwt-posts';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve list widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return string Widget title.
	 */
	public function get_title()
	{
		return esc_html__('MWT Posts', 'mwt-addons-for-elementor');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the list widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array Widget categories.
	 */
	public function get_categories()
	{
		return ['mwt'];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve image post widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon()
	{
		return 'eicon-single-post';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @since 2.1.0
	 * @access public
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords()
	{
		return ['mwt', 'visual', 'posts', 'carousel', 'grid', 'slider'];
	}

	/**
	 * Register list widget controls.
	 *
	 * Add input fields to allow the user to customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls()
	{

		$this->start_controls_section(
			'general_settings',
			[
				'label' => __('General', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'posts_layout',
			[
				'label' => esc_html__('Posts Layout', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'carousel',
				'options' => [
					'carousel' => esc_html__('Carousel', 'mwt-addons-for-elementor'),
					'grid' => esc_html__('Masonry Grid', 'mwt-addons-for-elementor'),
				],
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'number_of_posts',
			[
				'label'       => __('Posts Per Page', 'mwt-addons-for-elementor'),
				'description' => __('Set the number of per page', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::NUMBER,
				'min'         => 1,
				'default'     => 3,
			]
		);

		function mwt_post_type_categories()
		{
			$terms = '';
			$terms = get_terms(
				array(
					'taxonomy' => 'category',
					'hide_empty' => true,
				)
			);
			$categories = array();
			if (!empty($terms) && !is_wp_error($terms)) {
				foreach ($terms as $term) {
					$categories[$term->term_id] = $term->slug;
				}
			}
			return $categories;
		}

		$this->add_control(
			'posts_filter_cat',
			[
				'label'         => __('Filter By Category', 'mwt-addons-for-elementor'),
				'type'          => \Elementor\Controls_Manager::SELECT2,
				'label_block'   => true,
				'multiple'      => true,
				'options'       => mwt_post_type_categories(),
			]
		);

		$this->add_control(
			'posts_filter',
			[
				'label'   => __('Filter Tabs', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
			]
		);

		$this->add_control(
			'posts_first_cat_label',
			[
				'label'     => __('First Category Label', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::TEXT,
				'default'   => __('All', 'mwt-addons-for-elementor'),
				'dynamic' => [
					'active' => true,
				],
				'condition'   => [
					'posts_filter' => 'yes',
				],
			]
		);

		$this->add_responsive_control(
			'posts_filters_align',
			[
				'label'     => __('Category Alignment', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::CHOOSE,
				'options'   => [
					'left' => [
						'title' => __('Left', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-left',
					],
					'center'     => [
						'title' => __('Center', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-center',
					],
					'right'   => [
						'title' => __('Right', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-right',
					],
				],
				'default'   => 'center',
				'selectors' => [
					'{{WRAPPER}} .filters' => 'text-align: {{VALUE}}',
				],
				'condition'   => [
					'posts_filter' => 'yes',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'content_section',
			[
				'label' => esc_html__('Items', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'show_featured_image',
			[
				'label'     => __('Show Featured Image', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::SWITCHER,
				'default'   => 'yes',
			]
		);

		$this->add_control(
			'show_categories',
			[
				'label'     => __('Show Categories', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::SWITCHER,
				'default'   => 'yes',
			]
		);

		$this->add_control(
			'show_categories_over_thumbnail',
			[
				'label'     => __('Show Categories Over Thumbnail', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::SWITCHER,
				'default'   => 'yes',
				'condition' => [
					'show_featured_image' => 'yes',
					'show_categories' => 'yes',
				],
			]
		);

		$this->add_control(
			'show_date',
			[
				'label'     => __('Date Meta', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::SWITCHER,
				'default'   => 'yes',
			]
		);

		$this->add_control(
			'date_icon',
			[
				'label'            => __('Date Icon', 'mwt-addons-for-elementor'),
				'type'             => \Elementor\Controls_Manager::ICONS,
				'fa4compatibility' => 'icon',
				'default'          => [
					'value'   => 'fas fa-calendar-alt',
					'library' => 'fa-solid',
				],
				'recommended'      => [
					'fa-solid'   => [
						'calendar-alt',
					],
				],
				'skin'             => 'inline',
				'label_block'      => false,
				'condition' => [
					'show_date' => 'yes',
				]
			]
		);

		$this->add_control(
			'show_author',
			[
				'label'     => __('Author Meta', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::SWITCHER,
				'default'   => 'yes',
			]
		);

		$this->add_control(
			'author_icon',
			[
				'label'            => __('Author Icon', 'mwt-addons-for-elementor'),
				'type'             => \Elementor\Controls_Manager::ICONS,
				'fa4compatibility' => 'icon',
				'default'          => [
					'value'   => 'fas fa-user',
					'library' => 'fa-solid',
				],
				'recommended'      => [
					'fa-solid'   => [
						'user',
					],
				],
				'skin'             => 'inline',
				'label_block'      => false,
				'condition' => [
					'show_author' => 'yes',
				]
			]
		);

		$this->add_control(
			'show_author_avatar',
			[
				'label'     => __('Author Avatar', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::SWITCHER,
				'default'   => 'yes',
				'condition' => [
					'show_author' => 'yes',
				]
			]
		);

		$this->add_control(
			'show_excerpt',
			[
				'label'     => __('Show Excerpt', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::SWITCHER,
				'default'   => 'yes',
			]
		);

		$this->add_control(
			'posts_excerpt_length',
			[
				'label'     => __('Excerpt Length', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::NUMBER,
				'default'   => 55,
				'condition' => [
					'show_excerpt'   => 'yes',
				]
			]
		);

		$this->add_control(
			'show_more_link',
			[
				'label'     => __('Read More Link', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::SWITCHER,
				'default'   => 'yes',
				'condition' => [
					'show_more_link_icon!'   => 'yes',
				]
			]
		);

		$this->add_control(
			'more_link_icon',
			[
				'label'            => __('More Link Icon', 'mwt-addons-for-elementor'),
				'type'             => \Elementor\Controls_Manager::ICONS,
				'fa4compatibility' => 'icon',
				'default'          => [
					'value'   => 'fas fa-angle-right',
					'library' => 'fa-solid',
				],
				'recommended'      => [
					'fa-solid'   => [
						'angle-right',
					],
				],
				'skin'             => 'inline',
				'label_block'      => false,
				'condition' => [
					'show_more_link' => 'yes',
				]
			]
		);

		$this->add_control(
			'show_more_link_icon',
			[
				'label'     => __('Read More Link Icon', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::SWITCHER,
				'condition' => [
					'show_more_link!'   => 'yes',
				]
			]
		);

		$this->add_control(
			'show_footer',
			[
				'label'     => __('Show Footer Block', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::SWITCHER,
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'carousel_settings',
			[
				'label' => __('Carousel Settings', 'mwt-addons-for-elementor'),
				'condition' => [
					'posts_layout' => 'carousel',
				]
			]
		);

		$this->add_control(
			'data_loop',
			[
				'label'   => __('Loop', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_autoplay',
			[
				'label'   => __('Autoplay', 'mwt-addons-for-elementor'),
				'description' => esc_html__('Slide will start automatically', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
				'frontend_available' => true,
			]
		);


		$this->add_control(
			'data_autoplaytimeout',
			[
				'label' => esc_html__('Autoplay Speed (ms)', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'default' => 5000,
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_autoplay_hover_pause',
			[
				'label'   => __('Pause on Hover', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_center',
			[
				'label'   => __('Centered Carousel?', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
				'frontend_available' => true,
			]
		);

		//Slides to Scroll

		$this->add_control(
			'data_responsive_lg',
			[
				'label' => esc_html__('Desktop Slides', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'default' => 3,
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_responsive_sm',
			[
				'label' => esc_html__('Tabs Slides', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'default' => 2,
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_responsive_xs',
			[
				'label' => esc_html__('Mobile Slides', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'default' => 1,
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_margin',
			[
				'label' => esc_html__('Item Gap (px)', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'default' => 15,
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_nav',
			[
				'label'   => __('Navigation', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
				'frontend_available' => true,
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'posts_grid_settings',
			[
				'label' => __('Grid Settings', 'mwt-addons-for-elementor'),
				'condition' => [
					'posts_layout'   => 'grid',
				]
			]
		);

		$this->add_responsive_control(
			'posts_items_per_row',
			[
				'label' => esc_html__('Items per row', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'description' => esc_html__('How many items should be shown per row?', 'mwt-addons-for-elementor'),
				'min'  => 1,
				'max'  => 4,
				'step' => 1,
				'default' => 3,
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'posts_layout_gap',
			[
				'label' => esc_html__('Gap', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'solid',
				'options' => [
					'30' => esc_html__('Default', 'mwt-addons-for-elementor'),
					'0' => esc_html__('None', 'mwt-addons-for-elementor'),
					'1'  => esc_html__('1px', 'mwt-addons-for-elementor'),
					'2' => esc_html__('2px', 'mwt-addons-for-elementor'),
					'3' => esc_html__('3px', 'mwt-addons-for-elementor'),
					'4' => esc_html__('4px', 'mwt-addons-for-elementor'),
					'5' => esc_html__('5px', 'mwt-addons-for-elementor'),
					'10' => esc_html__('10px', 'mwt-addons-for-elementor'),
					'15' => esc_html__('15px', 'mwt-addons-for-elementor'),
					'20' => esc_html__('20px', 'mwt-addons-for-elementor'),
					'30' => esc_html__('30px', 'mwt-addons-for-elementor'),
					'40' => esc_html__('40px', 'mwt-addons-for-elementor'),
					'50' => esc_html__('50px', 'mwt-addons-for-elementor'),
					'60' => esc_html__('60px', 'mwt-addons-for-elementor'),
				],
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'posts_horizontal_order',
			[
				'label' => esc_html__('Horizontal Order', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => esc_html__('Yes', 'mwt-addons-for-elementor'),
				'label_off' => esc_html__('No', 'mwt-addons-for-elementor'),
				'return_value' => 'yes',
				'default' => 'no',
			]
		);

		$this->end_controls_section();

		// style
		$this->start_controls_section(
			'style_image_section',
			[
				'label' => esc_html__('Image', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
				'condition'   => [
					'show_featured_image' => 'yes',
				],
			]
		);

		$this->add_control(
			'overlay_color',
			[
				'label' => esc_html__('Overlay Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-posts-article .post-thumbnail a:before' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'image_border_radius',
			[
				'label'      => esc_html__('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-posts-article .post-thumbnail a:before' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					'{{WRAPPER}} .mwt-elementor-posts-article .post-thumbnail a img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_categories_section',
			[
				'label' => esc_html__('Categories', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'show_categories' => 'yes',
				]
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'categories_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-elementor-posts-categories a',
			]
		);

		$this->start_controls_tabs(
			'categories_style_tabs'
		);

		$this->start_controls_tab(
			'categories_style_normal_tab',
			[
				'label' => esc_html__('Normal', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'categories_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-posts-categories a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'categories_background_color',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-posts-categories a' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'categories_style_hover_tab',
			[
				'label' => esc_html__('Hover', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'categories_color_hover',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-posts-categories a:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'categories_background_color_hover',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-posts-categories a:hover' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'categories_border_color_hover',
			[
				'label'     => __('Border Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-posts-categories a:hover' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'categories_border',
				'selector' => '{{WRAPPER}} .mwt-elementor-posts-categories a',
			]
		);

		$this->add_control(
			'categories_border_radius',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-posts-categories a'  => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'categories_padding',
			[
				'label'      => __('Padding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-posts-categories a' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'categories_margin',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-posts-categories a' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_meta_section',
			[
				'label' => esc_html__('Meta', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
				'conditions' => [
					'relation' => 'or',
					'terms' => [
						[
							'name' => 'show_date',
							'operator' => '===',
							'value' => 'yes',
						],
						[
							'name' => 'show_author',
							'operator' => '===',
							'value' => 'yes',
						],
					],
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'meta_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-elementor-posts-date, {{WRAPPER}} .mwt-elementor-posts-date a, {{WRAPPER}} .mwt-elementor-posts-author, {{WRAPPER}} .mwt-elementor-posts-author a',

			]
		);

		$this->add_control(
			'meta_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-posts-date a' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-elementor-posts-author a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'meta_color_hover',
			[
				'label' => esc_html__('Color Hover', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-posts-date a:hover' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-elementor-posts-author a:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'meta_spacing',
			[
				'label'      => __('Bottom Spacing', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-posts-item-content .mwt-elementor-posts-meta'  => 'margin-bottom: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->add_control(
			'meta_icon',
			[
				'label' => esc_html__('Icon', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'meta_icon_color',
			[
				'label' => esc_html__('Icon Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-posts-date-icon' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-elementor-posts-date svg' => 'fill: {{VALUE}};',
					'{{WRAPPER}} .mwt-elementor-posts-author-icon' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-elementor-posts-author svg' => 'fill: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'meta_icon_size',
			[
				'label'      => __('Icon Size', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 64,
						'step' => 1,
					],
					'em' => [
						'min' => 0,
						'max' => 10,
						'step' => 1,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 14,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-posts-date i'  => 'font-size: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .mwt-elementor-posts-date svg'  => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .mwt-elementor-posts-author i'  => 'font-size: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .mwt-elementor-posts-author svg'  => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'meta_icon_spacing',
			[
				'label'      => __('Icon Right Spacing', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-posts-date-icon'  => 'margin-right: {{SIZE}}{{UNIT}}',
					'{{WRAPPER}} .mwt-elementor-posts-author-icon'  => 'margin-right: {{SIZE}}{{UNIT}}',
					'{{WRAPPER}} .mwt-elementor-posts-date svg'  => 'margin-right: {{SIZE}}{{UNIT}}',
					'{{WRAPPER}} .mwt-elementor-posts-author svg'  => 'margin-right: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->add_control(
			'author_avatar',
			[
				'label' => esc_html__('Author Avatar', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
				'condition'   => [
					'show_author' => 'yes',
					'show_author_avatar' => 'yes',
				],
			]
		);

		$this->add_responsive_control(
			'author_avatar_size',
			[
				'label'      => __('Author Avatar Size', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em'],
				'range' => [
					'px' => [
						'min' => 40,
						'max' => 100,
						'step' => 1,
					],
					'em' => [
						'min' => 0,
						'max' => 10,
						'step' => 1,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 60,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-posts-author-avatar'  => 'max-width: {{SIZE}}{{UNIT}}; max-height: {{SIZE}}{{UNIT}};',
				],
				'condition'   => [
					'show_author' => 'yes',
					'show_author_avatar' => 'yes',
				],
			]
		);

		$this->add_control(
			'author_avata_position',
			[
				'label' => esc_html__('Position', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'relative',
				'options' => [
					'relative' => esc_html__('Default', 'mwt-addons-for-elementor'),
					'absolute' => esc_html__('Absolute', 'mwt-addons-for-elementor'),
				],
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-posts-author-avatar' => 'position: {{VALUE}};',
				],
				'condition'   => [
					'show_author' => 'yes',
					'show_author_avatar' => 'yes',
				],
			]
		);

		$this->add_responsive_control(
			'author_avatar_position_x',
			[
				'label'      => __('Position X', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 200,
						'step' => 1,
					],
					'em' => [
						'min' => 0,
						'max' => 10,
						'step' => 1,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 50,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-posts-author-avatar'  => 'left: {{SIZE}}{{UNIT}};',
				],
				'condition'   => [
					'show_author' => 'yes',
					'show_author_avatar' => 'yes',
					'author_avata_position' => 'absolute'
				],
			]
		);

		$this->add_responsive_control(
			'author_avatar_position_y',
			[
				'label'      => __('Position Y', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 200,
						'step' => 1,
					],
					'em' => [
						'min' => 0,
						'max' => 10,
						'step' => 1,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => -30,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-posts-author-avatar'  => 'top: {{SIZE}}{{UNIT}};',
				],
				'condition'   => [
					'show_author' => 'yes',
					'show_author_avatar' => 'yes',
					'author_avata_position' => 'absolute'
				],
			]
		);

		$this->add_responsive_control(
			'author_avatar_bottom_spacing',
			[
				'label'      => __('Bottom Spacing', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-posts-author-avatar'  => 'margin-bottom: {{SIZE}}{{UNIT}}',
				],
				'condition'   => [
					'show_author' => 'yes',
					'show_author_avatar' => 'yes',
					'author_avata_position' => 'relative'
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_title_section',
			[
				'label' => esc_html__('Title', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'title_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-elementor-posts-item-content .mwt-elementor-posts-post-title',
				'{{WRAPPER}} .mwt-elementor-posts-title',
				'{{WRAPPER}} .mwt-elementor-posts-title > a',
			]
		);

		$this->add_control(
			'title_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-posts-title' => 'color: {{VALUE}};',
					'{{WRAPPER}} a.mwt-elementor-posts-title' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'title_color_hover',
			[
				'label' => esc_html__('Color Hover', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} a.mwt-elementor-posts-title:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'title_height',
			[
				'label'      => __('Height', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'default'    => [
					'unit' => 'px',
					'size' => 60,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-posts-item-content .mwt-elementor-posts-post-title'  => 'height: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->add_responsive_control(
			'title_margin',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-posts-item-content .mwt-elementor-posts-post-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);


		$this->end_controls_section();

		$this->start_controls_section(
			'style_btn_section',
			[
				'label' => esc_html__('Read More Link', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
				'condition'   => [
					'show_more_link' => 'yes',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'btn_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-elementor-posts-btn-arrow a',
			]
		);

		$this->start_controls_tabs(
			'btn_style_tabs'
		);

		$this->start_controls_tab(
			'btn_style_normal_tab',
			[
				'label' => esc_html__('Normal', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'btn_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-posts-btn-arrow a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'btn_background_color',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-posts-btn-arrow a' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'btn_style_hover_tab',
			[
				'label' => esc_html__('Hover', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'btn_color_active',
			[
				'label' => esc_html__('Color Hover', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-posts-btn-arrow a:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'btn_background_color_hover',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-posts-btn-arrow a:hover' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'btn_border_color_hover',
			[
				'label' => esc_html__('Border Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-posts-btn-arrow a:hover' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'btn_border',
				'selector' => '{{WRAPPER}} .mwt-elementor-posts-btn-arrow a',
			]
		);

		$this->add_control(
			'btn_border_radius',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-posts-btn-arrow a'  => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'btn_padding',
			[
				'label'      => __('Padding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-posts-btn-arrow a' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'btn_min_width',
			[
				'label'      => __('Min Width', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-posts-btn-arrow a'  => 'min-width: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->add_responsive_control(
			'btn_margin',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-posts-btn-arrow' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'btn_icon_heading',
			[
				'label' => esc_html__('Icon', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'btn_icon_color',
			[
				'label' => esc_html__('Icon Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-posts-btn-arrow i' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-elementor-posts-btn-arrow svg' => 'fill: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'btn_icon_hover_color',
			[
				'label' => esc_html__('Icon Hover Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-posts-btn-arrow a:hover i' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-elementor-posts-btn-arrow a:hover svg' => 'fill: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'btn_icon_size',
			[
				'label'      => __('Icon Size', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 64,
						'step' => 1,
					],
					'em' => [
						'min' => 0,
						'max' => 10,
						'step' => 1,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 14,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-posts-btn-arrow i'   => 'font-size: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .mwt-elementor-posts-btn-arrow svg' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'btn_icon_spacing',
			[
				'label'      => __('Icon Right Spacing', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-posts-more-link-icon'  => 'margin-left: {{SIZE}}{{UNIT}}',
					'{{WRAPPER}} .mwt-elementor-posts-btn-arrow svg'  => 'margin-left: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_btn_icon_section',
			[
				'label' => esc_html__('Read More Link Icon', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'show_more_link_icon' => 'yes'
				]
			]
		);

		$this->add_responsive_control(
			'more_link_icon_size',
			[
				'label'       => __('Size', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::SLIDER,
				'size_units'  => ['px', 'em'],
				'separator' => 'before',
				'label_block' => true,
				'range'       => [
					'px' => [
						'min' => 1,
						'max' => 48,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 16,
				],
				'selectors'   => [
					'{{WRAPPER}} .mwt-elementor-posts-read-more' => 'font-size: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'show_more_link_icon' => 'yes'
				]
			]
		);

		$this->add_control(
			'more_link_icon_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-posts-read-more::before' => 'color: {{VALUE}};',
				],
				'condition' => [
					'show_more_link_icon' => 'yes'
				]
			]
		);

		$this->add_control(
			'more_link_icon_color_hover',
			[
				'label' => esc_html__('Hover Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-posts-read-more:hover::before' => 'color: {{VALUE}};',
				],
				'condition' => [
					'show_more_link_icon' => 'yes'
				]
			]
		);

		$this->add_responsive_control(
			'more_link_icon_margin',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-posts-read-more' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition' => [
					'show_more_link_icon' => 'yes'
				]
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_content_section',
			[
				'label' => esc_html__('Content Box', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'content_box_text_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-posts-excerpt' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'content_box_text_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-elementor-posts-excerpt',
			]
		);

		$excerpt_conditions = array(
			'show_excerpt' => 'yes',
		);

		$this->add_responsive_control(
			'content_box_excerpt_margin',
			[
				'label'      => __('Excerpt Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-posts-excerpt' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition' => $excerpt_conditions,
			]
		);

		$this->add_control(
			'content_box_background_color',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-posts-item-content'  => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'content_box_padding',
			[
				'label'      => __('Padding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-posts-item-content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'content_box_radius',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-posts-item-content' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
				],
			]
		);

		$this->add_responsive_control(
			'content_box_margin',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-posts-item-content' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'content_box_align',
			[
				'label'     => __('Alignment', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::CHOOSE,
				'options'   => [
					'left' => [
						'title' => __('Left', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-left',
					],
					'center'     => [
						'title' => __('Center', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-center',
					],
					'right'   => [
						'title' => __('Right', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-right',
					],
				],
				'default'   => 'Left',
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-posts-item-content' => 'text-align: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'content_box_shadow',
				'selector' => '{{WRAPPER}} .mwt-elementor-posts-item-content',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_box_section',
			[
				'label' => esc_html__('Box', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'box_background_color',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-posts-article'  => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'box_border',
				'selector' => '{{WRAPPER}} .mwt-elementor-posts-article',
			]
		);

		$this->add_control(
			'box_border_radius',
			[
				'label'      => esc_html__('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-posts-article' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'outer_box_shadow',
				'selector' => '{{WRAPPER}} .mwt-elementor-posts-article',
			]
		);

		$this->add_responsive_control(
			'box_padding',
			[
				'label'      => __('Box Padding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .owl-carousel-container.mwt-elementor-posts-carousel .owl-carousel .owl-item' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'box_overflow',
			[
				'label' => esc_html__('Overflow', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'visible',
				'options' => [
					'hidden' => esc_html__('Hidden', 'mwt-addons-for-elementor'),
					'visible' => esc_html__('Visible', 'mwt-addons-for-elementor'),
				],
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-posts-article' => 'overflow: {{VALUE}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_navigation_section',
			[
				'label' => esc_html__('Navigation', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
				'condition'   => [
					'posts_layout' => 'carousel',
					'data_nav' => 'yes',
				],
			]
		);

		$this->add_control(
			'nav_layout',
			[
				'label' => esc_html__('Navigation Layout', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::HIDDEN,
				'default' => 'nav-layout-1',
				'options' => [
					'nav-layout-1' => esc_html__('Layout 1', 'mwt-addons-for-elementor'),
					'nav-layout-2' => esc_html__('Layout 2', 'mwt-addons-for-elementor'),
					'nav-layout-3' => esc_html__('Layout 3', 'mwt-addons-for-elementor'),
					'nav-layout-4' => esc_html__('Layout 4', 'mwt-addons-for-elementor'),
				],
				'frontend_available' => true,
			]
		);

		$this->add_responsive_control(
			'nav_icon_size',
			[
				'label'      => __('Icon Size', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em'],
				'range'      => [
					'px' => [
						'min'  => 12,
						'max'  => 40,
						'step' => 1,
					],
					'em' => [
						'min' => 0,
						'max' => 10,
						'step' => 1,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 20,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-posts-carousel .owl-nav > button:after' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->start_controls_tabs(
			'style_posts_nav_tabs'
		);

		$this->start_controls_tab(
			'style_posts_nav_normal_tab',
			[
				'label' => esc_html__('Normal', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'posts_nav_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-posts-carousel .posts-slider .owl-nav > button:after' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'posts_nav_bg_color',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-posts-carousel .posts-slider .owl-nav > button' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'style_posts_nav_hover_tab',
			[
				'label' => esc_html__('Hover', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'posts_nav_color_hover',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-posts-carousel .posts-slider .owl-nav > button:hover:after' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'posts_nav_bg_color_hover',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-posts-carousel .posts-slider .owl-nav > button:hover' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'posts_nav_border_color_hover',
			[
				'label' => esc_html__('Border Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-posts-carousel .posts-slider .owl-nav > button:hover' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'posts_nav_border',
				'selector' => '{{WRAPPER}} .mwt-elementor-posts-carousel .owl-nav > button',
			]
		);

		$this->add_responsive_control(
			'posts_nav_radius',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-posts-carousel .posts-slider .owl-nav > button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
				],
			]
		);

		$this->add_responsive_control(
			'posts_nav_padding',
			[
				'label'      => __('Padding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-posts-carousel .posts-slider .owl-nav > button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'posts_nav_margin',
			[
				'label'      => esc_html__('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-posts-carousel .posts-slider .owl-nav > button' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'posts_nav_offset',
			[
				'label'      => esc_html__('Offset', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%'],
				'range' => [
					'px' => [
						'min' => -1000,
						'max' => 1000,
						'step' => 1,
					],
					'%' => [
						'min' => -100,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 0,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-posts-carousel .owl-theme .owl-nav'  => 'left: {{SIZE}}{{UNIT}}; right: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'posts_filter_style',
			[
				'label'     => __('Filter', 'mwt-addons-for-elementor'),
				'tab'       => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'posts_filter' => 'yes',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'posts_filter_typo',
				'selector' => '{{WRAPPER}} .filters a.mwt-elementor-posts-category',
			]
		);

		$this->start_controls_tabs('posts_filters');

		$this->start_controls_tab(
			'posts_filters_normal',
			[
				'label' => __('Normal', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'posts_filter_color',
			[
				'label'     => __('Text Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .filters a.mwt-elementor-posts-category' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'posts_background_color',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .filters a.mwt-elementor-posts-category' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'posts_filter_border',
				'selector' => '{{WRAPPER}} .filters a.mwt-elementor-posts-category',
			]
		);

		$this->add_control(
			'posts_filter_border_radius',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .filters a.mwt-elementor-posts-category'  => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);


		$this->end_controls_tab();

		$this->start_controls_tab(
			'posts_filters_hover',
			[
				'label' => __('Hover', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'posts_filter_hover_color',
			[
				'label'     => __('Text Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .filters a.mwt-elementor-posts-category:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'posts_background_hover_color',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .filters a.mwt-elementor-posts-category:hover' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'posts_filter_border_hover',
				'selector' => '{{WRAPPER}} .filters a.mwt-elementor-posts-category:hover',
			]
		);

		$this->add_control(
			'posts_filter_border_radius_hover',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .filters a.mwt-elementor-posts-category:hover'  => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'posts_filters_active',
			[
				'label' => __('Active', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'posts_filter_active_color',
			[
				'label'     => __('Text Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .filters a.mwt-elementor-posts-category.active' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'posts_background_active_color',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .filters a.mwt-elementor-posts-category.active' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'posts_filter_border_active',
				'selector' => '{{WRAPPER}} .filters a.mwt-elementor-posts-category.active',
			]
		);

		$this->add_control(
			'posts_filter_border_radius_active',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .filters a.mwt-elementor-posts-category.active'  => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'posts_filter_shadow',
				'selector' => '{{WRAPPER}} .filters a.mwt-elementor-posts-category',
			]
		);

		$this->add_responsive_control(
			'posts_filter_margin',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .filters a.mwt-elementor-posts-category' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'posts_filter_padding',
			[
				'label'      => __('Padding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .filters a.mwt-elementor-posts-category' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_footer_section',
			[
				'label' => esc_html__('Footer', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'show_footer' => 'yes',
				],
			]
		);

		$this->add_responsive_control(
			'footer_align',
			[
				'label' => esc_html__('Alignment', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::CHOOSE,
				'options' => [
					'left' => [
						'title' => esc_html__('Left', 'mwt-addons-for-elementor'),
						'icon' => 'eicon-text-align-left',
					],
					'center' => [
						'title' => esc_html__('Center', 'mwt-addons-for-elementor'),
						'icon' => 'eicon-text-align-center',
					],
					'right' => [
						'title' => esc_html__('Right', 'mwt-addons-for-elementor'),
						'icon' => 'eicon-text-align-right',
					],
				],
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-posts-entry-footer' => 'text-align: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'footer_padding',
			[
				'label'      => __('Padding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-posts-entry-footer' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'footer_bg_color',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-posts-entry-footer' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'footer_border',
				'selector' => '{{WRAPPER}} .mwt-elementor-posts-entry-footer',
			]
		);

		$this->add_responsive_control(
			'footer_border_radius',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-posts-entry-footer' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
				],
			]
		);

		$this->add_control(
			'footer_content',
			[
				'label' => esc_html__('Content', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'footer_text_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-elementor-posts-entry-footer',
			]
		);

		$this->add_control(
			'footer_text_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-posts-entry-footer' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-elementor-posts-entry-footer a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'footer_text_color_hover',
			[
				'label' => esc_html__('Hover Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-posts-entry-footer a:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'footer_text_item_margin',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-posts-entry-footer>span' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		function mwt_icon($mwt_name, $mwt_return = false, $mwt_container_css_class = 'svg-icon')
		{
			//in the future we'll add option for this
			$mwt_icons_pack = 'google';

			if ($mwt_return) {
				ob_start();
			}

			echo '<span class="' . esc_attr($mwt_container_css_class) . ' icon-' . esc_attr($mwt_name) . '">';
			get_template_part('/template-parts/svg/' . $mwt_icons_pack . '/' . $mwt_name);
			echo '</span>';

			if ($mwt_return) {
				return ob_get_clean();
			}
		}

		/* Check if has post thumbnail and thumbnail file exists */
		function mwt_has_post_thumbnail($mwt_id = false)
		{
			if (empty($mwt_id)) {
				$mwt_id = get_the_ID();
			}

			return !(post_password_required($mwt_id)
				||
				is_attachment()
				||
				!has_post_thumbnail($mwt_id)
				||
				!file_exists(get_attached_file(get_post_thumbnail_id($mwt_id))));
		}

		/* Displays an optional post thumbnail. */
		function mwt_post_thumbnail($mwt_size = 'full', $mwt_css_class = '')
		{
			if (!mwt_has_post_thumbnail()) {
				return;
			}

			//detect video
			$oembed_url = false;
			$oembed_post_thumbnail = false;
			//only video post format
			if ('video' === get_post_format()) {
				$oembed_post_thumbnail = true;
			}

			if ($oembed_post_thumbnail) {
				$post_content = get_the_content();
				//get oEmbed URL
				$reg = preg_match('|^\s*(https?://[^\s"]+)\s*$|im', $post_content, $matches);

				$oembed_url = !empty($reg) ? trim($matches[0]) : false;
				//if no youtube, trying to find self hosted

				$embeds = array();
				if (empty($oembed_url)) {
					$post_content = apply_filters('the_content', $post_content);
					$embeds = get_media_embedded_in_content($post_content);
				}
			}

			//detecting gallery
			$is_gallery = false;
			$image_size = 'full';
			if ('gallery' === get_post_format()) {
				$galleries_images = get_post_galleries_images();
				//gutenberg block parse
				if (!empty($galleries_images)) {
					global $post;
					if (has_block('gallery', $post->post_content)) {
						$post_blocks = parse_blocks($post->post_content);
						foreach ($post_blocks as $post_block) {
							if ('core/gallery' === $post_block['blockName']) {
								$src_array = array();
								$gallery_imgs_ids = [];
								foreach ($post_block['innerBlocks'] as $subel) {
									$gallery_imgs_ids[] = $subel['attrs']['id'];
								}
								foreach ($gallery_imgs_ids as $id) {
									$src_array[] = wp_get_attachment_image_url($id, $image_size);
								}
								$galleries_images = $src_array;
								break;
							}
						}
					}
				}

				$galleries_images_count = count($galleries_images);
				if ($galleries_images_count) {
					$is_gallery = true;
				}
			} ?>
			<div class="post-thumbnail-wrap">
				<figure class="<?php echo esc_attr('post-thumbnail ' . $mwt_css_class); ?>">
					<a href="<?php if (empty($oembed_url)) {
									the_permalink();
								} else {
									echo esc_url($oembed_url);
								} ?>">
						<?php
						if (empty($is_gallery)) {
							the_post_thumbnail(
								$mwt_size,
								array(
									'itemprop' => 'image',
									'alt'      => get_the_title(),
								)
							);
						} else {
							//gallery
							echo '<div class="flexslider"><ul class="slides">';

							//adding featured image as a first element in carousel
							//featured image url
							$post_featured_image_src = wp_get_attachment_image_url(get_post_thumbnail_id(), $image_size);
							if ($post_featured_image_src) : ?>
								<li><img src="<?php echo esc_url($post_featured_image_src); ?>" alt="<?php echo esc_attr(get_the_title()); ?>"></li>
							<?php endif;
							$count = 1;

							foreach ($galleries_images as $gallerie) :
								//showing only 3 images from gallery
								if ($count > 9) {
									break;
								}
							?>
								<li><img src="<?php echo esc_url($gallerie); ?>" alt="<?php echo esc_attr(get_the_title()); ?>"></li>
						<?php
								$count++;
							endforeach;
							echo '</ul></div><!--.flexslider-->';
						}
						?>
						<span class="mwt-elementor-posts-icon-video">
							<?php
							if ('video' === get_post_format()) {
								mwt_icon('video');
							}
							?>
						</span>
					</a>
				</figure>
			</div>
			<!-- post-thumbnail-wrap -->
		<?php
		}
	}

	/**
	 * Render list widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render()
	{
		$settings = $this->get_settings_for_display();

		$unique_id = uniqid();

		$posts_cat_array = (!empty($settings['posts_filter_cat'])) ? implode(",", $settings['posts_filter_cat']) : '-1';

		$mwt_posts = new WP_Query(array(
			'post_type'           => 'post',
			'cat' 				  => $posts_cat_array,
			'orderby'             => 'post_date',
			'order '              => 'DESC',
			'ignore_sticky_posts' => true,
			'posts_per_page'      => $settings['number_of_posts'],
			'tax_query' => array(
				array(
					'taxonomy' => 'post_format',
					'field' => 'slug',
					'terms' => array('post-format-quote', 'post-format-status', 'post-format-link', 'post-format-aside', 'post-format-chat'),
					'operator' => 'NOT IN'
				)
			)
		)); ?>
		<?php
		//carousel layout
		if ($settings['posts_layout'] === 'carousel') : ?>
			<div class="owl-carousel-container mwt-elementor-posts-carousel">
			<?php endif; //end carousel layout 

		//grid layout
		if ($settings['posts_layout'] === 'grid') : ?>
				<div class="mwt-elementor-posts-grid">
				<?php endif; //end grid layout 

			$filter = $settings['posts_filter'];
			if ('yes' === $filter) :
				//get unique terms only for posts that are showing
				$showing_terms = array();
				foreach ($mwt_posts->posts as $post) {
					foreach (get_the_terms($post->ID, 'category') as $post_term) {
						$showing_terms[$post_term->term_id] = $post_term;
					}
				} ?>
					<div class="filters carousel_filters-<?php echo esc_attr($unique_id); ?>">
						<a href="#" class="mwt-elementor-posts-category selected active" data-filter="*">
							<?php echo wp_kses_post($settings['posts_first_cat_label']); ?>
						</a>
						<?php
						foreach ($showing_terms as $term_key => $term_id) {
							$current_term = get_term($term_id, 'category');
						?>
							<a class="mwt-elementor-posts-category" href="#" data-filter="<?php ($settings['posts_layout'] === 'carousel') ? '.' : ''; ?><?php echo esc_attr($current_term->slug); ?>"><?php echo esc_html($current_term->name); ?></a>
						<?php } //foreach 
						?>
					</div>
				<?php endif;

			//carousel layout
			if ($settings['posts_layout'] === 'carousel') :
				$data_loop = ($settings['data_loop'] === 'yes') ? 'true' : 'false';
				$data_autoplay = ($settings['data_autoplay'] === 'yes') ? 'true' : 'false';
				$data_center = ($settings['data_center'] === 'yes') ? 'true' : 'false';
				$data_nav = ($settings['data_nav'] === 'yes') ? 'true' : 'false';
				$data_autoplay_hover_pause = ($settings['data_autoplay_hover_pause'] === 'yes') ? 'true' : 'false'; ?>
					<div id="widget_portfolio_carousel_<?php echo esc_attr($unique_id); ?>" class="owl-carousel posts-slider <?php echo esc_attr($settings['nav_layout']); ?>" data-prev="Prev post" data-next="Next post" data-loop="<?php echo esc_attr($data_loop) ?>" data-autoplay="<?php echo esc_attr($data_autoplay) ?>" data-autoplaytimeout="<?php echo esc_attr($settings['data_autoplaytimeout']) ?>" data-autoplayhoverpause="<?php echo esc_attr($data_autoplay_hover_pause) ?>" data-center="<?php echo esc_attr($data_center) ?>" data-nav="<?php echo esc_attr($data_nav) ?>" data-margin="<?php echo esc_attr($settings['data_margin']) ?>" data-responsive-lg="<?php echo esc_attr($settings['data_responsive_lg']) ?>" data-responsive-sm="<?php echo esc_attr($settings['data_responsive_sm']) ?>" data-responsive-xs="<?php echo esc_attr($settings['data_responsive_xs']) ?>" data-filters=".carousel_filters-<?php echo esc_attr($unique_id); ?>">
					<?php endif; //end carousel layout  

				//grid layout
				if ($settings['posts_layout'] === 'grid') :
					$horizontal_order = ($settings['posts_horizontal_order'] === 'yes') ? true : false;
					?>
						<div class="layout-cols layout-cols-<?php echo esc_attr($settings['posts_items_per_row']); ?> layout-gap-<?php echo esc_attr($settings['posts_layout_gap']); ?>">
							<div class="grid-columns-wrapper">
								<div id="widget_portfolio_grid_<?php echo esc_attr($unique_id); ?>" class="masonry" data-filters=".grid_filters-<?php echo esc_attr($unique_id); ?>" data-order="<?php echo esc_attr($horizontal_order); ?>">
									<div class="grid-sizer"></div>
									<?php endif; //end grid layout 

								while ($mwt_posts->have_posts()) :
									$mwt_posts->the_post();
									$mwt_post_title     = get_the_title($mwt_posts->ID);
									$mwt_title          = (!empty($mwt_post_title)) ? $mwt_post_title : esc_html__('(no title)', 'mwt-addons-for-elementor');

									$post_terms = get_the_terms(get_the_ID(), 'category');
									$post_terms_class = '';
									$post_terms_name = '';
									$post_terms_id = '';
									$cat = '';
									foreach ($post_terms as $post_term) {
										$post_terms_class .= $post_term->slug . ' ';
										$post_terms_name = $post_term->name . ' ';
										$post_terms_id = $post_term->term_id . ' ';
										$cat .= '<a href="' . esc_url(get_category_link($post_terms_id)) . '" rel="tag">' . $post_terms_name . '</a>';
									}
									add_image_size('mwt-post-thumbnail-size', 1170, 780, true);

									//carousel layout
									if ($settings['posts_layout'] === 'carousel') : ?>
										<div class="owl-carousel-item <?php echo esc_attr($post_terms_class); ?>">
										<?php endif; //end carousel layout 

									//grid layout
									if ($settings['posts_layout'] === 'grid') : ?>
											<div class="grid-item <?php echo esc_attr($post_terms_class); ?>">
											<?php endif; //end grid layout  
											?>
											<article <?php post_class('mwt-elementor-posts-article'); ?>>
												<?php $show_featured_image = $settings['show_featured_image'];
												if ('yes' === $show_featured_image) :
													mwt_post_thumbnail('mwt-post-thumbnail-size');
												?>
													<?php
													if ($settings['show_categories'] === 'yes' && $settings['show_categories_over_thumbnail'] === 'yes') : ?>
														<div class="mwt-elementor-posts-categories">
															<?php echo ($cat); ?>
														</div>
													<?php
													endif;
													?>
												<?php endif ?>
												<div class="mwt-elementor-posts-item-content">
													<?php
													if ('yes' === ($settings['show_author_avatar'])) :
														echo '<span class="mwt-elementor-posts-author-avatar">';
														$author_id        = get_the_author_meta('ID');
														$custom_image_url = get_the_author_meta('custom_profile_image', $author_id);
														if (!empty($custom_image_url)) {
															echo '<img src="' . esc_url($custom_image_url) . '" alt="' . esc_attr(get_the_author_meta('display_name', $author_id)) . '">';
														} else {
															echo get_avatar($author_id, 60);
														}
														echo '</span>';
													endif; //$author_avatar
													?>
													<?php
													$show_date = $settings['show_date'];
													$show_author = $settings['show_author'];
													if ('yes' === $show_date || 'yes' === $show_author) :
													?>
														<div class="mwt-elementor-posts-meta">
															<?php
															if ('yes' === $show_date) :
															?>
																<div class="mwt-elementor-posts-date">
																	<?php
																	$date_icon = isset($settings['date_icon']) ? $settings['date_icon'] : '';
																	if ($date_icon) : ?>
																		<?php \Elementor\Icons_Manager::render_icon($settings['date_icon'], ['aria-hidden' => 'true', 'class' => 'mwt-elementor-posts-date-icon']); ?>
																	<?php endif; ?>
																	<a href="<?php the_permalink($mwt_posts->ID); ?>"><?php echo get_the_date('', $mwt_posts->ID); ?></a>
																</div>
															<?php endif ?>
															<?php
															if ($settings['show_categories'] === 'yes' && $settings['show_categories_over_thumbnail'] !== 'yes') : ?>
																<div class="mwt-elementor-posts-categories">
																	<?php echo ($cat); ?>
																</div>
															<?php
															endif ?>
															<?php
															if ('yes' === $show_author) :
															?>
																<div class="mwt-elementor-posts-author">
																	<?php
																	$author_icon = isset($settings['author_icon']) ? $settings['author_icon'] : '';
																	if ($author_icon !== '') : ?>
																		<?php \Elementor\Icons_Manager::render_icon($settings['author_icon'], ['aria-hidden' => 'true', 'class' => 'mwt-elementor-posts-author-icon']); ?>
																	<?php endif; ?>
																	<a href="<?php echo esc_url(get_author_posts_url(get_the_author_meta('ID'))) ?>"><?php esc_html_e('by ', 'mwt-addons-for-elementor'); ?><?php echo get_the_author() ?></a>
																</div>
															<?php endif ?>
														</div>
													<?php endif ?>
													<h4 class="mwt-elementor-posts-post-title">
														<a class="mwt-elementor-posts-title" href="<?php the_permalink($mwt_posts->ID); ?>"><?php echo wp_kses_post($mwt_title); ?></a>
													</h4>
													<?php $show_excerpt = $settings['show_excerpt'];
													if ('yes' === $show_excerpt) : ?>
														<div class="mwt-elementor-posts-excerpt">
															<?php
															if (!empty($settings['posts_excerpt_length'])) {
																$length = $settings['posts_excerpt_length'];
															} else {
																$length = 55;
															}
															echo wp_kses_post(wp_trim_words(get_the_excerpt($mwt_posts->ID), $length));
															?>
														</div>
													<?php endif ?>
													<?php $show_more_link = $settings['show_more_link'];
													if ('yes' === $show_more_link) : ?>
														<div class="mwt-elementor-posts-btn-arrow">
															<a class="btn-arrow" href="<?php the_permalink(); ?>"><?php echo esc_html__('Read More', 'mwt-addons-for-elementor') ?>
																<?php
																$more_link_icon = isset($settings['more_link_icon']) ? $settings['more_link_icon'] : '';
																if ($more_link_icon) : ?>
																	<?php \Elementor\Icons_Manager::render_icon($settings['more_link_icon'], ['aria-hidden' => 'true', 'class' => 'mwt-elementor-posts-more-link-icon']); ?>
																<?php endif; ?>
															</a>
														</div>
													<?php endif ?>
													<?php if ('yes' === $settings['show_more_link_icon']) { ?>
														<a href="<?php the_permalink(); ?>" class="mwt-elementor-posts-read-more"></a>
													<?php } ?>
												</div>
												<?php
												if ($settings['show_footer'] === 'yes') {
												?>
													<footer class="mwt-elementor-posts-entry-footer">
														<?php if (function_exists('flicker_post_like_count')) : ?>
															<span>
																<?php flicker_post_like_button(get_the_ID()); ?>
																<span class="mwt-elementor-posts-entry-footer-value"><?php flicker_post_like_count(get_the_ID()); ?></span>
															</span>
														<?php endif; ?>
														<?php if (comments_open()) : ?>
															<span>
																<a href="<?php comments_link(); ?>">
																	<i class="fa fa-comment" aria-hidden="true"></i>
																	<span class="mwt-elementor-posts-entry-footer-value"><?php comments_number("0", "1", "%"); ?></span>
																</a>
															</span>
														<?php else : ?>
															<span>
																<i class="fa fa-comment" aria-hidden="true"></i>
																<span><?php echo esc_html__('Closed', 'flicker'); ?></span>
															</span>
														<?php endif; ?>
														<?php if (function_exists('flicker_show_post_views_count')) : ?>
															<span>
																<i class="fa fa-eye" aria-hidden="true"></i>
																<span class="mwt-elementor-posts-entry-footer-value"><?php flicker_show_post_views_count(); ?></span>
															</span>
														<?php endif; ?>
													</footer>
												<?php } ?>
											</article>
											</div>
										<?php endwhile; ?>
										<?php wp_reset_postdata(); //reset the query 
										?>
										</div>
								</div>
								<?php
								//grid layout 
								if ($settings['posts_layout'] === 'grid') : ?>
							</div>
						</div>
			<?php endif; //end grid layout 
							}
						} ?>