<?php
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

/**
 * Elementor MWT Gallery Grid Widget.
 *
 * Elementor widget that inserts an embbedable content into the page, from any given URL.
 *
 * @since 1.0.0
 */

class Elementor_MWT_Gallery_Grid extends \Elementor\Widget_Base

{

	/**
	 * Retrieve Widget Dependent JS.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array JS script handles.
	 */
	public function get_script_depends()
	{
		return ['gallery-grid-script'];
	}

	/**
	 * Retrieve Widget Dependent CSS.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array CSS style handles.
	 */
	public function get_style_depends()
	{
		return ['gallery-grid-style'];
	}

	/**
	 * Get widget name.
	 *
	 * Retrieve list widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return string Widget name.
	 */
	public function get_name()
	{
		return 'gallery-grid';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve list widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return string Widget title.
	 */
	public function get_title()
	{
		return esc_html__('MWT Gallery Grid', 'mwt-addons-for-elementor');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the list widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array Widget categories.
	 */
	public function get_categories()
	{
		return ['mwt'];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve image grid widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon()
	{
		return 'eicon-gallery-grid';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @since 2.1.0
	 * @access public
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords()
	{
		return ['mwt', 'image', 'photo', 'visual', 'gallery', 'grid'];
	}

	/**
	 * Register list widget controls.
	 *
	 * Add input fields to allow the user to customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls()
	{

		$this->start_controls_section(
			'content_section',
			[
				'label' => esc_html__('Grid Items', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'item_layout',
			[
				'label'       => __('Skin', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::SELECT,
				'description' => __('Choose a layout style for the gallery item', 'mwt-addons-for-elementor'),
				'options'     => [
					'default' => __('Style 1', 'mwt-addons-for-elementor'),
					'style2'  => __('Style 2', 'mwt-addons-for-elementor'),
					'style3'  => __('Style 3', 'mwt-addons-for-elementor'),
					'style4'  => __('Style 4', 'mwt-addons-for-elementor'),
				],
				'default'     => 'default',
				'separator'   => 'after',
				'label_block' => true,
			]
		);

		/* Start repeater */

		$repeater = new \Elementor\Repeater();

		$repeater->add_control(
			'image',
			[
				'label' => esc_html__('Choose Image', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::MEDIA,
				'dynamic' => [
					'active' => true,
				],
				'default' => [
					'url' => \Elementor\Utils::get_placeholder_image_src(),
				],
			]
		);

		$repeater->add_control(
			'item_title',
			[
				'label' => esc_html__('Title', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::TEXT,
				'placeholder' => esc_html__('Title', 'mwt-addons-for-elementor'),
				'default' => esc_html__('Title', 'mwt-addons-for-elementor'),
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
			]
		);

		$repeater->add_control(
			'item_title_link',
			[
				'label'       => __('Title Link', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::URL,
				'placeholder' => 'https://example.com/',
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
			]
		);

		$repeater->add_control(
			'show_link',
			[
				'label' => esc_html__('Show links icon over thumbnail', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'description' => esc_html__('Works only when layout set to \'Style 1\' and \'Title Link\' is not empty', 'mwt-addons-for-elementor'),
				'label_off' => esc_html__('No', 'mwt-addons-for-elementor'),
				'label_on' => esc_html__('Yes', 'mwt-addons-for-elementor'),
				'return_value' => 'yes',
				'default' => 'no',
				'condition' => [
					'gallery_lightbox_whole!' => 'yes',
					'gallery_link_whole!' => 'yes',
				],
			]
		);

		$repeater->add_control(
			'item_category',
			[
				'label'       => __('Category', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::TEXT,
				'placeholder' => esc_html__('Category', 'mwt-addons-for-elementor'),
				'description' => __('To assign for multiple categories, separate by a comma \',\'', 'mwt-addons-for-elementor'),
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
			]
		);

		$repeater->add_control(
			'item_description',
			[
				'label'       => __('Description', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::TEXTAREA,
				'description' => esc_html__('Works only when layout set to \'Style 1\'', 'mwt-addons-for-elementor'),
				'placeholder' => esc_html__('Description', 'mwt-addons-for-elementor'),
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
			]
		);

		$repeater->add_control(
			'gallery_link_whole',
			[
				'label' => esc_html__('Whole Image Link', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'description' => esc_html__('Works only when layout set to \'Style 1\' or \'Style 4\' and \'Title Link\' is not empty', 'mwt-addons-for-elementor'),
				'label_off' => esc_html__('No', 'mwt-addons-for-elementor'),
				'label_on' => esc_html__('Yes', 'mwt-addons-for-elementor'),
				'return_value' => 'yes',
				'default' => 'no',
				'condition' => [
					'gallery_lightbox_whole!' => 'yes',
				],
			]
		);

		$repeater->add_control(
			'gallery_lightbox_whole',
			[
				'label' => esc_html__('Whole Image Lightbox', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'description' => esc_html__('Works only when layout set to \'Style 1\' and Lightbox is active', 'mwt-addons-for-elementor'),
				'label_off' => esc_html__('No', 'mwt-addons-for-elementor'),
				'label_on' => esc_html__('Yes', 'mwt-addons-for-elementor'),
				'return_value' => 'yes',
				'default' => 'no',
				'condition' => [
					'gallery_link_whole!' => 'yes',
				],
			]
		);

		/* End repeater */

		$this->add_control(
			'gallery-list',
			[
				'label' => esc_html__('Items', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::REPEATER,
				'fields' => $repeater->get_controls(),
				'default' => [
					[
						'image' => '',
						'item_title' => esc_html__('Title 1', 'mwt-addons-for-elementor'),
						'item_title_link' => '',
						'show_link' => '',
						'item_category' => esc_html__('Category 1', 'mwt-addons-for-elementor'),
						'item_description' => '',
						'gallery_link_whole' => '',
						'gallery_lightbox_whole' => '',
					],
					[
						'image' => '',
						'item_title' => esc_html__('Title 2', 'mwt-addons-for-elementor'),
						'item_title_link' => '',
						'show_link' => '',
						'item_category' => esc_html__('Category 2', 'mwt-addons-for-elementor'),
						'item_description' => '',
						'gallery_link_whole' => '',
						'gallery_lightbox_whole' => '',
					],
					[
						'image' => '',
						'item_title' => esc_html__('Title 3', 'mwt-addons-for-elementor'),
						'item_title_link' => '',
						'show_link' => '',
						'item_category' => esc_html__('Category 3', 'mwt-addons-for-elementor'),
						'item_description' => '',
						'gallery_link_whole' => '',
						'gallery_lightbox_whole' => '',
					],
				],
				'title_field' => '{{{ item_title }}}',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Image_Size::get_type(),
			[
				'name' => 'thumbnail',
				'separator' => 'none',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'gallery_cats',
			[
				'label' => __('Categories', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'gallery_filter',
			[
				'label'   => __('Filter Tabs', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
			]
		);

		$this->add_control(
			'gallery_first_cat_label',
			[
				'label'     => __('First Category Label', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::TEXT,
				'default'   => __('All', 'mwt-addons-for-elementor'),
				'dynamic' => [
					'active' => true,
				],
				'condition'   => [
					'gallery_filter' => 'yes',
				],
			]
		);

		$category_repeater = new \Elementor\REPEATER();

		$category_repeater->add_control(
			'gallery_cat',
			[
				'label'   => __('Category', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::TEXT,
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
			]
		);

		$this->add_control(
			'gallery_cats_content',
			[
				'label'       => __('Categories', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::REPEATER,
				'fields'      => $category_repeater->get_controls(),
				'default'     => [
					[
						'gallery_cat' => esc_html__('Category 1', 'mwt-addons-for-elementor'),
					],
					[
						'gallery_cat' => esc_html__('Category 2', 'mwt-addons-for-elementor'),
					],
					[
						'gallery_cat' => esc_html__('Category 3', 'mwt-addons-for-elementor'),
					],
				],
				'title_field' => '{{{ gallery_cat }}}',
				'condition'   => [
					'gallery_filter' => 'yes',
				],
			]
		);

		$this->add_responsive_control(
			'gallery_filters_align',
			[
				'label'     => __('Alignment', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::CHOOSE,
				'options'   => [
					'left' => [
						'title' => __('Left', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-left',
					],
					'center'     => [
						'title' => __('Center', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-center',
					],
					'right'   => [
						'title' => __('Right', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-right',
					],
				],
				'default'   => 'center',
				'selectors' => [
					'{{WRAPPER}} .filters' => 'text-align: {{VALUE}}',
				],
				'condition'   => [
					'gallery_filter' => 'yes',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'gallery_lightbox_section',
			[
				'label' => __('Lightbox', 'mwt-addons-for-elementor'),
				'condition' => [
					'item_layout' => ['default', 'style2'],
				],
			]
		);

		$this->add_control(
			'open_lightbox',
			[
				'label' => esc_html__('Lightbox', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'description' => sprintf(
					esc_html__('Manage your site’s lightbox settings in the %1$sLightbox panel%2$s.', 'mwt-addons-for-elementor'),
					'<a href="javascript: $e.run( \'panel/global/open\' ).then( () => $e.route( \'panel/global/settings-lightbox\' ) )">',
					'</a>'
				),
				'default' => 'no',
				'options' => [
					'no' => esc_html__('No', 'elementor'),
					'yes' => esc_html__('Yes', 'elementor'),
				],
			]
		);

		$this->add_control(
			'gallery_lightbox_icon',
			[
				'label'     => __('Lightbox Icon', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::ICONS,
				'default'   => [
					'library' => 'fa-solid',
					'value'   => 'fas fa-search',
				],
				'condition' => [
					'open_lightbox' => 'yes',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'grid_settings',
			[
				'label' => __('Grid Settings', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_responsive_control(
			'items_per_row',
			[
				'label' => esc_html__('Items per row', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'description' => esc_html__('How many items should be shown per row?', 'mwt-addons-for-elementor'),
				'min'  => 1,
				'max'  => 4,
				'step' => 1,
				'default' => 3,
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'layout_gap',
			[
				'label' => esc_html__('Gap', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'solid',
				'options' => [
					'30' => esc_html__('Default', 'mwt-addons-for-elementor'),
					'0' => esc_html__('None', 'mwt-addons-for-elementor'),
					'1'  => esc_html__('1px', 'mwt-addons-for-elementor'),
					'2' => esc_html__('2px', 'mwt-addons-for-elementor'),
					'3' => esc_html__('3px', 'mwt-addons-for-elementor'),
					'4' => esc_html__('4px', 'mwt-addons-for-elementor'),
					'5' => esc_html__('5px', 'mwt-addons-for-elementor'),
					'10' => esc_html__('10px', 'mwt-addons-for-elementor'),
					'15' => esc_html__('15px', 'mwt-addons-for-elementor'),
					'20' => esc_html__('20px', 'mwt-addons-for-elementor'),
					'30' => esc_html__('30px', 'mwt-addons-for-elementor'),
					'40' => esc_html__('40px', 'mwt-addons-for-elementor'),
					'50' => esc_html__('50px', 'mwt-addons-for-elementor'),
					'60' => esc_html__('60px', 'mwt-addons-for-elementor'),
				],
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'horizontal_order',
			[
				'label' => esc_html__('Horizontal Order', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => esc_html__('Yes', 'mwt-addons-for-elementor'),
				'label_off' => esc_html__('No', 'mwt-addons-for-elementor'),
				'return_value' => 'yes',
				'default' => 'no',
			]
		);

		$this->add_control(
			'gallery_links_icon',
			[
				'label'     => __('Links Icon', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::ICONS,
				'default' => [
					'library' => 'fa-solid',
					'value'   => 'fas fa-link',
				],
				'condition' => [
					'item_layout' => 'default',
				],
			]
		);

		$this->end_controls_section();

		// style
		$this->start_controls_section(
			'style_grid_item',
			[
				'label' => esc_html__('Item', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'grid_item_bg_color',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-grid .grid-item .grid-item-wrap' => 'background-color: {{VALUE}};',
				],
				'condition' => [
					'item_layout' => 'default',
				],
			]
		);

		$this->add_responsive_control(
			'grid_item_align',
			[
				'label'     => __('Alignment', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::CHOOSE,
				'options'   => [
					'left' => [
						'title' => __('Left', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-left',
					],
					'center'     => [
						'title' => __('Center', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-center',
					],
					'right'   => [
						'title' => __('Right', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-right',
					],
				],
				'default'   => 'Left',
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-grid .grid-item .mwt-elementor-gallery-grid-content-wrap' => 'text-align: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'grid_item_border',
				'selector' => '{{WRAPPER}} .mwt-gallery-grid .grid-item .grid-item-wrap',
			]
		);

		$this->add_control(
			'grid_item_border_radius',
			[
				'label'      => esc_html__('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-gallery-grid .grid-item .grid-item-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition' => [
					'item_layout!' => 'style3'
				]
			]
		);

		$this->add_responsive_control(
			'grid_item_padding',
			[
				'label'      => __('Content Padding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-gallery-grid .grid-item .mwt-elementor-gallery-grid-content-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'grid_item_margin',
			[
				'label'      => __('Content Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-gallery-grid .grid-item .mwt-elementor-gallery-grid-content-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_image',
			[
				'label' => esc_html__('Image', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'image_hover_overlay_color',
			[
				'label' => esc_html__('Hover Overlay Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-grid .grid-item:hover .mwt-elementor-gallery-grid-links:before' => 'background-color: {{VALUE}};',
					'{{WRAPPER}} .mwt-gallery-grid .grid-item:hover .mwt-elementor-gallery-grid-image-wrap .links-wrap' => 'background-color: {{VALUE}};',
					'{{WRAPPER}} .mwt-gallery-grid .grid-item:hover .mwt-elementor-gallery-grid-image-wrap .mwt-elementor-gallery-grid-lightbox::before' => 'background-color: {{VALUE}};',
					'{{WRAPPER}} .mwt-gallery-grid .grid-item.style-2:hover .mwt-elementor-gallery-grid-content-outer-wrap' => 'background-color: {{VALUE}};',
					'{{WRAPPER}} .mwt-gallery-grid .grid-item.style-4 .mwt-elementor-gallery-grid-media-links:before' => 'background-color: {{VALUE}};',
				],
				'condition' => [
					'item_layout!' => 'style3'
				]
			]
		);

		$this->add_control(
			'image_border_radius',
			[
				'label'      => esc_html__('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-gallery-grid .grid-item .mwt-elementor-gallery-grid-image-wrap img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					'{{WRAPPER}} .mwt-gallery-grid .grid-item .mwt-elementor-gallery-grid-image-wrap .mwt-elementor-gallery-grid-lightbox::before' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					'{{WRAPPER}} .mwt-gallery-grid .grid-item.style-2 .mwt-elementor-gallery-grid-content-outer-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'image_margin',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-gallery-grid .grid-item .mwt-elementor-gallery-grid-image-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition' => [
					'item_layout' => 'default',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_title_section',
			[
				'label' => esc_html__('Title', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'title_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-gallery-grid-title' => 'color: {{VALUE}};',
					'{{WRAPPER}} a.mwt-elementor-gallery-grid-title' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'title_color_active',
			[
				'label' => esc_html__('Color Hover', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} a.mwt-elementor-gallery-grid-title:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'title_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-elementor-gallery-grid-title, {{WRAPPER}} .mwt-elementor-gallery-grid-title > a',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Text_Shadow::get_type(),
			[
				'label' => esc_html__('Text Shadow', 'mwt-addons-for-elementor'),
				'name' => 'title_shadow',
				'selector' => '{{WRAPPER}} .mwt-elementor-gallery-grid-title',
			]
		);

		$this->add_responsive_control(
			'title_bottom_spacing',
			[
				'label'      => __('Bottom Spacing', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-gallery-grid-title'  => 'margin-bottom: {{SIZE}}{{UNIT}}',
				],
				'condition' => [
					'item_layout' => ['default', 'style4'],
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_categories_section',
			[
				'label' => esc_html__('Categories', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'item_layout!' => ['style2', 'style3', 'style4'],
				]
			]
		);

		$this->add_control(
			'cats_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-gallery-grid-categories' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'cats_сomma_color',
			[
				'label' => esc_html__('Сomma Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-gallery-grid-categories .mwt-elementor-gallery-grid-categories-comma' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'categories_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-elementor-gallery-grid-categories',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Text_Shadow::get_type(),
			[
				'label' => esc_html__('Text Shadow', 'mwt-addons-for-elementor'),
				'name' => 'categories_shadow',
				'selector' => '{{WRAPPER}} .mwt-elementor-gallery-grid-categories',
			]
		);

		$this->add_responsive_control(
			'cats_bottom_spacing',
			[
				'label'      => __('Bottom Spacing', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-gallery-grid-categories'  => 'margin-bottom: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_description_section',
			[
				'label' => esc_html__('Description', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'item_layout' => ['default', 'style4'],
				],
			]
		);

		$this->add_control(
			'description_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-gallery-grid-description' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'description_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-elementor-gallery-grid-description',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'gallery_icons_style',
			[
				'label' => __('Icons', 'mwt-addons-for-elementor'),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'item_layout' => ['default', 'style2'],
				],
			]
		);

		$this->add_responsive_control(
			'gallery_icons_size',
			[
				'label'       => __('Size', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::SLIDER,
				'size_units'  => ['px', 'em'],
				'range'       => [
					'px' => [
						'min' => 1,
						'max' => 48,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 16,
				],
				'label_block' => true,
				'selectors'   => [
					'{{WRAPPER}} .links-inner-wrap i' => 'font-size: {{SIZE}}{{UNIT}}',
					'{{WRAPPER}} .links-inner-wrap svg' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->start_controls_tabs('gallery_icons_style_tabs');

		$this->start_controls_tab(
			'gallery_icons_style_normal',
			[
				'label' => __('Normal', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'gallery_icons_style_color',
			[
				'label'     => __('Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-gallery-grid-lightbox-icon i, {{WRAPPER}} .mwt-elementor-gallery-grid-links-icon i' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-elementor-gallery-grid-lightbox-icon svg, {{WRAPPER}} .mwt-elementor-gallery-grid-links-icon svg' => 'fill: {{VALUE}}; color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'gallery_icons_style_background',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-gallery-grid-lightbox-icon span, {{WRAPPER}} .mwt-elementor-gallery-grid-links-icon span' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'gallery_icons_style_border',
				'selector' => '{{WRAPPER}} .mwt-elementor-gallery-grid-lightbox-icon span, {{WRAPPER}} .mwt-elementor-gallery-grid-links-icon span',
			]
		);

		$this->add_control(
			'gallery_icons_style_border_radius',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-gallery-grid-lightbox-icon span, {{WRAPPER}} .mwt-elementor-gallery-grid-links-icon span' => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'label'    => __('Shadow', 'mwt-addons-for-elementor'),
				'name'     => 'gallery_icons_style_shadow',
				'selector' => '{{WRAPPER}} .mwt-elementor-gallery-grid-lightbox-icon span, {{WRAPPER}} .mwt-elementor-gallery-grid-links-icon span',
			]
		);

		$this->add_responsive_control(
			'gallery_icons_style_margin',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-gallery-grid-lightbox-icon span, {{WRAPPER}} .mwt-elementor-gallery-grid-links-icon span' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'gallery_icons_style_padding',
			[
				'label'      => __('Padding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-gallery-grid-lightbox-icon span, {{WRAPPER}} .mwt-elementor-gallery-grid-links-icon span' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'gallery_icons_style_hover',
			[
				'label' => __('Hover', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'gallery_icons_style_color_hover',
			[
				'label'     => __('Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-gallery-grid-lightbox-icon:hover i, {{WRAPPER}} .mwt-elementor-gallery-grid-links-icon:hover i' => 'color: {{VALUE}}',
					'{{WRAPPER}} .mwt-elementor-gallery-grid-lightbox-icon:hover svg, {{WRAPPER}} .mwt-elementor-gallery-grid-links-icon:hover svg' => 'fill: {{VALUE}}; color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'gallery_icons_style_background_hover',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-gallery-grid-lightbox-icon:hover span, {{WRAPPER}} .mwt-elementor-gallery-grid-links-icon:hover span' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'gallery_icons_style_border_hover',
				'selector' => '{{WRAPPER}} .mwt-elementor-gallery-grid-lightbox-icon:hover span, {{WRAPPER}} .mwt-elementor-gallery-grid-links-icon:hover span',
			]
		);

		$this->add_control(
			'gallery_icons_style_border_radius_hover',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-gallery-grid-lightbox-icon:hover span, {{WRAPPER}} .mwt-elementor-gallery-grid-links-icon:hover span' => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'label'    => __('Shadow', 'mwt-addons-for-elementor'),
				'name'     => 'gallery_icons_style_shadow_hover',
				'selector' => '{{WRAPPER}} {{WRAPPER}} .mwt-elementor-gallery-grid-lightbox-icon:hover span, {{WRAPPER}} .mwt-elementor-gallery-grid-links-icon:hover span',
			]
		);

		$this->add_responsive_control(
			'gallery_icons_style_margin_hover',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-gallery-grid-lightbox-icon:hover span, {{WRAPPER}} .mwt-elementor-gallery-grid-links-icon:hover span' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'gallery_icons_style_padding_hover',
			[
				'label'      => __('Padding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-gallery-grid-lightbox-icon:hover span, {{WRAPPER}} .mwt-elementor-gallery-grid-links-icon:hover span' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->end_controls_section();

		$this->start_controls_section(
			'gallery_filter_style',
			[
				'label'     => __('Filter', 'mwt-addons-for-elementor'),
				'tab'       => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'gallery_filter' => 'yes',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'gallery_filter_typo',
				'selector' => '{{WRAPPER}} .mwt-gallery-grid .filters a.category',
			]
		);

		$this->start_controls_tabs('gallery_filters');

		$this->start_controls_tab(
			'gallery_filters_normal',
			[
				'label' => __('Normal', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'gallery_filter_color',
			[
				'label'     => __('Text Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-grid .filters a.category' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'gallery_background_color',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-grid .filters a.category' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'gallery_filter_border',
				'selector' => '{{WRAPPER}} .mwt-gallery-grid .filters a.category',
			]
		);

		$this->add_control(
			'gallery_filter_border_radius',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-gallery-grid .filters a.category'  => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);


		$this->end_controls_tab();

		$this->start_controls_tab(
			'gallery_filters_hover',
			[
				'label' => __('Hover', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'gallery_filter_hover_color',
			[
				'label'     => __('Text Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-grid .filters a:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'gallery_background_hover_color',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-grid .filters a:hover' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'gallery_filter_border_hover',
				'selector' => '{{WRAPPER}} .mwt-gallery-grid .filters a.category:hover',
			]
		);

		$this->add_control(
			'gallery_filter_border_radius_hover',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-gallery-grid .filters a.category:hover'  => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'gallery_filters_active',
			[
				'label' => __('Active', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'gallery_filter_active_color',
			[
				'label'     => __('Text Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-grid .filters a.active' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'gallery_background_active_color',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-grid .filters a.active' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'gallery_filter_border_active',
				'selector' => '{{WRAPPER}} .mwt-gallery-grid .filters a.active',
			]
		);

		$this->add_control(
			'gallery_filter_border_radius_active',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-gallery-grid .filters a.active'  => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'gallery_filter_shadow',
				'selector' => '{{WRAPPER}} .mwt-gallery-grid .filters a.category',
			]
		);

		$this->add_responsive_control(
			'gallery_filter_margin',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-gallery-grid .filters a.category' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'gallery_filter_padding',
			[
				'label'      => __('Padding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-gallery-grid .filters a.category' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'gallery_filters_bottom_spacing',
			[
				'label'      => __('Filters Bottom Spacing', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-gallery-grid .filters'  => 'margin-bottom: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Filter Cats
	 *
	 * Formats Category to be inserted in class attribute.
	 *
	 * @since 2.1.0
	 * @access public
	 *
	 * @param string $string category slug.
	 *
	 * @return string $cat_filtered slug filtered.
	 */
	public function filter_cats($string)
	{

		$cat_filtered = trim($string);

		$cat_filtered = mb_strtolower($cat_filtered);

		if (strpos($cat_filtered, 'class') || strpos($cat_filtered, 'src')) {
			$cat_filtered = substr($cat_filtered, strpos($cat_filtered, '"') + 1);
			$cat_filtered = strtok($cat_filtered, '"');
			$cat_filtered = preg_replace('/[http:.]/', '', $cat_filtered);
			$cat_filtered = str_replace('/', '', $cat_filtered);
		}

		$cat_filtered = str_replace(', ', ',', $cat_filtered);
		$cat_filtered = preg_replace('/[\s_&@!#%]/', '-', $cat_filtered);
		$cat_filtered = str_replace(',', ' ', $cat_filtered);

		return $cat_filtered;
	}

	public function get_link_url($item)
	{
		return [
			'url' => wp_get_attachment_url($item['image']['id']),
		];
	}

	/**
	 * Render list widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render()
	{
		$settings = $this->get_settings_for_display();

		$unique_id = uniqid();

		if (empty($settings['gallery-list'])) {
			return;
		} ?>
		<div class="mwt-gallery-grid">
			<?php
			$filter = $settings['gallery_filter']; ?>
			<?php
			if ('yes' === $filter) :
			?>
				<div class="filters grid_filters-<?php echo esc_attr($unique_id); ?>">
					<a href="#" class="category selected active" data-filter="*">
						<?php echo wp_kses_post($settings['gallery_first_cat_label']); ?>
					</a>
					<?php
					foreach ($settings['gallery_cats_content'] as $index => $category) {
						if (!empty($category['gallery_cat'])) {
							$cat_filtered = $this->filter_cats($category['gallery_cat']); ?>
							<a href="#" class="category" data-filter="<?php echo esc_attr($cat_filtered) ?>">
								<?php echo wp_kses_post($category['gallery_cat']); ?>
							</a>
					<?php
						}
					}
					?>
				</div>
			<?php
			endif;
			?>
			<?php
			$style4_layout = ($settings['item_layout'] === 'style4') ? 'layout-style-4' : '';
			$horizontal_order = ($settings['horizontal_order'] === 'yes') ? true : false;
			?>
			<div class="layout-cols layout-cols-<?php echo esc_attr($settings['items_per_row']); ?> layout-gap-<?php echo esc_attr($settings['layout_gap']) . ' ' . $style4_layout ?>">
				<div class="grid-columns-wrapper">
					<div id="widget_portfolio_grid_<?php echo esc_attr($unique_id); ?>" class="masonry" data-filters=".grid_filters-<?php echo esc_attr($unique_id); ?>" data-order="<?php echo esc_attr($horizontal_order); ?>">
						<div class="grid-sizer"></div>
						<?php
						foreach ($settings['gallery-list'] as $index => $item) {
							$item_layout = $settings['item_layout'];
							switch ($item_layout) {
								case 'default': ?>
									<div class="grid-item <?php echo esc_attr($this->filter_cats($item['item_category'])) ?>">
										<div class="grid-item-wrap">
											<div class="mwt-elementor-gallery-grid-image-wrap">
												<?php
												$image_url = \Elementor\Group_Control_Image_Size::get_attachment_image_src($item['image']['id'], 'thumbnail', $settings);
												if (!$image_url && isset($item['image']['url'])) {
													$image_url = $item['image']['url'];
												}
												if (!empty($item['item_title'])) {
													$image_alt = $item['item_title'];
												} else {
													$image_alt = 'This is a title';
												}
												$link = $this->get_link_url($item);
												if ($link) {
													$link_key = 'link_' . $index;

													$this->add_lightbox_data_attributes($link_key, $item['image']['id'], $settings['open_lightbox'], $this->get_id());

													$this->add_link_attributes($link_key, $link);
												}
												$gallery_lightbox_icon = isset($settings['gallery_lightbox_icon']);
												$gallery_links_icon = isset($settings['gallery_links_icon']);

												if ('yes' !== $settings['gallery-list'][$index]['gallery_link_whole'] && 'yes' !== $settings['gallery-list'][$index]['gallery_lightbox_whole'] && ('yes' === $settings['open_lightbox'] || 'yes' === $settings['gallery-list'][$index]['show_link'])) { ?>
													<div class="links-wrap">
														<div class="links-inner-wrap">
															<?php
															if ('yes' === $settings['open_lightbox'] && !empty($gallery_lightbox_icon)) { ?>
																<a class="mwt-elementor-gallery-grid-lightbox-icon" <?php echo wp_kses_post($this->get_render_attribute_string($link_key)) ?>>
																	<span><?php \Elementor\Icons_Manager::render_icon($settings['gallery_lightbox_icon'], ['aria-hidden' => 'true', 'class' => 'icon-lightbox']) ?></span>
																</a>
															<?php } ?>
															<?php
															if ('yes' === $settings['gallery-list'][$index]['show_link'] && !empty($gallery_links_icon)) { ?>
																<a class="mwt-elementor-gallery-grid-links-icon" href="<?php echo esc_url($settings['gallery-list'][$index]['item_title_link']['url']) ?>">
																	<span><?php \Elementor\Icons_Manager::render_icon($settings['gallery_links_icon'], ['aria-hidden' => 'true', 'class' => 'icon-links']) ?></span>
																</a>
															<?php } ?>
														</div>
													</div>
												<?php
												}
												if ('yes' === $settings['gallery-list'][$index]['gallery_link_whole']) { ?>
													<a class="mwt-elementor-gallery-grid-links" href="<?php echo esc_url($settings['gallery-list'][$index]['item_title_link']['url']) ?>">
													<?php }
												if ('yes' === $settings['open_lightbox'] && 'yes' === $settings['gallery-list'][$index]['gallery_lightbox_whole']) { ?>
														<a class="mwt-elementor-gallery-grid-lightbox" <?php echo wp_kses_post($this->get_render_attribute_string($link_key)) ?>>
														<?php }
													$image_html = '<img class="slide-image" src="' . esc_attr($image_url) . '" alt="' . esc_attr($image_alt) . '" />';
													echo wp_kses_post($image_html);
													if ('yes' === $settings['gallery-list'][$index]['gallery_link_whole'] || ('yes' === $settings['open_lightbox'] && 'yes' === $settings['gallery-list'][$index]['gallery_lightbox_whole'])) { ?>
														</a>
													<?php } ?>
											</div>
											<?php
											$title = $settings['gallery-list'][$index]['item_title'];
											$title_link = $settings['gallery-list'][$index]['item_title_link']['url'];
											$category = $settings['gallery-list'][$index]['item_category'];
											$category_array = explode(",", $category);
											$description = $settings['gallery-list'][$index]['item_description'];
											?>
											<div class="mwt-elementor-gallery-grid-content-wrap">
												<?php
												if (!empty($category_array)) { ?>
													<div class="mwt-elementor-gallery-grid-categories">
														<?php foreach ($category_array as $index => $category_aray_item) { ?>
															<span class="mwt-elementor-gallery-grid-categories-item"><?php echo $category_aray_item; ?></span><span class="mwt-elementor-gallery-grid-categories-comma">,</span>
														<?php } ?>
													</div>
												<?php }
												if (!empty($title_link)) { ?>
													<a href="<?php echo esc_attr($title_link) ?>" class="mwt-elementor-gallery-grid-title">
														<?php echo $title; ?>
													</a>
												<?php } else { ?>
													<div class="mwt-elementor-gallery-grid-title">
														<?php echo $title; ?>
													</div>
												<?php }
												?>
												<div class="mwt-elementor-gallery-grid-description">
													<?php echo $description; ?>
												</div>
											</div>
										</div>
									</div>
								<?php break;

								case 'style2': ?>
									<div class="grid-item style-2 <?php echo esc_attr($this->filter_cats($item['item_category'])) ?>">
										<div class="grid-item-wrap">
											<div class="mwt-elementor-gallery-grid-image-wrap">
												<?php
												$image_url = \Elementor\Group_Control_Image_Size::get_attachment_image_src($item['image']['id'], 'thumbnail', $settings);
												if (!$image_url && isset($item['image']['url'])) {
													$image_url = $item['image']['url'];
												}
												if (!empty($item['item_title'])) {
													$image_alt = $item['item_title'];
												} else {
													$image_alt = 'This is a title';
												}
												$link = $this->get_link_url($item);
												if ($link) {
													$link_key = 'link_' . $index;

													$this->add_lightbox_data_attributes($link_key, $item['image']['id'], $settings['open_lightbox'], $this->get_id());

													$this->add_link_attributes($link_key, $link);
												}
												$gallery_lightbox_icon = isset($settings['gallery_lightbox_icon']);
												if ('yes' === $settings['open_lightbox']) { ?>
													<div class="links-wrap">
														<div class="links-inner-wrap">
															<?php
															if ('yes' === $settings['open_lightbox'] && !empty($gallery_lightbox_icon)) { ?>
																<a class="mwt-elementor-gallery-grid-lightbox-icon" <?php echo wp_kses_post($this->get_render_attribute_string($link_key)) ?>>
																	<span><?php \Elementor\Icons_Manager::render_icon($settings['gallery_lightbox_icon'], ['aria-hidden' => 'true', 'class' => 'icon-lightbox']) ?></span>
																</a>
															<?php } ?>
														</div>
													</div>
												<?php
												}
												$image_html = '<img class="slide-image" src="' . esc_attr($image_url) . '" alt="' . esc_attr($image_alt) . '" />';
												echo wp_kses_post($image_html);
												?>
											</div>
											<?php
											$title = $settings['gallery-list'][$index]['item_title'];
											$title_link = $settings['gallery-list'][$index]['item_title_link']['url'];
											?>
											<div class="mwt-elementor-gallery-grid-content-outer-wrap">
												<div class="mwt-elementor-gallery-grid-content-wrap">
													<?php
													if (!empty($title_link)) { ?>
														<a href="<?php echo esc_attr($title_link) ?>" class="mwt-elementor-gallery-grid-title">
															<?php echo $title; ?>
														</a>
													<?php } else { ?>
														<div class="mwt-elementor-gallery-grid-title">
															<?php echo $title; ?>
														</div>
													<?php } ?>
												</div>
											</div>
										</div>
									</div>

								<?php break;

								case 'style3': ?>
									<div class="grid-item style-3 <?php echo esc_attr($this->filter_cats($item['item_category'])) ?>">
										<div class="grid-item-wrap">
											<div class="mwt-elementor-gallery-grid-image-wrap">
												<?php
												$image_url = \Elementor\Group_Control_Image_Size::get_attachment_image_src($item['image']['id'], 'thumbnail', $settings);
												if (!$image_url && isset($item['image']['url'])) {
													$image_url = $item['image']['url'];
												}
												if (!empty($item['item_title'])) {
													$image_alt = $item['item_title'];
												} else {
													$image_alt = 'This is a title';
												}
												$image_html = '<img class="slide-image" src="' . esc_attr($image_url) . '" alt="' . esc_attr($image_alt) . '" />';
												echo wp_kses_post($image_html);
												?>
											</div>
											<?php
											$title = $settings['gallery-list'][$index]['item_title'];
											$title_link = $settings['gallery-list'][$index]['item_title_link']['url'];
											?>
											<div class="mwt-elementor-gallery-grid-content-outer-wrap">
												<div class="mwt-elementor-gallery-grid-content-wrap">
													<?php
													if (!empty($title_link)) { ?>
														<a href="<?php echo esc_attr($title_link) ?>" class="mwt-elementor-gallery-grid-title">
															<?php echo $title; ?><i class="fa fa-angle-right"></i>
														</a>
													<?php } else { ?>
														<div class="mwt-elementor-gallery-grid-title">
															<?php echo $title; ?>
														</div>
													<?php } ?>
												</div>
											</div>
										</div>
									</div>

								<?php break;
								case 'style4': ?>
									<div class="grid-item style-4 <?php echo esc_attr($this->filter_cats($item['item_category'])) ?>">
										<div class="grid-item-wrap">
											<div class="mwt-elementor-gallery-grid-image-wrap">
												<?php
												$image_url = \Elementor\Group_Control_Image_Size::get_attachment_image_src($item['image']['id'], 'thumbnail', $settings);
												if (!$image_url && isset($item['image']['url'])) {
													$image_url = $item['image']['url'];
												}
												if (!empty($item['item_title'])) {
													$image_alt = $item['item_title'];
												} else {
													$image_alt = 'This is a title';
												}
												$image_html = '<img class="slide-image" src="' . esc_attr($image_url) . '" alt="' . esc_attr($image_alt) . '" />';
												echo wp_kses_post($image_html);
												if ('yes' === $settings['gallery-list'][$index]['gallery_link_whole']) { ?>
													<div class="mwt-elementor-gallery-grid-media-links">
														<a class="mwt-elementor-gallery-grid-links" href="<?php echo esc_url($settings['gallery-list'][$index]['item_title_link']['url']) ?>">
														</a>
													</div>
												<?php }
												?>
											</div>
											<?php
											$title = $settings['gallery-list'][$index]['item_title'];
											$title_link = $settings['gallery-list'][$index]['item_title_link']['url'];
											$description = $settings['gallery-list'][$index]['item_description'];
											?>
											<div class="mwt-elementor-gallery-grid-content-outer-wrap">
												<div class="mwt-elementor-gallery-grid-content-wrap">
													<?php
													if (!empty($title_link)) { ?>
														<a href="<?php echo esc_attr($title_link) ?>" class="mwt-elementor-gallery-grid-title">
															<?php echo $title; ?>
														</a>
													<?php } else { ?>
														<div class="mwt-elementor-gallery-grid-title">
															<?php echo $title; ?>
														</div>
													<?php } ?>
													<div class="mwt-elementor-gallery-grid-description">
														<?php echo $description; ?>
													</div>
												</div>
											</div>
										</div>
									</div>

							<?php break;
							} ?>
						<?php } ?>
					</div>
				</div>
			</div>
		</div>
<?php }
}
