<?php
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

/**
 * Elementor MWT Gallery Carousel Widget.
 *
 * Elementor widget that inserts an embbedable content into the page, from any given URL.
 *
 * @since 1.0.0
 */

class Elementor_MWT_Gallery_Carousel extends \Elementor\Widget_Base

{

	/**
	 * Retrieve Widget Dependent JS.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array JS script handles.
	 */
	public function get_script_depends()
	{
		return ['gallery-carousel-script'];
	}

	/**
	 * Retrieve Widget Dependent CSS.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array CSS style handles.
	 */
	public function get_style_depends()
	{
		return ['owl-carousel-style'];
		return ['gallery-carousel-style'];
	}

	/**
	 * Get widget name.
	 *
	 * Retrieve list widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return string Widget name.
	 */
	public function get_name()
	{
		return 'gallery-carousel';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve list widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return string Widget title.
	 */
	public function get_title()
	{
		return esc_html__('MWT Gallery Carousel', 'mwt-addons-for-elementor');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the list widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array Widget categories.
	 */
	public function get_categories()
	{
		return ['mwt'];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve image carousel widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon()
	{
		return 'eicon-slider-push';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @since 2.1.0
	 * @access public
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords()
	{
		return ['mwt', 'image', 'photo', 'visual', 'gallery', 'carousel', 'slider'];
	}

	/**
	 * Register list widget controls.
	 *
	 * Add input fields to allow the user to customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls()
	{

		$this->start_controls_section(
			'content_section',
			[
				'label' => esc_html__('Carousel Items', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'items_layout',
			[
				'label' => esc_html__('Item Layout', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'label_block' => true,
				'default' => 'item-regular',
				'options' => [
					'item-extended' => esc_html__('Image with title and excerpt', 'mwt-addons-for-elementor'),
					'item-image' => esc_html__('Image only', 'mwt-addons-for-elementor'),
				],
				'frontend_available' => true,
			]
		);

		/* Start repeater */

		$repeater = new \Elementor\Repeater();

		$repeater->add_control(
			'image',
			[
				'label' => esc_html__('Choose Image', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::MEDIA,
				'dynamic' => [
					'active' => true,
				],
				'default' => [
					'url' => \Elementor\Utils::get_placeholder_image_src(),
				],
			]
		);

		$repeater->add_control(
			'item_title',
			[
				'label' => esc_html__('Title', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::TEXT,
				'placeholder' => esc_html__('Title', 'mwt-addons-for-elementor'),
				'default' => esc_html__('Title', 'mwt-addons-for-elementor'),
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
			]
		);

		$repeater->add_control(
			'item_title_link',
			[
				'label'       => __('Title Link', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::URL,
				'placeholder' => 'https://example.com/',
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
			]
		);

		$repeater->add_control(
			'show_link',
			[
				'label' => esc_html__('Show links icon over thumbnail', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'description' => esc_html__('Works only when layout set to \'Image only\' and \'Title Link\' is not empty', 'mwt-addons-for-elementor'),
				'label_off' => esc_html__('No', 'mwt-addons-for-elementor'),
				'label_on' => esc_html__('Yes', 'mwt-addons-for-elementor'),
				'return_value' => 'yes',
				'default' => 'no',
			]
		);

		$repeater->add_control(
			'item_category',
			[
				'label'       => __('Category', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::TEXT,
				'placeholder' => esc_html__('Category', 'mwt-addons-for-elementor'),
				'description' => __('To assign for multiple categories, separate by a comma \',\'', 'mwt-addons-for-elementor'),
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
			]
		);

		$repeater->add_control(
			'item_description',
			[
				'label'       => __('Description', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__('Description', 'mwt-addons-for-elementor'),
				'description' => __('For Layout \'Image with title and excerpt\'', 'mwt-addons-for-elementor'),
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
			]
		);

		/* End repeater */

		$this->add_control(
			'gallery-list',
			[
				'label' => esc_html__('Items', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::REPEATER,
				'fields' => $repeater->get_controls(),
				'default' => [
					[
						'image' => '',
						'item_title' => esc_html__('Title 1', 'mwt-addons-for-elementor'),
						'item_title_link' => '',
						'show_link' => '',
						'item_category' => esc_html__('Category 1', 'mwt-addons-for-elementor'),
						'item_description' => '',
					],
					[
						'image' => '',
						'item_title' => esc_html__('Title 2', 'mwt-addons-for-elementor'),
						'item_title_link' => '',
						'show_link' => '',
						'item_category' => esc_html__('Category 2', 'mwt-addons-for-elementor'),
						'item_description' => '',
					],
					[
						'image' => '',
						'item_title' => esc_html__('Title 3', 'mwt-addons-for-elementor'),
						'item_title_link' => '',
						'show_link' => '',
						'item_category' => esc_html__('Category 3', 'mwt-addons-for-elementor'),
						'item_description' => '',
					],
				],
				'title_field' => '{{{ item_title }}}',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Image_Size::get_type(),
			[
				'name' => 'thumbnail',
				'separator' => 'none',
			]
		);

		$this->add_control(
			'gallery_links_icon',
			[
				'label'     => __('Links Icon', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::ICONS,
				'default' => [
					'library' => 'fa-solid',
					'value'   => 'fas fa-link',
				],
				'condition' => [
					'items_layout' => 'item-image',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'gallery_cats',
			[
				'label' => __('Categories', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'gallery_filter',
			[
				'label'   => __('Filter Tabs', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
			]
		);

		$this->add_control(
			'gallery_first_cat_label',
			[
				'label'     => __('First Category Label', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::TEXT,
				'default'   => __('All', 'mwt-addons-for-elementor'),
				'dynamic' => [
					'active' => true,
				],
				'condition'   => [
					'gallery_filter' => 'yes',
				],
			]
		);

		$category_repeater = new \Elementor\REPEATER();

		$category_repeater->add_control(
			'gallery_cat',
			[
				'label'   => __('Category', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::TEXT,
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
			]
		);

		$this->add_control(
			'gallery_cats_content',
			[
				'label'       => __('Categories', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::REPEATER,
				'fields'      => $category_repeater->get_controls(),
				'default'     => [
					[
						'gallery_cat' => esc_html__('Category 1', 'mwt-addons-for-elementor'),
					],
					[
						'gallery_cat' => esc_html__('Category 2', 'mwt-addons-for-elementor'),
					],
					[
						'gallery_cat' => esc_html__('Category 3', 'mwt-addons-for-elementor'),
					],
				],
				'title_field' => '{{{ gallery_cat }}}',
				'condition'   => [
					'gallery_filter' => 'yes',
				],
			]
		);

		$this->add_responsive_control(
			'gallery_filters_align',
			[
				'label'     => __('Alignment', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::CHOOSE,
				'options'   => [
					'left' => [
						'title' => __('Left', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-left',
					],
					'center'     => [
						'title' => __('Center', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-center',
					],
					'right'   => [
						'title' => __('Right', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-right',
					],
				],
				'default'   => 'center',
				'selectors' => [
					'{{WRAPPER}} .filters' => 'text-align: {{VALUE}}',
				],
				'condition'   => [
					'gallery_filter' => 'yes',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'gallery_lightbox_section',
			[
				'label' => __('Lightbox', 'mwt-addons-for-elementor'),
				'condition' => [
					'items_layout' => 'item-image',
				],
			]
		);

		$this->add_control(
			'open_lightbox',
			[
				'label' => esc_html__('Lightbox', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'description' => sprintf(
					esc_html__('Manage your site’s lightbox settings in the %1$sLightbox panel%2$s.', 'mwt-addons-for-elementor'),
					'<a href="javascript: $e.run( \'panel/global/open\' ).then( () => $e.route( \'panel/global/settings-lightbox\' ) )">',
					'</a>'
				),
				'default' => 'no',
				'options' => [
					'no' => esc_html__('No', 'elementor'),
					'yes' => esc_html__('Yes', 'elementor'),
				],
			]
		);

		$this->add_control(
			'gallery_lightbox_icon',
			[
				'label'     => __('Lightbox Icon', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::ICONS,
				'default'   => [
					'library' => 'fa-solid',
					'value'   => 'fas fa-search',
				],
				'condition' => [
					'open_lightbox' => 'yes',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'carousel_settings',
			[
				'label' => __('Carousel Settings', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'data_loop',
			[
				'label'   => __('Loop', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_autoplay',
			[
				'label'   => __('Autoplay', 'mwt-addons-for-elementor'),
				'description' => esc_html__('Slide will start automatically', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
				'frontend_available' => true,
			]
		);


		$this->add_control(
			'data_autoplaytimeout',
			[
				'label' => esc_html__('Autoplay Speed (ms)', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'default' => 3000,
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_autoplay_hover_pause',
			[
				'label'   => __('Pause on Hover', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
				'frontend_available' => true,
			]
		);

		//Slides to Scroll

		$this->add_control(
			'data_responsive_lg',
			[
				'label' => esc_html__('Desktop Slides', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'min' => 1,
				'max' => 8,
				'step' => 1,
				'default' => 3,
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_responsive_sm',
			[
				'label' => esc_html__('Tabs Slides', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'min' => 1,
				'max' => 8,
				'step' => 1,
				'default' => 2,
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_responsive_xs',
			[
				'label' => esc_html__('Mobile Slides', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'min' => 1,
				'max' => 8,
				'step' => 1,
				'default' => 1,
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_margin',
			[
				'label' => esc_html__('Item Gap (px)', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'min' => 0,
				'max' => 60,
				'step' => 1,
				'default' => 15,
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_nav',
			[
				'label'   => __('Navigation', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
				'frontend_available' => true,
			]
		);

		$this->end_controls_section();

		//style
		$this->start_controls_section(
			'style_item_section',
			[
				'label' => esc_html__('Items', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
				'condition'   => [
					'items_layout!' => 'item-image',
				],
			]
		);

		$this->add_responsive_control(
			'gallery_item_align',
			[
				'label'     => __('Alignment', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::CHOOSE,
				'options'   => [
					'left' => [
						'title' => __('Left', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-left',
					],
					'center'     => [
						'title' => __('Center', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-center',
					],
					'right'   => [
						'title' => __('Right', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-right',
					],
				],
				'default'   => 'Left',
				'selectors' => [
					'{{WRAPPER}} .owl-carousel.gallery-slider .mwt-elementor-gallery-carousel-content-wrap' => 'text-align: {{VALUE}}',
				],
			]
		);

		$this->add_responsive_control(
			'item_padding',
			[
				'label'      => __('Padding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .owl-carousel.gallery-slider .mwt-elementor-gallery-carousel-content-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_image',
			[
				'label' => esc_html__('Image', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'image_hover_overlay_color',
			[
				'label' => esc_html__('Hover Overlay Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .owl-carousel.gallery-slider .owl-carousel-item:hover::before' => 'background-color: {{VALUE}};',
					'{{WRAPPER}} .owl-carousel.layout-item-image .owl-carousel-item:hover::before' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_title_section',
			[
				'label' => esc_html__('Title', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'items_layout!' => 'item-image',
				],
			]
		);

		$this->add_control(
			'title_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-gallery-carousel-title' => 'color: {{VALUE}};',
					'{{WRAPPER}} a.mwt-elementor-gallery-carousel-title' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'title_color_active',
			[
				'label' => esc_html__('Color Hover', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} a.mwt-elementor-gallery-carousel-title:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'title_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-elementor-gallery-carousel-title, {{WRAPPER}} .mwt-elementor-gallery-carousel-title > a',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Text_Shadow::get_type(),
			[
				'label' => esc_html__('Text Shadow', 'mwt-addons-for-elementor'),
				'name' => 'title_shadow',
				'selector' => '{{WRAPPER}} .mwt-elementor-gallery-carousel-title',
			]
		);

		$this->add_responsive_control(
			'title_bottom_spacing',
			[
				'label'      => __('Bottom Spacing', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-gallery-carousel-title'  => 'margin-bottom: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_description_section',
			[
				'label' => esc_html__('Description', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'items_layout!' => 'item-image',
				],
			]
		);

		$this->add_control(
			'description_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-gallery-carousel-description' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'description_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-elementor-gallery-carousel-description',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Text_Shadow::get_type(),
			[
				'label' => esc_html__('Text Shadow', 'mwt-addons-for-elementor'),
				'name' => 'description_shadow',
				'selector' => '{{WRAPPER}} .mwt-elementor-gallery-carousel-description',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'gallery_icons_style',
			[
				'label' => __('Icons', 'mwt-addons-for-elementor'),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'items_layout' => 'item-image',
				],
			]
		);

		$this->add_responsive_control(
			'gallery_icons_size',
			[
				'label'       => __('Size', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::SLIDER,
				'size_units'  => ['px', 'em'],
				'range'       => [
					'px' => [
						'min' => 1,
						'max' => 48,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 16,
				],
				'label_block' => true,
				'selectors'   => [
					'{{WRAPPER}} .gallery-slider .links-inner-wrap i' => 'font-size: {{SIZE}}{{UNIT}}',
					'{{WRAPPER}} .gallery-slider .links-inner-wrap svg' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->start_controls_tabs('gallery_icons_style_tabs');

		$this->start_controls_tab(
			'gallery_icons_style_normal',
			[
				'label' => __('Normal', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'gallery_icons_style_color',
			[
				'label'     => __('Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-gallery-carousel-lightbox-icon i, {{WRAPPER}} .mwt-elementor-gallery-carousel-links-icon i' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-elementor-gallery-carousel-lightbox-icon svg, {{WRAPPER}} .mwt-elementor-gallery-carousel-links-icon svg' => 'fill: {{VALUE}}; color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'gallery_icons_style_background',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-gallery-carousel-lightbox-icon span, {{WRAPPER}} .mwt-elementor-gallery-carousel-links-icon span' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'gallery_icons_style_border',
				'selector' => '{{WRAPPER}} .mwt-elementor-gallery-carousel-lightbox-icon span, {{WRAPPER}} .mwt-elementor-gallery-carousel-links-icon span',
			]
		);

		$this->add_control(
			'gallery_icons_style_border_radius',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-gallery-carousel-lightbox-icon span, {{WRAPPER}} .mwt-elementor-gallery-carousel-links-icon span' => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'label'    => __('Shadow', 'mwt-addons-for-elementor'),
				'name'     => 'gallery_icons_style_shadow',
				'selector' => '{{WRAPPER}} .mwt-elementor-gallery-carousel-lightbox-icon span, {{WRAPPER}} .mwt-elementor-gallery-carousel-links-icon span',
			]
		);

		$this->add_responsive_control(
			'gallery_icons_style_margin',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-gallery-carousel-lightbox-icon span, {{WRAPPER}} .mwt-elementor-gallery-carousel-links-icon span' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'gallery_icons_style_padding',
			[
				'label'      => __('Padding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-gallery-carousel-lightbox-icon span, {{WRAPPER}} .mwt-elementor-gallery-carousel-links-icon span' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'gallery_icons_style_hover',
			[
				'label' => __('Hover', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'gallery_icons_style_color_hover',
			[
				'label'     => __('Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-gallery-carousel-lightbox-icon:hover i, {{WRAPPER}} .mwt-elementor-gallery-carousel-links-icon:hover i' => 'color: {{VALUE}}',
					'{{WRAPPER}} .mwt-elementor-gallery-carousel-lightbox-icon:hover svg, {{WRAPPER}} .mwt-elementor-gallery-carousel-links-icon:hover svg' => 'fill: {{VALUE}}; color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'gallery_icons_style_background_hover',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-gallery-carousel-lightbox-icon:hover span, {{WRAPPER}} .mwt-elementor-gallery-carousel-links-icon:hover span' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'gallery_icons_style_border_hover',
				'selector' => '{{WRAPPER}} .mwt-elementor-gallery-carousel-lightbox-icon:hover span, {{WRAPPER}} .mwt-elementor-gallery-carousel-links-icon:hover span',
			]
		);

		$this->add_control(
			'gallery_icons_style_border_radius_hover',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-gallery-carousel-lightbox-icon:hover span, {{WRAPPER}} .mwt-elementor-gallery-carousel-links-icon:hover span' => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'label'    => __('Shadow', 'mwt-addons-for-elementor'),
				'name'     => 'gallery_icons_style_shadow_hover',
				'selector' => '{{WRAPPER}} {{WRAPPER}} .mwt-elementor-gallery-carousel-lightbox-icon:hover span, {{WRAPPER}} .mwt-elementor-gallery-carousel-links-icon:hover span',
			]
		);

		$this->add_responsive_control(
			'gallery_icons_style_margin_hover',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-gallery-carousel-lightbox-icon:hover span, {{WRAPPER}} .mwt-elementor-gallery-carousel-links-icon:hover span' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'gallery_icons_style_padding_hover',
			[
				'label'      => __('Padding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-gallery-carousel-lightbox-icon:hover span, {{WRAPPER}} .mwt-elementor-gallery-carousel-links-icon:hover span' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->end_controls_section();

		$this->start_controls_section(
			'gallery_filter_style',
			[
				'label'     => __('Filter', 'mwt-addons-for-elementor'),
				'tab'       => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'gallery_filter' => 'yes',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'gallery_filter_typo',
				'selector' => '{{WRAPPER}} .gallery-carousel-container .filters a.category',
			]
		);

		$this->start_controls_tabs('gallery_filters');

		$this->start_controls_tab(
			'gallery_filters_normal',
			[
				'label' => __('Normal', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'gallery_filter_color',
			[
				'label'     => __('Text Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .gallery-carousel-container .filters a.category' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'gallery_background_color',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .gallery-carousel-container .filters a.category' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'gallery_filter_border',
				'selector' => '{{WRAPPER}} .gallery-carousel-container .filters a.category',
			]
		);

		$this->add_control(
			'gallery_filter_border_radius',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .gallery-carousel-container .filters a.category'  => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);


		$this->end_controls_tab();

		$this->start_controls_tab(
			'gallery_filters_hover',
			[
				'label' => __('Hover', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'gallery_filter_hover_color',
			[
				'label'     => __('Text Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .gallery-carousel-container .filters a:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'gallery_background_hover_color',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .gallery-carousel-container .filters a:hover' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'gallery_filter_border_hover',
				'selector' => '{{WRAPPER}} .gallery-carousel-container .filters a.category:hover',
			]
		);

		$this->add_control(
			'gallery_filter_border_radius_hover',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .gallery-carousel-container .filters a.category:hover'  => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'gallery_filters_active',
			[
				'label' => __('Active', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'gallery_filter_active_color',
			[
				'label'     => __('Text Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .gallery-carousel-container .filters a.active' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'gallery_background_active_color',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .gallery-carousel-container .filters a.active' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'gallery_filter_border_active',
				'selector' => '{{WRAPPER}} .gallery-carousel-container .filters a.active',
			]
		);

		$this->add_control(
			'gallery_filter_border_radius_active',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .gallery-carousel-container .filters a.active'  => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'gallery_filter_shadow',
				'selector' => '{{WRAPPER}} .gallery-carousel-container .filters a.category',
			]
		);

		$this->add_responsive_control(
			'gallery_filter_padding',
			[
				'label'      => __('Padding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .gallery-carousel-container .filters a.category' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'gallery_filter_margin',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .gallery-carousel-container .filters a.category' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'gallery_filters_bottom_spacing',
			[
				'label'      => __('Filters Bottom Spacing', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .gallery-carousel-container .filters'  => 'margin-bottom: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_navigation_section',
			[
				'label' => esc_html__('Navigation', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'data_nav' => 'yes',
				]
			]
		);

		$this->add_control(
			'nav_layout',
			[
				'label' => esc_html__('Navigation Layout', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'nav-layout-2',
				'options' => [
					'nav-layout-1' => esc_html__('Layout 1', 'mwt-addons-for-elementor'),
					'nav-layout-2' => esc_html__('Layout 2', 'mwt-addons-for-elementor'),
					'nav-layout-3' => esc_html__('Layout 3', 'mwt-addons-for-elementor'),
					'nav-layout-4' => esc_html__('Dots', 'mwt-addons-for-elementor'),
					'nav-layout-5' => esc_html__('Dots and Arrows (for Item Layout "Image only")', 'mwt-addons-for-elementor'),
				],
				'frontend_available' => true,
			]
		);

		$this->add_responsive_control(
			'nav_icon_size',
			[
				'label'      => __('Icon Size', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em'],
				'range'      => [
					'px' => [
						'min'  => 12,
						'max'  => 40,
						'step' => 1,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 20,
				],
				'selectors'  => [
					'{{WRAPPER}} .gallery-slider .owl-nav > button:after' => 'font-size: {{SIZE}}{{UNIT}}; width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'nav_layout!' => ['nav-layout-4', 'nav-layout-5'],
				],
			]
		);

		$this->add_control(
			'navigation_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .gallery-slider .owl-nav > button:after' => 'color: {{VALUE}}; background-color: {{VALUE}};',
					'{{WRAPPER}} .owl-carousel.layout-item-image.nav-layout-5 .owl-dots .owl-dot span' => 'background-color: {{VALUE}};',
					'{{WRAPPER}} .owl-carousel.layout-item-image.nav-layout-5 .owl-nav > button::after' => 'background-color: {{VALUE}};',
				],
				'condition' => [
					'nav_layout!' => 'nav-layout-4',
				],
			]
		);

		$this->add_control(
			'navigation_color_hover',
			[
				'label' => esc_html__('Color Hover', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .gallery-slider .owl-nav > button:hover:after' => 'color: {{VALUE}}; background-color: {{VALUE}};',
					'{{WRAPPER}} .owl-carousel.layout-item-image.nav-layout-5 .owl-dots .owl-dot.active span' => 'background-color: {{VALUE}};',
					'{{WRAPPER}} .owl-carousel.layout-item-image.nav-layout-5 .owl-nav > button:hover::after' => 'background-color: {{VALUE}};',
				],
				'condition' => [
					'nav_layout!' => 'nav-layout-4',
				],
			]
		);

		$this->add_responsive_control(
			'navigation_position',
			[
				'label'      => __('Offset', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%'],
				'default' => [
					'size' => -50,
				],
				'range' => [
					'px' => [
						'min' => -100,
						'max' => 100,
						'step' => 1,
					],
					'%' => [
						'min' => -100,
						'max' => 100,
						'step' => 1,
					],
				],
				'selectors'  => [
					'{{WRAPPER}} .owl-carousel.layout-item-image.nav-layout-5 .owl-nav' => 'right: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .owl-carousel.layout-item-image.nav-layout-5 .owl-dots' => 'right: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'nav_layout' => 'nav-layout-5',
				],
			]
		);

		$this->add_responsive_control(
			'navigation_arrows_position',
			[
				'label'      => __('Arrows Position', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%'],
				'default' => [
					'size' => 100,
				],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100,
						'step' => 1,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
						'step' => 1,
					],
				],
				'selectors'  => [
					'{{WRAPPER}} .owl-carousel.layout-item-image.nav-layout-5 .owl-nav' => 'height: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'nav_layout' => 'nav-layout-5',
				],
			]
		);

		$this->add_control(
			'navigation_bg_color',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .gallery-slider .owl-nav > button' => 'background-color: {{VALUE}};',
				],
				'condition' => [
					'nav_layout!' => ['nav-layout-4', 'nav-layout-5'],
				],
			]
		);

		$this->add_control(
			'navigation_bg_color_hover',
			[
				'label' => esc_html__('Background Color Hover', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .gallery-slider .owl-nav > button:hover' => 'background-color: {{VALUE}};',
				],
				'condition' => [
					'nav_layout!' => ['nav-layout-4', 'nav-layout-5'],
				],
			]
		);

		$this->add_control(
			'navigation_border_radius_1',
			[
				'label'      => esc_html__('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .gallery-slider .owl-nav > button.owl-prev' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					'{{WRAPPER}} .gallery-slider .owl-nav > button.owl-next' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition' => [
					'nav_layout' => 'nav-layout-1',
				],
			]
		);

		$this->add_responsive_control(
			'navigation_padding',
			[
				'label'      => esc_html__('Pdding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .gallery-slider .owl-nav > button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition' => [
					'nav_layout' => 'nav-layout-1',
				],
			]
		);

		$this->add_responsive_control(
			'navigation_margin',
			[
				'label'      => esc_html__('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .gallery-slider .owl-nav > button' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition' => [
					'nav_layout' => 'nav-layout-1',
				],
			]
		);

		$this->add_control(
			'navigation_border_radius_2',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .gallery-slider .owl-nav > button.owl-prev'  => 'border-top-right-radius: {{SIZE}}{{UNIT}}; border-bottom-right-radius: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .gallery-slider .owl-nav > button.owl-next'  => 'border-top-left-radius: {{SIZE}}{{UNIT}}; border-bottom-left-radius: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'nav_layout' => 'nav-layout-2',
				],
			]
		);

		$this->start_controls_tabs(
			'style_tabs'
		);

		$this->start_controls_tab(
			'style_normal_tab',
			[
				'label' => esc_html__('Normal', 'textdomain'),
				'condition' => [
					'nav_layout' => 'nav-layout-4',
				],
			]
		);

		$this->add_control(
			'dots_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .gallery-slider .owl-dots > .owl-dot span' => 'background-color: {{VALUE}};',
				],
				'condition' => [
					'nav_layout' => 'nav-layout-4',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'style_hover_tab',
			[
				'label' => esc_html__('Hover', 'textdomain'),
				'condition' => [
					'nav_layout' => 'nav-layout-4',
				],
			]
		);

		$this->add_control(
			'dots_color_hover',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .gallery-slider .owl-dots > .owl-dot span:hover' => 'background-color: {{VALUE}};',
				],
				'condition' => [
					'nav_layout' => 'nav-layout-4',
				],
			]
		);

		$this->add_control(
			'dots_border_color_hover',
			[
				'label' => esc_html__('Border Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .gallery-slider .owl-dots > .owl-dot span:hover' => 'border-color: {{VALUE}};',
				],
				'condition' => [
					'nav_layout' => 'nav-layout-4',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'style_active_tab',
			[
				'label' => esc_html__('Active', 'textdomain'),
				'condition' => [
					'nav_layout' => 'nav-layout-4',
				],
			]
		);

		$this->add_control(
			'dots_color_active',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .gallery-slider .owl-dots > .owl-dot.active span' => 'background-color: {{VALUE}};',
				],
				'condition' => [
					'nav_layout' => 'nav-layout-4',
				],
			]
		);

		$this->add_control(
			'dots_border_color_active',
			[
				'label' => esc_html__('Border Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .gallery-slider .owl-dots > .owl-dot.active span' => 'border-color: {{VALUE}};',
				],
				'condition' => [
					'nav_layout' => 'nav-layout-4',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_responsive_control(
			'dots_size',
			[
				'label'      => __('Size', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em'],
				'range'      => [
					'px' => [
						'min'  => 5,
						'max'  => 50,
						'step' => 1,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 10,
				],
				'selectors'  => [
					'{{WRAPPER}} .gallery-slider .owl-dots > .owl-dot span' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'nav_layout' => 'nav-layout-4',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'dots_border',
				'selector' => '{{WRAPPER}} .gallery-slider .owl-dots > .owl-dot span',
				'condition' => [
					'nav_layout' => 'nav-layout-4',
				],
			]
		);

		$this->add_control(
			'dots_border_radius',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .gallery-slider .owl-dots > .owl-dot span'  => 'border-radius: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'nav_layout' => 'nav-layout-4',
				],
			]
		);

		$this->add_responsive_control(
			'dots_margin',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .gallery-slider .owl-dots > .owl-dot span' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition' => [
					'nav_layout' => 'nav-layout-4',
				],
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Filter Cats
	 *
	 * Formats Category to be inserted in class attribute.
	 *
	 * @since 2.1.0
	 * @access public
	 *
	 * @param string $string category slug.
	 *
	 * @return string $cat_filtered slug filtered.
	 */
	public function filter_cats($string)
	{

		$cat_filtered = trim($string);

		$cat_filtered = mb_strtolower($cat_filtered);

		if (strpos($cat_filtered, 'class') || strpos($cat_filtered, 'src')) {
			$cat_filtered = substr($cat_filtered, strpos($cat_filtered, '"') + 1);
			$cat_filtered = strtok($cat_filtered, '"');
			$cat_filtered = preg_replace('/[http:.]/', '', $cat_filtered);
			$cat_filtered = str_replace('/', '', $cat_filtered);
		}

		$cat_filtered = str_replace(', ', ',', $cat_filtered);
		$cat_filtered = preg_replace('/[\s_&@!#%]/', '-', $cat_filtered);
		$cat_filtered = str_replace(',', ' ', $cat_filtered);

		return $cat_filtered;
	}

	public function get_link_url($item)
	{
		return [
			'url' => wp_get_attachment_url($item['image']['id']),
		];
	}

	/**
	 * Render list widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render()
	{
		$settings = $this->get_settings_for_display();

		$unique_id = uniqid();

		if (empty($settings['gallery-list'])) {
			return;
		} ?>
		<div class="gallery-carousel-container">
			<?php
			$filter = $settings['gallery_filter']; ?>
			<?php
			if ('yes' === $filter) :
			?>
				<div class="filters carousel_filters-<?php echo esc_attr($unique_id); ?>">
					<a href="#" class="category selected active" data-filter="*">
						<?php echo wp_kses_post($settings['gallery_first_cat_label']); ?>
					</a>
					<?php
					foreach ($settings['gallery_cats_content'] as $index => $category) {
						if (!empty($category['gallery_cat'])) {
							$cat_filtered = $this->filter_cats($category['gallery_cat']); ?>
							<a href="#" class="category" data-filter=".<?php echo esc_attr($cat_filtered) ?>">
								<?php echo wp_kses_post($category['gallery_cat']); ?>
							</a>
					<?php
						}
					}
					?>
				</div>
			<?php
			endif;
			?>
			<?php
			$data_loop = ($settings['data_loop'] === 'yes') ? 'true' : 'false';
			$data_autoplay = ($settings['data_autoplay'] === 'yes') ? 'true' : 'false';
			$data_center = ($settings['data_center'] === 'yes') ? 'true' : 'false';
			$data_nav = ($settings['data_nav'] === 'yes') ? 'true' : 'false';
			$data_dots = ($settings['nav_layout'] === 'nav-layout-5' || $settings['nav_layout'] === 'nav-layout-4') ? 'true' : 'false';
			$data_autoplay_hover_pause = ($settings['data_autoplay_hover_pause'] === 'yes') ? 'true' : 'false';

			$item_layout = $settings['items_layout'];
			switch ($item_layout) {
				case 'item-extended': ?>
					<div id="widget_portfolio_carousel_<?php echo esc_attr($unique_id); ?>" class="owl-carousel gallery-slider layout-item-extended2 <?php echo esc_attr($settings['nav_layout']); ?>" data-dots="<?php echo esc_attr($data_dots) ?>" data-loop="<?php echo esc_attr($data_loop) ?>" data-autoplay="<?php echo esc_attr($data_autoplay) ?>" data-autoplaytimeout="<?php echo esc_attr($settings['data_autoplaytimeout']) ?>" data-autoplayhoverpause="<?php echo esc_attr($data_autoplay_hover_pause) ?>" data-center="<?php echo esc_attr($data_center) ?>" data-nav="<?php echo esc_attr($data_nav) ?>" data-margin="<?php echo esc_attr($settings['data_margin']) ?>" data-responsive-lg="<?php echo esc_attr($settings['data_responsive_lg']) ?>" data-responsive-sm="<?php echo esc_attr($settings['data_responsive_sm']) ?>" data-responsive-xs="<?php echo esc_attr($settings['data_responsive_xs']) ?>" data-filters=".carousel_filters-<?php echo esc_attr($unique_id); ?>">
						<?php
						foreach ($settings['gallery-list'] as $index => $item) { ?>
							<div class="owl-carousel-item <?php echo esc_attr($this->filter_cats($item['item_category'])) ?>">
								<?php
								$image_url = \Elementor\Group_Control_Image_Size::get_attachment_image_src($item['image']['id'], 'thumbnail', $settings);
								if (!$image_url && isset($item['image']['url'])) {
									$image_url = $item['image']['url'];
								}
								if (!empty($item['item_title'])) {
									$image_alt = $item['item_title'];
								} else {
									$image_alt = 'This is a title';
								}
								$image_html = '<img class="slide-image" src="' . esc_attr($image_url) . '" alt="' . esc_attr($image_alt) . '" />';
								echo wp_kses_post($image_html);
								?>
								<?php
								$title = $settings['gallery-list'][$index]['item_title'];
								$title_link = $settings['gallery-list'][$index]['item_title_link']['url'];
								$description = $settings['gallery-list'][$index]['item_description'];
								?>
								<div class="mwt-elementor-gallery-carousel-content-wrap">
									<?php
									if (!empty($title_link)) { ?>
										<a href="<?php echo esc_attr($title_link) ?>" class="mwt-elementor-gallery-carousel-title">
											<?php echo $title; ?>
										</a>
									<?php } else { ?>
										<span class="mwt-elementor-gallery-carousel-title">
											<?php echo $title; ?>
										</span>
									<?php }
									?>
									<span class="mwt-elementor-gallery-carousel-description">
										<?php echo $description; ?>
									</span>
								</div>
							</div>
						<?php } ?>
					</div>
				<?php break;

				case 'item-image':
				?>
					<div id="widget_portfolio_carousel_<?php echo esc_attr($unique_id); ?>" class="owl-carousel gallery-slider layout-item-image <?php echo esc_attr($settings['nav_layout']); ?>" data-loop="<?php echo esc_attr($data_loop) ?>" data-autoplay="<?php echo esc_attr($data_autoplay) ?>" data-autoplaytimeout="<?php echo esc_attr($settings['data_autoplaytimeout']) ?>" data-autoplayhoverpause="<?php echo esc_attr($data_autoplay_hover_pause) ?>" data-center="<?php echo esc_attr($data_center) ?>" data-nav="<?php echo esc_attr($data_nav) ?>" data-dots="<?php echo esc_attr($data_dots) ?>" data-margin="<?php echo esc_attr($settings['data_margin']) ?>" data-responsive-lg="<?php echo esc_attr($settings['data_responsive_lg']) ?>" data-responsive-sm="<?php echo esc_attr($settings['data_responsive_sm']) ?>" data-responsive-xs="<?php echo esc_attr($settings['data_responsive_xs']) ?>" data-filters=".carousel_filters-<?php echo esc_attr($unique_id); ?>">
						<?php
						foreach ($settings['gallery-list'] as $index => $item) { ?>
							<div class="owl-carousel-item <?php echo esc_attr($this->filter_cats($item['item_category'])) ?>">
								<?php
								$title_link = $settings['gallery-list'][$index]['item_title_link']['url'];
								$image_url = \Elementor\Group_Control_Image_Size::get_attachment_image_src($item['image']['id'], 'thumbnail', $settings);
								if (!$image_url && isset($item['image']['url'])) {
									$image_url = $item['image']['url'];
								}
								if (!empty($item['item_title'])) {
									$image_alt = $item['item_title'];
								} else {
									$image_alt = 'This is a title';
								}
								$link = $this->get_link_url($item);

								if ($link) {
									$link_key = 'link_' . $index;

									$this->add_lightbox_data_attributes($link_key, $item['image']['id'], $settings['open_lightbox'], $this->get_id());

									$this->add_link_attributes($link_key, $link);
								}
								$gallery_lightbox_icon = isset($settings['gallery_lightbox_icon']);
								$gallery_links_icon = isset($settings['gallery_links_icon']);

								if ('yes' === $settings['open_lightbox'] && !empty($gallery_lightbox_icon) || 'yes' === $settings['gallery-list'][$index]['show_link']) { ?>
									<div class="links-wrap">
										<div class="links-inner-wrap">
											<?php if ('yes' === $settings['open_lightbox'] && !empty($gallery_lightbox_icon)) { ?>
												<a class="mwt-elementor-gallery-carousel-lightbox-icon" <?php echo wp_kses_post($this->get_render_attribute_string($link_key)) ?>>
													<span><?php \Elementor\Icons_Manager::render_icon($settings['gallery_lightbox_icon'], ['aria-hidden' => 'true', 'class' => 'icon-lightbox']) ?></span>
												</a>
											<?php } ?>
											<?php
											if ('yes' === $settings['gallery-list'][$index]['show_link'] && !empty($gallery_links_icon)) { ?>
												<a class="mwt-elementor-gallery-carousel-links-icon" href="<?php echo esc_url($settings['gallery-list'][$index]['item_title_link']['url']) ?>">
													<span><?php \Elementor\Icons_Manager::render_icon($settings['gallery_links_icon'], ['aria-hidden' => 'true', 'class' => 'icon-links']) ?></span>
												</a>
											<?php } ?>
										</div>
									</div>
								<?php
								}
								$image_html = '<img class="slide-image" src="' . esc_attr($image_url) . '" alt="' . esc_attr($image_alt) . '" />';
								if (!empty($title_link)) { ?>
									<a href="<?php echo esc_attr($title_link) ?>" class="mwt-elementor-gallery-carousel-link">
									<?php }
								echo wp_kses_post($image_html);
								if (!empty($title_link)) { ?>
									</a>
								<?php }
								?>
							</div>
						<?php } ?>
					</div>
			<?php break;
			} ?>
		</div>
<?php }
}
