<?php
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

/**
 * Elementor MWT Events Carousel Widget.
 *
 * Elementor widget that inserts an embbedable content into the page, from any given URL.
 *
 * @since 1.0.0
 */

class Elementor_MWT_Events_Carousel extends \Elementor\Widget_Base

{

	/**
	 * Retrieve Widget Dependent JS.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array JS script handles.
	 */
	public function get_script_depends()
	{
		return ['events-carousel-script'];
	}

	/**
	 * Retrieve Widget Dependent CSS.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array CSS style handles.
	 */
	public function get_style_depends()
	{
		return ['owl-carousel-style'];
		return ['events-carousel-style'];
	}

	/**
	 * Get widget name.
	 *
	 * Retrieve list widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return string Widget name.
	 */
	public function get_name()
	{
		return 'events-carousel';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve list widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return string Widget title.
	 */
	public function get_title()
	{
		return esc_html__('MWT Events Carousel', 'mwt-addons-for-elementor');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the list widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array Widget categories.
	 */
	public function get_categories()
	{
		return ['mwt'];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve image carousel widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon()
	{
		return 'eicon-slider-push';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @since 2.1.0
	 * @access public
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords()
	{
		return ['mwt', 'visual', 'events', 'carousel', 'slider'];
	}

	/**
	 * Register list widget controls.
	 *
	 * Add input fields to allow the user to customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls()
	{
		$this->start_controls_section(
			'general_settings',
			[
				'label' => __('General', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'number_of_events',
			[
				'label'       => __('Events Per Page', 'mwt-addons-for-elementor'),
				'description' => __('Set the number of per page', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::NUMBER,
				'min'         => 1,
				'default'     => 3,
			]
		);

		$this->add_control(
			'events_start_date',
			[
				'label' => esc_html__('Start Date', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::DATE_TIME,
				'description' => __('Set the start of the date range', 'mwt-addons-for-elementor'),
				'default'     => '',
			]
		);

		$this->add_control(
			'events_end_date',
			[
				'label' => esc_html__('End Date', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::DATE_TIME,
				'description' => __('Set the end of the date range', 'mwt-addons-for-elementor'),
				'default'     => '',
			]
		);

		$this->add_control(
			'show_featured',
			[
				'label'     => __('Featured Events', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::SWITCHER,
				'description' => __('Show only featured events', 'mwt-addons-for-elementor'),
				'default'   => 'no',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'content_section',
			[
				'label' => esc_html__('Items', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'show_meta',
			[
				'label'     => __('Show Meta', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::SWITCHER,
				'default'   => 'yes',
			]
		);

		$this->add_control(
			'show_excerpt',
			[
				'label'     => __('Show Excerpt', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::SWITCHER,
				'default'   => 'yes',
			]
		);

		$this->add_control(
			'show_buy_link',
			[
				'label'     => __('Buy Link', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::SWITCHER,
				'default'   => 'yes',
			]
		);

		$this->add_control(
			'buy_link_text',
			[
				'label'     => __('Buy Link Text', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::TEXT,
				'default'   => __('Get ticket', 'mwt-addons-for-elementor'),
				'condition'   => [
					'show_buy_link' => 'yes',
				],
				'ai' => [
					'active' => false,
				],
			]
		);

		$this->add_control(
			'show_more_link',
			[
				'label'     => __('Read More Link', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::SWITCHER,
				'default'   => 'yes',
			]
		);

		$this->add_control(
			'more_link_text',
			[
				'label'     => __('Read More Link Text', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::TEXT,
				'default'   => __('Read more', 'mwt-addons-for-elementor'),
				'condition'   => [
					'show_more_link' => 'yes',
				],
				'ai' => [
					'active' => false,
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'carousel_settings',
			[
				'label' => __('Carousel Settings', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'data_loop',
			[
				'label'   => __('Loop', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_autoplay',
			[
				'label'   => __('Autoplay', 'mwt-addons-for-elementor'),
				'description' => esc_html__('Slide will start automatically', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
				'frontend_available' => true,
			]
		);


		$this->add_control(
			'data_autoplaytimeout',
			[
				'label' => esc_html__('Autoplay Speed (ms)', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'default' => 5000,
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_autoplay_hover_pause',
			[
				'label'   => __('Pause on Hover', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_center',
			[
				'label'   => __('Centered Carousel?', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'no',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_responsive_lg',
			[
				'label' => esc_html__('Desktop Slides', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'default' => 3,
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_responsive_sm',
			[
				'label' => esc_html__('Tabs Slides', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'default' => 2,
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_responsive_xs',
			[
				'label' => esc_html__('Mobile Slides', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'default' => 1,
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_margin',
			[
				'label' => esc_html__('Item Gap (px)', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'default' => 15,
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_nav',
			[
				'label'   => __('Navigation', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
				'frontend_available' => true,
			]
		);

		$this->end_controls_section();

		// Style
		$this->start_controls_section(
			'style_item_section',
			[
				'label' => esc_html__('Items', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'events_items_border',
				'selector' => '{{WRAPPER}} .mwt-elementor-events-article',
			]
		);

		$this->add_responsive_control(
			'events_items_radius',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-events-article' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
				],
			]
		);

		$this->add_control(
			'events_items_image_opacity',
			[
				'label'      => __('Image Opacity', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'range' => [
					[
						'min' => 0.1,
						'max' => 1,
						'step' => 0.01,
					]
				],
				'default' => [
					'size' => 1,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-events-item-media img'  => 'opacity: {{SIZE}}',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_price_section',
			[
				'label' => esc_html__('Price', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Price Typography', 'mwt-addons-for-elementor'),
				'name' => 'events_price_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-elementor-events-ticket-price',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Text Typography', 'mwt-addons-for-elementor'),
				'name' => 'events_price_text_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-elementor-events-top-text, .mwt-elementor-events-bottom-text',
			]
		);

		$this->add_control(
			'events_price_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-events-entry-meta-corner>span' => 'color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_content_section',
			[
				'label' => esc_html__('Content', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'content_padding',
			[
				'label'      => __('Padding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-events-item-content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'content_align',
			[
				'label'     => __('Alignment', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::CHOOSE,
				'options'   => [
					'left' => [
						'title' => __('Left', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-left',
					],
					'center'     => [
						'title' => __('Center', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-center',
					],
					'right'   => [
						'title' => __('Right', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-right',
					],
				],
				'default'   => 'center',
				'dynamic' => [
					'active' => true,
				],
				'frontend_available' => true,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-events-item-content' => 'text-align: {{VALUE}}',
					'{{WRAPPER}} .mwt-elementor-events-event-buttons' => 'justify-content: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'events_title',
			[
				'label' => esc_html__('Title', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'events_title_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-elementor-events-event-title',
			]
		);

		$this->add_control(
			'events_title_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-events-event-title' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'events_title_color_hover',
			[
				'label' => esc_html__('Color Hover', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-events-event-title a:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'events_title_margin',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-events-event-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'events_meta',
			[
				'label' => esc_html__('Meta', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'show_meta' => 'yes',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'events_meta_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-elementor-events-meta',
				'condition' => [
					'show_meta' => 'yes',
				],
			]
		);

		$this->add_control(
			'events_meta_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-events-meta' => 'color: {{VALUE}};',
				],
				'condition' => [
					'show_meta' => 'yes',
				],
			]
		);

		$this->add_responsive_control(
			'events_meta_margin',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-events-meta' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition' => [
					'show_meta' => 'yes',
				],
			]
		);

		$this->add_control(
			'events_excerpt',
			[
				'label' => esc_html__('Excerpt', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'show_excerpt' => 'yes',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'events_excerpt_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-elementor-events-excerpt',
				'condition' => [
					'show_excerpt' => 'yes',
				],
			]
		);

		$this->add_control(
			'events_excerpt_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-events-excerpt' => 'color: {{VALUE}};',
				],
				'condition' => [
					'show_excerpt' => 'yes',
				],
			]
		);

		$this->add_responsive_control(
			'events_excerpt_margin',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-events-excerpt' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition' => [
					'show_excerpt' => 'yes',
				],
			]
		);

		$this->add_responsive_control(
			'events_excerpt_height',
			[
				'label'      => __('Excerpt Height', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%', 'em', 'rem'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 180,
						'step' => 1,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 80,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-events-excerpt'  => 'height: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'show_excerpt' => 'yes',
				],
			]
		);

		$this->add_control(
			'events_buy_button',
			[
				'label' => esc_html__('Buy Tickets Button', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'show_buy_link' => 'yes',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'events_buy_button_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-elementor-events-buy-tickets-button',
				'condition' => [
					'show_buy_link' => 'yes',
				],
			]
		);

		$this->start_controls_tabs(
			'events_buy_button_style_tabs'
		);

		$this->start_controls_tab(
			'events_buy_button_style_normal_tab',
			[
				'label' => esc_html__('Normal', 'mwt-addons-for-elementor'),
				'condition' => [
					'show_buy_link' => 'yes',
				],
			]
		);

		$this->add_control(
			'events_buy_button_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-events-buy-tickets-button' => 'color: {{VALUE}};',
				],
				'condition' => [
					'show_buy_link' => 'yes',
				],
			]
		);

		$this->add_control(
			'events_buy_button_background_color',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-events-buy-tickets-button' => 'background-color: {{VALUE}};',
				],
				'condition' => [
					'show_buy_link' => 'yes',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'events_buy_button_style_hover_tab',
			[
				'label' => esc_html__('Hover', 'mwt-addons-for-elementor'),
				'condition' => [
					'show_buy_link' => 'yes',
				],
			]
		);

		$this->add_control(
			'events_buy_button_color_hover',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-events-buy-tickets-button:hover' => 'color: {{VALUE}};',
				],
				'condition' => [
					'show_buy_link' => 'yes',
				],
			]
		);

		$this->add_control(
			'events_buy_button_background_color_hover',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-events-buy-tickets-button:hover' => 'background-color: {{VALUE}};',
				],
				'condition' => [
					'show_buy_link' => 'yes',
				],
			]
		);

		$this->add_control(
			'events_buy_button_border_color_hover',
			[
				'label' => esc_html__('Border Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-events-buy-tickets-button:hover' => 'border-color: {{VALUE}};',
				],
				'condition' => [
					'show_buy_link' => 'yes',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'events_buy_button_border',
				'selector' => '{{WRAPPER}} .mwt-elementor-events-buy-tickets-button',
				'condition' => [
					'show_buy_link' => 'yes',
				],
			]
		);

		$this->add_control(
			'events_buy_button_border_radius',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-events-buy-tickets-button'  => 'border-radius: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'show_buy_link' => 'yes',
				],
			]
		);

		$this->add_responsive_control(
			'events_buy_button_padding',
			[
				'label'      => __('Padding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-events-buy-tickets-button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition' => [
					'show_buy_link' => 'yes',
				],
			]
		);

		$this->add_control(
			'events_more_button',
			[
				'label' => esc_html__('Read More Button', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'show_more_link' => 'yes',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'events_more_button_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-elementor-events-button',
				'condition' => [
					'show_more_link' => 'yes',
				],
			]
		);

		$this->start_controls_tabs(
			'events_more_button_style_tabs'
		);

		$this->start_controls_tab(
			'events_more_button_style_normal_tab',
			[
				'label' => esc_html__('Normal', 'mwt-addons-for-elementor'),
				'condition' => [
					'show_more_link' => 'yes',
				],
			]
		);

		$this->add_control(
			'events_more_button_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-events-button' => 'color: {{VALUE}};',
				],
				'condition' => [
					'show_more_link' => 'yes',
				],
			]
		);

		$this->add_control(
			'events_more_button_background_color',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-events-button' => 'background-color: {{VALUE}};',
				],
				'condition' => [
					'show_more_link' => 'yes',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'events_more_button_style_hover_tab',
			[
				'label' => esc_html__('Hover', 'mwt-addons-for-elementor'),
				'condition' => [
					'show_more_link' => 'yes',
				],
			]
		);

		$this->add_control(
			'events_more_button_color_hover',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-events-button:hover' => 'color: {{VALUE}};',
				],
				'condition' => [
					'show_more_link' => 'yes',
				],
			]
		);

		$this->add_control(
			'events_more_button_background_color_hover',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-events-button:hover' => 'background-color: {{VALUE}};',
				],
				'condition' => [
					'show_more_link' => 'yes',
				],
			]
		);

		$this->add_control(
			'events_more_button_border_color_hover',
			[
				'label' => esc_html__('Border Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-events-button:hover' => 'border-color: {{VALUE}};',
				],
				'condition' => [
					'show_more_link' => 'yes',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'events_more_button_border',
				'selector' => '{{WRAPPER}} .mwt-elementor-events-button',
				'condition' => [
					'show_more_link' => 'yes',
				],
			]
		);

		$this->add_control(
			'events_more_button_border_radius',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-events-button'  => 'border-radius: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'show_more_link' => 'yes',
				],
			]
		);

		$this->add_responsive_control(
			'events_more_button_padding',
			[
				'label'      => __('Padding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-events-button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition' => [
					'show_more_link' => 'yes',
				],
			]
		);

		$this->add_responsive_control(
			'events_buttons_margin',
			[
				'label'      => __('Buttons Block Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-events-event-buttons' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'conditions' => [
					'relation' => 'or',
					'terms' => [
						[
							'name' => 'show_buy_link',
							'operator' => '===',
							'value' => 'yes',
						],
						[
							'name' => 'show_more_link',
							'operator' => '===',
							'value' => 'yes',
						],
					],
				],
			]
		);

		$this->add_responsive_control(
			'events_hover_block_height',
			[
				'label'      => __('Hover Block Max Height', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'separator' => 'before',
				'size_units' => ['px', '%', 'em', 'rem'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 290,
						'step' => 1,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 190,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-events-article:hover .mwt-elementor-events-entry-content'  => 'max-height: calc(110px + {{SIZE}}{{UNIT}});',
				],
				'conditions' => [
					'relation' => 'or',
					'terms' => [
						[
							'name' => 'show_excerpt',
							'operator' => '===',
							'value' => 'yes',
						],
						[
							'name' => 'show_buy_link',
							'operator' => '===',
							'value' => 'yes',
						],
						[
							'name' => 'show_more_link',
							'operator' => '===',
							'value' => 'yes',
						],
					],
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_events_nav_section',
			[
				'label' => esc_html__('Navigation', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'data_nav' => 'yes',
				],
			]
		);

		$this->add_responsive_control(
			'events_nav_icon_size',
			[
				'label'      => __('Icon Size', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em'],
				'range'      => [
					'px' => [
						'min'  => 12,
						'max'  => 40,
						'step' => 1,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 36,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-events-carousel .events-slider .owl-nav > button:after' => 'font-size: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .mwt-elementor-events-carousel .events-slider .owl-nav > button:after' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->start_controls_tabs(
			'style_events_nav_tabs'
		);

		$this->start_controls_tab(
			'style_events_nav_normal_tab',
			[
				'label' => esc_html__('Normal', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'events_nav_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-events-carousel .events-slider .owl-nav > button:after' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'events_nav_bg_color',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-events-carousel .events-slider .owl-nav > button' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'events_nav_border_color',
			[
				'label' => esc_html__('Border Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-events-carousel .events-slider .owl-nav > button' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'style_events_nav_hover_tab',
			[
				'label' => esc_html__('Hover', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'events_nav_color_hover',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-events-carousel .events-slider .owl-nav > button:hover:after' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'events_nav_bg_color_hover',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-events-carousel .events-slider .owl-nav > button:hover' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'events_nav_border_color_hover',
			[
				'label' => esc_html__('Border Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-events-carousel .events-slider .owl-nav > button:hover' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'events_nav_border',
				'selector' => '{{WRAPPER}} .mwt-elementor-events-carousel .events-slider .owl-nav > button',
			]
		);

		$this->add_responsive_control(
			'events_nav_radius',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-events-carousel .events-slider .owl-nav > button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
				],
			]
		);

		$this->add_responsive_control(
			'events_nav_padding',
			[
				'label'      => __('Padding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-events-carousel .events-slider .owl-nav > button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'events_nav_margin',
			[
				'label'      => esc_html__('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-events-carousel .events-slider .owl-nav > button' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'events_nav_offset',
			[
				'label'      => esc_html__('Offset', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%'],
				'range' => [
					'px' => [
						'min' => -1000,
						'max' => 1000,
						'step' => 1,
					],
					'%' => [
						'min' => -100,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 0,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-events-carousel .owl-theme .owl-nav'  => 'left: {{SIZE}}{{UNIT}}; right: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render list widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render()
	{
		$settings = $this->get_settings_for_display();

		$featured = ($settings['show_featured'] === 'yes') ? true : '';

		$mwt_events = tribe_get_events([
			'posts_per_page' => $settings['number_of_events'],
			'start_date'   => $settings['events_start_date'],
			'end_date'     => $settings['events_end_date'],
			'featured'       => $featured,
		]);

		if (!isset($mwt_events)) {
			return;
		}
?>
		<div class="owl-carousel-container mwt-elementor-events-carousel">
			<?php
			$data_loop = ($settings['data_loop'] === 'yes') ? 'true' : 'false';
			$data_autoplay = ($settings['data_autoplay'] === 'yes') ? 'true' : 'false';
			$data_center = ($settings['data_center'] === 'yes') ? 'true' : 'false';
			$data_nav = ($settings['data_nav'] === 'yes') ? 'true' : 'false';
			$data_autoplay_hover_pause = ($settings['data_autoplay_hover_pause'] === 'yes') ? 'true' : 'false';
			?>
			<div class="owl-carousel events-slider" data-loop="<?php echo esc_attr($data_loop) ?>" data-autoplay="<?php echo esc_attr($data_autoplay) ?>" data-autoplaytimeout="<?php echo esc_attr($settings['data_autoplaytimeout']) ?>" data-autoplayhoverpause="<?php echo esc_attr($data_autoplay_hover_pause) ?>" data-center="<?php echo esc_attr($data_center) ?>" data-nav="<?php echo esc_attr($data_nav) ?>" data-margin="<?php echo esc_attr($settings['data_margin']) ?>" data-responsive-lg="<?php echo esc_attr($settings['data_responsive_lg']) ?>" data-responsive-sm="<?php echo esc_attr($settings['data_responsive_sm']) ?>" data-responsive-xs="<?php echo esc_attr($settings['data_responsive_xs']) ?>">
				<?php
				global $post;
				foreach ($mwt_events as $post) {
					setup_postdata($post);

					$event_thumbnail = get_the_post_thumbnail($post, array(500, 780), array());
					$ticket_price = tribe_get_cost($post, true);
					$event_link = tribe_get_event_link($post->ID);
					$event_title = get_the_title($post->ID);
					$event_start_date = tribe_get_start_date($post);
					$event_venue = tribe_get_venue($post);
					$event_excerpt = $post->post_excerpt;

					if ($event_thumbnail) {
				?>
						<div class="owl-carousel-item">
							<article class="mwt-elementor-events-article">
								<div class="mwt-elementor-events-item-media">
									<?php echo wp_kses_post($event_thumbnail);  ?>
									<?php if ($ticket_price) : ?>
										<div class="mwt-elementor-events-entry-meta-corner">
											<span class="mwt-elementor-events-top-text"><?php echo esc_html__('from', 'mwt-addons-for-elementor'); ?></span>
											<span class="mwt-elementor-events-ticket-price"><?php echo esc_html($ticket_price); ?></span>
											<span class="mwt-elementor-events-bottom-text"><?php echo esc_html__('per ticket', 'mwt-addons-for-elementor'); ?></span>
										</div>
									<?php endif; ?>
								</div>
								<div class="mwt-elementor-events-item-content">
									<h3 class="mwt-elementor-events-event-title">
										<a href="<?php echo esc_url($event_link) ?>"><?php echo esc_html($event_title); ?></a>
									</h3>
									<?php if (($event_start_date || $event_venue) && $settings['show_meta'] === 'yes') : ?>
										<p class="mwt-elementor-events-meta">
											<?php if ($event_start_date) : ?>
												<?php echo esc_html($event_start_date); ?>
											<?php endif; ?>
											<?php if ($event_venue) : ?>
												<span class="mwt-elementor-events-delimiter"> / </span><?php echo esc_html($event_venue); ?>
											<?php endif; ?>
										</p>
									<?php endif; ?>
									<div class="mwt-elementor-events-entry-content">
										<?php if ($event_excerpt && ($settings['show_excerpt'] === 'yes')) : ?>
											<div class="mwt-elementor-events-content-3lines-ellipsis mwt-elementor-events-excerpt">
												<?php echo esc_html($event_excerpt); ?>
											</div>
										<?php endif; ?>
										<?php if ($settings['show_buy_link'] === 'yes' || $settings['show_more_link'] === 'yes') : ?>
											<div class="mwt-elementor-events-event-buttons">
												<?php if (!empty($ticket_price) && ($settings['show_buy_link'] === 'yes')) { ?>
													<a href="<?php echo esc_url($event_link) ?>#tribe-tickets__tickets-form" class="mwt-elementor-events-buy-tickets-button"><?php echo esc_html__($settings['buy_link_text'], 'mwt-addons-for-elementor'); ?></a>
												<?php } ?>
												<?php if ($settings['show_more_link'] === 'yes') : ?>
													<a href="<?php echo esc_url($event_link); ?>" class="mwt-elementor-events-button"><?php echo esc_html__($settings['more_link_text'], 'mwt-addons-for-elementor'); ?></a>
												<?php endif; ?>
											</div>
										<?php endif; ?>
									</div>
								</div>
							</article>
						</div>
				<?php }
				} ?>
			</div>
		</div>
<?php }
}
