<?php
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

/**
 * Elementor MWT Team Carousel Widget.
 *
 * Elementor widget that inserts an embbedable content into the page, from any given URL.
 *
 * @since 1.0.0
 */

class Elementor_MWT_Before_After extends \Elementor\Widget_Base

{
    /**
     * Retrieve Widget Dependent CSS.
     *
     * @since 1.0.0
     * @access public
     * @return array CSS style handles.
     */
    public function get_style_depends()
    {
        return ['before-after-style'];
    }

    /**
     * Retrieve Widget Dependent JS.
     *
     * @since 1.0.0
     * @access public
     * @return array JS script handles.
     */
    public function get_script_depends()
    {
        return ['before-after-script'];
    }

    /**
     * Get widget name.
     *
     * Retrieve list widget name.
     *
     * @since 1.0.0
     * @access public
     * @return string Widget name.
     */
    public function get_name()
    {
        return 'before-after';
    }

    /**
     * Get widget title.
     *
     * Retrieve list widget title.
     *
     * @since 1.0.0
     * @access public
     * @return string Widget title.
     */
    public function get_title()
    {
        return esc_html__('MWT Before/After', 'mwt-addons-for-elementor');
    }

    /**
     * Get widget categories.
     *
     * Retrieve the list of categories the list widget belongs to.
     *
     * @since 1.0.0
     * @access public
     * @return array Widget categories.
     */
    public function get_categories()
    {
        return ['mwt'];
    }

    /**
     * Get widget icon.
     *
     * Retrieve image carousel widget icon.
     *
     * @since 1.0.0
     * @access public
     *
     * @return string Widget icon.
     */
    public function get_icon()
    {
        return 'eicon-image';
    }

    /**
     * Get widget keywords.
     *
     * Retrieve the list of keywords the widget belongs to.
     *
     * @since 2.1.0
     * @access public
     *
     * @return array Widget keywords.
     */
    public function get_keywords()
    {
        return ['mwt', 'image', 'photo', 'visual', 'before', 'after'];
    }

    /**
     * Register list widget controls.
     *
     * Add input fields to allow the user to customize the widget settings.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function register_controls()
    {

        $this->start_controls_section(
            'content_section',
            [
                'label' => esc_html__('Images', 'mwt-addons-for-elementor'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->start_controls_tabs(
            'style_tabs'
        );

        $this->start_controls_tab(
            'style_before_tab',
            [
                'label' => esc_html__('Before', 'mwt-addons-for-elementor'),
            ]
        );

        $this->add_control(
            'image_before',
            [
                'label' => esc_html__('Choose Image', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::MEDIA,
                'dynamic' => [
                    'active' => true,
                ],
                'default' => [
                    'url' => \Elementor\Utils::get_placeholder_image_src(),
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'style_after_tab',
            [
                'label' => esc_html__('After', 'mwt-addons-for-elementor'),
            ]
        );

        $this->add_control(
            'image_after',
            [
                'label' => esc_html__('Choose Image', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::MEDIA,
                'dynamic' => [
                    'active' => true,
                ],
                'default' => [
                    'url' => \Elementor\Utils::get_placeholder_image_src(),
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->add_group_control(
            \Elementor\Group_Control_Image_Size::get_type(),
            [
                'name' => 'thumbnail',
                'separator' => 'none',
            ]
        );

        $this->add_control(
            'handle_img',
            [
                'label' => esc_html__('Handle Image', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'mwt-addons-for-elementor'),
                'label_off' => esc_html__('Hide', 'mwt-addons-for-elementor'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'choose_handle_img',
            [
                'label' => esc_html__('Choose Image', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::MEDIA,
                'default' => [
                    'url' => \Elementor\Utils::get_placeholder_image_src(),
                ],
                'selectors' => [
                    '{{WRAPPER}} .twentytwenty-handle::before' => 'background-image: url( {{URL}} ); background-position: center; background-repeat: no-repeat; background-size: cover;',
                ],
                'condition' => [
                    'handle_img' => 'yes'
                ]
            ]
        );

        $this->end_controls_section();

        // style
        $this->start_controls_section(
            'style_section',
            [
                'label' => esc_html__('Handle', 'mwt-addons-for-elementor'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'handle_bg_color',
            [
                'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-before-after-container .twentytwenty-handle' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'handle_border_color',
            [
                'label' => esc_html__('Border Color', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-before-after-container .twentytwenty-handle' => 'border-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'handle_bg_color_hover',
            [
                'label' => esc_html__('Hover Background Color', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'separator' => 'before',
                'selectors' => [
                    '{{WRAPPER}} .mwt-before-after-container .twentytwenty-handle:hover, {{WRAPPER}} .mwt-before-after-container .twentytwenty-handle:focus, {{WRAPPER}} .mwt-before-after-container .twentytwenty-handle:active' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'handle_border_color_hover',
            [
                'label' => esc_html__('Hover Border Color', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-before-after-container .twentytwenty-handle:hover, {{WRAPPER}} .mwt-before-after-container .twentytwenty-handle:focus, {{WRAPPER}} .mwt-before-after-container .twentytwenty-handle:active' => 'border-color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_section();
    }

    /**
     * Render list widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display();

        $hide_arrows = ('yes' === $settings['handle_img']) ? 'hide-arrows' : ''; ?>

        <div class='mwt-before-after-container <?php echo esc_attr($hide_arrows) ?>'>
            <?php
            if ($settings['image_before']['url'] || $settings['image_before']['id']) :
                $this->add_render_attribute('before_image', 'src', $settings['image_before']['url']);
                $this->add_render_attribute('before_image', 'alt', \Elementor\Control_Media::get_image_alt($settings['image_before']));
                echo \Elementor\Group_Control_Image_Size::get_attachment_image_html($settings, 'thumbnail', 'image_before');
            endif;
            if ($settings['image_after']['url'] || $settings['image_after']['id']) :
                $this->add_render_attribute('after_image', 'src', $settings['image_after']['url']);
                $this->add_render_attribute('after_image', 'alt', \Elementor\Control_Media::get_image_alt($settings['image_after']));
                echo \Elementor\Group_Control_Image_Size::get_attachment_image_html($settings, 'thumbnail', 'image_after');
            endif;
            ?>
        </div>
<?php }
}
