<?php
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

/**
 * Elementor MWT Banner Widget.
 *
 * Elementor widget that inserts an embbedable content into the page, from any given URL.
 *
 * @since 1.0.0
 */

class Elementor_MWT_Banner extends \Elementor\Widget_Base

{

	/**
	 * Retrieve Widget Dependent CSS.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array CSS style handles.
	 */
	public function get_style_depends()
	{
		return ['banner-style'];
	}

	/**
	 * Get widget name.
	 *
	 * Retrieve list widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return string Widget name.
	 */
	public function get_name()
	{
		return 'banner';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve list widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return string Widget title.
	 */
	public function get_title()
	{
		return esc_html__('MWT Banner', 'mwt-addons-for-elementor');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the list widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array Widget categories.
	 */
	public function get_categories()
	{
		return ['mwt'];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve image carousel widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon()
	{
		return 'eicon-image';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @since 2.1.0
	 * @access public
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords()
	{
		return ['mwt', 'image', 'photo', 'visual', 'banner', 'box'];
	}

	/**
	 * Register list widget controls.
	 *
	 * Add input fields to allow the user to customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls()
	{

		$this->start_controls_section(
			'content_section',
			[
				'label' => esc_html__('Content', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'image',
			[
				'label' => esc_html__('Choose Image', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::MEDIA,
				'dynamic' => [
					'active' => true,
				],
				'default' => [
					'url' => \Elementor\Utils::get_placeholder_image_src(),
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Image_Size::get_type(),
			[
				'name' => 'thumbnail',
				'separator' => 'none',
			]
		);

		$this->add_control(
			'item_title',
			[
				'label' => esc_html__('Title', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::TEXT,
				'placeholder' => esc_html__('Title', 'mwt-addons-for-elementor'),
				'default' => esc_html__('Title', 'mwt-addons-for-elementor'),
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
			]
		);

		$this->add_control(
			'item_title_link',
			[
				'label'       => __('Title Link', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::URL,
				'placeholder' => 'https://example.com/',
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
			]
		);

		$this->add_control(
			'item_description',
			[
				'label'       => __('Description', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__('Description', 'mwt-addons-for-elementor'),
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
			]
		);

		$this->add_control(
			'always_hovered',
			[
				'label' => esc_html__('Always Hovered', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'return_value' => 'yes',
				'default' => 'no',
			]
		);

		$this->end_controls_section();

		// style
		$this->start_controls_section(
			'style_item_section',
			[
				'label' => esc_html__('Items', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'team_item_align',
			[
				'label'     => __('Alignment', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::CHOOSE,
				'options'   => [
					'left' => [
						'title' => __('Left', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-left',
					],
					'center'     => [
						'title' => __('Center', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-center',
					],
					'right'   => [
						'title' => __('Right', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-right',
					],
				],
				'default'   => 'Left',
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-banner-content-wrap .mwt-elementor-banner-width-containter' => 'text-align: {{VALUE}}',
				],
			]
		);

		$this->add_responsive_control(
			'item_max_width',
			[
				'label'      => __('Content Width', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 1000,
						'step' => 1,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => '%',
					'size' => 100,
				],
				'selectors'  => [
					'{{WRAPPER}} .banner-container .mwt-elementor-banner-width-containter'  => 'width: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'item_padding',
			[
				'label'      => __('Content Padding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .banner-container .mwt-elementor-banner-width-containter' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'items_radius',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .banner-container' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
				],
			]
		);

		$this->start_controls_tabs(
			'style_tabs'
		);

		$this->start_controls_tab(
			'style_normal_tab',
			[
				'label' => esc_html__('Normal', 'textdomain'),
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'member_box_shadow',
				'selector' => '{{WRAPPER}} .banner-container',
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'style_hover_tab',
			[
				'label' => esc_html__('Hover', 'textdomain'),
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'member_box_shadow_hover',
				'selector' => '{{WRAPPER}} .banner-container:hover',
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_control(
			'items_hover_border_width',
			[
				'label'      => __('Hover Border Width', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 10,
						'step' => 1,
					],
					'%' => [
						'min' => 0,
						'max' => 10,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 1,
				],
				'selectors'  => [
					'{{WRAPPER}} .banner-container::before'  => 'border-width: {{SIZE}}{{UNIT}}',
					'{{WRAPPER}} .banner-container::after'  => 'border-width: {{SIZE}}{{UNIT}}',
				],
				'separator' => 'before',
			]
		);

		$this->add_control(
			'items_hover_border_color',
			[
				'label' => esc_html__('Hover Border Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .banner-container::before' => 'border-color: {{VALUE}};',
					'{{WRAPPER}} .banner-container::after' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_image',
			[
				'label' => esc_html__('Image', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'image_hover_overlay_color',
			[
				'label' => esc_html__('Hover Overlay Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .banner-container .mwt-elementor-banner-overlay' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_title_section',
			[
				'label' => esc_html__('Title', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'title_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-banner-title' => 'color: {{VALUE}};',
					'{{WRAPPER}} a.mwt-elementor-banner-title' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'title_color_active',
			[
				'label' => esc_html__('Color Hover', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} a.mwt-elementor-banner-title:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'title_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-elementor-banner-title, {{WRAPPER}} .mwt-elementor-banner-title > a',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Text_Shadow::get_type(),
			[
				'label' => esc_html__('Text Shadow', 'mwt-addons-for-elementor'),
				'name' => 'title_shadow',
				'selector' => '{{WRAPPER}} .mwt-elementor-banner-title',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_description_section',
			[
				'label' => esc_html__('Description', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'description_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-elementor-banner-description' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'description_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-elementor-banner-description',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Text_Shadow::get_type(),
			[
				'label' => esc_html__('Text Shadow', 'mwt-addons-for-elementor'),
				'name' => 'description_shadow',
				'selector' => '{{WRAPPER}} .mwt-elementor-banner-description',
			]
		);

		$this->add_responsive_control(
			'description_height',
			[
				'label'      => __('Height', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 1000,
						'step' => 1,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 60,
				],
				'selectors'  => [
					'{{WRAPPER}} .banner-container:hover .mwt-elementor-banner-description'  => 'height: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->add_responsive_control(
			'description_top_spacing',
			[
				'label'      => __('Top Spacing', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-elementor-banner-description'  => 'margin-top: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render list widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render()
	{
		$settings = $this->get_settings_for_display();

		$always_hovered = ('yes' === ($settings['always_hovered'])) ? 'always-hovered' : ''; ?>

		<div class="banner-container <?php echo esc_attr($always_hovered) ?>">
			<?php
			$image_url = \Elementor\Group_Control_Image_Size::get_attachment_image_src($settings['image']['id'], 'thumbnail', $settings);
			if (!$image_url && isset($settings['image']['url'])) {
				$image_url = $settings['image']['url'];
			}
			if (!empty($settings['item_title'])) {
				$image_alt = $settings['item_title'];
			} else {
				$image_alt = 'This is a title';
			}
			$image_html = '<img class="mwt-elementor-banner-image" src="' . esc_attr($image_url) . '" alt="' . esc_attr($image_alt) . '" />';
			echo wp_kses_post($image_html);
			?>
			<div class="mwt-elementor-banner-overlay"></div>
			<?php
			$title = $settings['item_title'];
			$title_link = $settings['item_title_link']['url'];
			$description = $settings['item_description'];
			?>
			<div class="mwt-elementor-banner-content-wrap">
				<div class="mwt-elementor-banner-width-containter">
					<?php
					if (!empty($title_link)) { ?>
						<a href="<?php echo esc_attr($title_link) ?>" class="mwt-elementor-banner-title">
							<?php echo $title; ?>
						</a>
					<?php } else { ?>
						<div class="mwt-elementor-banner-title">
							<?php echo $title; ?>
						</div>
					<?php } ?>
					<?php if (!empty($description)) { ?>
						<div class="mwt-elementor-banner-description">
							<?php echo $description; ?>
						</div>
					<?php } ?>
				</div>
			</div>
		</div>
<?php }
}
