<?php
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

/**
 * Elementor MWT Team Carousel Widget.
 *
 * Elementor widget that inserts an embbedable content into the page, from any given URL.
 *
 * @since 1.0.0
 */

class Elementor_MWT_Album extends \Elementor\Widget_Base

{
    /**
     * Retrieve Widget Dependent CSS.
     *
     * @since 1.0.0
     * @access public
     * @return array CSS style handles.
     */
    public function get_style_depends()
    {
        return ['album-style'];
    }

    /**
     * Get widget name.
     *
     * Retrieve list widget name.
     *
     * @since 1.0.0
     * @access public
     * @return string Widget name.
     */
    public function get_name()
    {
        return 'album';
    }

    /**
     * Get widget title.
     *
     * Retrieve list widget title.
     *
     * @since 1.0.0
     * @access public
     * @return string Widget title.
     */
    public function get_title()
    {
        return esc_html__('MWT Album', 'mwt-addons-for-elementor');
    }

    /**
     * Get widget categories.
     *
     * Retrieve the list of categories the list widget belongs to.
     *
     * @since 1.0.0
     * @access public
     * @return array Widget categories.
     */
    public function get_categories()
    {
        return ['mwt'];
    }

    /**
     * Get widget icon.
     *
     * Retrieve image carousel widget icon.
     *
     * @since 1.0.0
     * @access public
     *
     * @return string Widget icon.
     */
    public function get_icon()
    {
        return 'eicon-headphones';
    }

    /**
     * Get widget keywords.
     *
     * Retrieve the list of keywords the widget belongs to.
     *
     * @since 2.1.0
     * @access public
     *
     * @return array Widget keywords.
     */
    public function get_keywords()
    {
        return ['mwt', 'image', 'photo', 'visual', 'before', 'after'];
    }

    /**
     * Register list widget controls.
     *
     * Add input fields to allow the user to customize the widget settings.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function register_controls()
    {
        $this->start_controls_section(
            'album_settings',
            [
                'label' => __('Album', 'mwt-addons-for-elementor'),
            ]
        );

        $this->add_control(
            'front_image',
            [
                'label' => esc_html__('Choose Front Image', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::MEDIA,
                'default' => [
                    'url' => \Elementor\Utils::get_placeholder_image_src(),
                ],
                'dynamic' => [
                    'active' => true,
                ],
                'ai' => [
                    'active' => false,
                ],
            ]
        );

        $this->add_control(
            'front_image_add_frame_block',
            [
                'label' => esc_html__('Add Frame Block', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'return_value' => 'yes',
                'default' => 'no',
            ]
        );

        $this->add_control(
            'back_image',
            [
                'label' => esc_html__('Choose Back Image', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::MEDIA,
                'default' => [
                    'url' => \Elementor\Utils::get_placeholder_image_src(),
                ],
                'dynamic' => [
                    'active' => true,
                ],
                'ai' => [
                    'active' => false,
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Image_Size::get_type(),
            [
                'name' => 'album_thumbnail',
                'exclude' => [],
                'include' => [],
                'default' => 'full',
            ]
        );

        $this->add_control(
            'show_play_icon',
            [
                'label' => esc_html__('Show Play Icon', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'return_value' => 'yes',
                'default' => 'no',
            ]
        );

        $this->add_control(
            'play_icon',
            [
                'label' => esc_html__('Play Icon', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::ICONS,
                'default' => [
                    'value' => 'fas fa-play',
                    'library' => 'fa-solid',
                ],
                'recommended' => [
                    'fa-solid' => [
                        'play',
                    ],
                ],
                'condition' => [
                    'show_play_icon' => 'yes',
                ]
            ]
        );

        $this->add_control(
            'play_icon_link',
            [
                'label' => esc_html__('Play Icon Link', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::URL,
                'options' => ['url', 'is_external', 'nofollow'],
                'default' => [
                    'url' => '',
                    'is_external' => false,
                    'nofollow' => false,
                ],
                'label_block' => true,
                'condition' => [
                    'show_play_icon' => 'yes',
                ]
            ]
        );

        $this->add_control(
            'album_always_hovered',
            [
                'label' => esc_html__('Always Hovered', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'return_value' => 'yes',
                'default' => 'no',
            ]
        );

        $this->add_responsive_control(
            'album_content_align',
            [
                'label' => esc_html__('Alignment', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [
                        'title' => esc_html__('Left', 'mwt-addons-for-elementor'),
                        'icon' => 'eicon-text-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__('Center', 'mwt-addons-for-elementor'),
                        'icon' => 'eicon-text-align-center',
                    ],
                    'right' => [
                        'title' => esc_html__('Right', 'mwt-addons-for-elementor'),
                        'icon' => 'eicon-text-align-right',
                    ],
                ],
                'default' => 'left',
                'selectors' => [
                    '{{WRAPPER}} .mwt-elementor-album-item-content' => 'text-align: {{VALUE}};',
                    '{{WRAPPER}} .mwt-elementor-album-item-content .mwt-elementor-album-links-wrap' => 'justify-content: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'album_price',
            [
                'label' => esc_html__('Album Price', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'placeholder' => esc_html__('Type your price here', 'mwt-addons-for-elementor'),
                'label_block' => true,
                'ai' => [
                    'active' => false,
                ],
            ]
        );

        $this->add_control(
            'album_title',
            [
                'label' => esc_html__('Album Title', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'placeholder' => esc_html__('Type your title here', 'mwt-addons-for-elementor'),
                'label_block' => true,
                'ai' => [
                    'active' => false,
                ],
            ]
        );

        $this->add_control(
            'album_title_link',
            [
                'label' => esc_html__('Album Title Link', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::URL,
                'options' => ['url', 'is_external', 'nofollow'],
                'default' => [
                    'url' => '',
                    'is_external' => false,
                    'nofollow' => false,
                ],
                'label_block' => true,
            ]
        );

        $repeater = new \Elementor\Repeater();

        $repeater->add_control(
            'info_title',
            [
                'label' => esc_html__('Title', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'placeholder' => esc_html__('Type your title here', 'mwt-addons-for-elementor'),
                'label_block' => true,
                'ai' => [
                    'active' => false,
                ],
            ]
        );

        $repeater->add_control(
            'info_text',
            [
                'label' => esc_html__('Text', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'placeholder' => esc_html__('Type your text here', 'mwt-addons-for-elementor'),
                'label_block' => true,
                'ai' => [
                    'active' => false,
                ],
            ]
        );

        $this->add_control(
            'album-info-list',
            [
                'label' => esc_html__('Album Info', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'info_title' => '',
                        'info_text' =>  '',
                    ],
                    [
                        'info_title' => '',
                        'info_text' =>  '',
                    ],
                ],
                'title_field' => '{{{ info_title }}}',
            ]
        );

        $repeater_links = new \Elementor\Repeater();

        $repeater_links->add_control(
            'link_title',
            [
                'label' => esc_html__('Title', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'placeholder' => esc_html__('Type your title here', 'mwt-addons-for-elementor'),
                'label_block' => true,
                'ai' => [
                    'active' => false,
                ],
            ]
        );

        $repeater_links->add_control(
            'link_icon',
            [
                'label' => esc_html__('Icon', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::ICONS,
                'default' => [
                    'value' => 'fas fa-circle',
                    'library' => 'fa-solid',
                ],
                'recommended' => [
                    'fa-solid' => [
                        'circle',
                        'dot-circle',
                        'square-full',
                    ],
                    'fa-regular' => [
                        'circle',
                        'dot-circle',
                        'square-full',
                    ],
                ],
            ]
        );

        $repeater_links->add_control(
            'website_link',
            [
                'label' => esc_html__('Link', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::URL,
                'options' => ['url', 'is_external', 'nofollow'],
                'default' => [
                    'url' => '',
                    'is_external' => true,
                    'nofollow' => true,
                ],
                'label_block' => true,
            ]
        );

        $this->add_control(
            'album-links-list',
            [
                'label' => esc_html__('Links to the stores', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::REPEATER,
                'fields' => $repeater_links->get_controls(),
                'default' => [
                    [
                        'link_title' => '',
                        'link_icon' => '',
                        'website_link' => '',
                    ],
                    [
                        'link_title' => '',
                        'link_icon' => '',
                        'website_link' => '',
                    ],
                ],
                'title_field' => '{{{ link_title }}}',
            ]
        );

        $this->end_controls_section();

        // Style
        $this->start_controls_section(
            'style_item_section',
            [
                'label' => esc_html__('Frame Block', 'mwt-addons-for-elementor'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
                'condition' => [
                    'front_image_add_frame_block' => 'yes',
                ]
            ]
        );

        $this->add_responsive_control(
            'album_frame_block_width',
            [
                'label'      => __('Border Width', 'mwt-addons-for-elementor'),
                'type'       => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px'],
                'default' => [
                    'top' => 4,
                    'right' => 4,
                    'bottom' => 4,
                    'left' => 4,
                    'unit' => 'px',
                ],
                'selectors'  => [
                    '{{WRAPPER}} .mwt-elementor-album-with-frame-block::before' => 'border-width: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
                ],
                'condition' => [
                    'front_image_add_frame_block' => 'yes',
                ]
            ]
        );

        $this->add_responsive_control(
            'album_frame_block_radius',
            [
                'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
                'type'       => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'default' => [
                    'top' => 5,
                    'right' => 5,
                    'bottom' => 5,
                    'left' => 5,
                    'unit' => 'px',
                ],
                'selectors'  => [
                    '{{WRAPPER}} .mwt-elementor-album-with-frame-block::before' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
                ],
                'condition' => [
                    'front_image_add_frame_block' => 'yes',
                ]
            ]
        );

        $this->add_control(
            'album_frame_block_color',
            [
                'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-elementor-album-with-frame-block::before' => 'border-color: {{VALUE}};',
                ],
                'condition' => [
                    'front_image_add_frame_block' => 'yes',
                ]
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'play_icon_section',
            [
                'label' => esc_html__('Play Icon', 'mwt-addons-for-elementor'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
                'condition' => [
                    'show_play_icon' => 'yes',
                ]
            ]
        );

        $this->add_responsive_control(
            'play_icon_size',
            [
                'label'      => __('Icon Size (px)', 'mwt-addons-for-elementor'),
                'type'       => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'range' => [
                    'px' => [
                        'min' => 8,
                        'max' => 24,
                        'step' => 1,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 12,
                ],
                'selectors'  => [
                    '{{WRAPPER}} .mwt-elementor-album-media-links .mwt-elementor-album-link-wrap>span, .mwt-elementor-album-media-links .mwt-elementor-album-link-wrap>a'  => 'font-size: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'show_play_icon' => 'yes',
                ]
            ]
        );

        $this->start_controls_tabs(
            'play_icon_style_tabs'
        );

        $this->start_controls_tab(
            'play_icon_style_normal_tab',
            [
                'label' => esc_html__('Normal', 'mwt-addons-for-elementor'),
                'condition' => [
                    'show_play_icon' => 'yes',
                ]
            ]
        );

        $this->add_control(
            'play_icon_color',
            [
                'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-elementor-album-media-links .mwt-elementor-album-link-wrap>span, .mwt-elementor-album-media-links .mwt-elementor-album-link-wrap>a' => 'color: {{VALUE}};',
                ],
                'condition' => [
                    'show_play_icon' => 'yes',
                ]
            ]
        );

        $this->add_control(
            'play_icon_bg_color',
            [
                'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-elementor-album-media-links .mwt-elementor-album-link-wrap>span, .mwt-elementor-album-media-links .mwt-elementor-album-link-wrap>a' => 'background-color: {{VALUE}};',
                ],
                'condition' => [
                    'show_play_icon' => 'yes',
                ]
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'play_icon_style_hover_tab',
            [
                'label' => esc_html__('Hover', 'mwt-addons-for-elementor'),
                'condition' => [
                    'show_play_icon' => 'yes',
                ]
            ]
        );

        $this->add_control(
            'play_icon_color_hover',
            [
                'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-elementor-album-media-links .mwt-elementor-album-link-wrap>a:hover' => 'color: {{VALUE}};',
                ],
                'condition' => [
                    'show_play_icon' => 'yes',
                ]
            ]
        );

        $this->add_control(
            'play_icon_bg_color_hover',
            [
                'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-elementor-album-media-links .mwt-elementor-album-link-wrap>a:hover' => 'background-color: {{VALUE}};',
                ],
                'condition' => [
                    'show_play_icon' => 'yes',
                ]
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->end_controls_section();

        $this->start_controls_section(
            'album_price_section',
            [
                'label' => esc_html__('Price', 'mwt-addons-for-elementor'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'label' => esc_html__('Price Typography', 'mwt-addons-for-elementor'),
                'name' => 'album_price_typography',
                'selector' =>
                '{{WRAPPER}} .mwt-elementor-album-price',
            ]
        );

        $this->add_control(
            'album_price_color',
            [
                'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-elementor-album-price' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'album_price_margin',
            [
                'label'      => __('Margin', 'mwt-addons-for-elementor'),
                'type'       => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .mwt-elementor-album-price' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'album_title_section',
            [
                'label' => esc_html__('Title', 'mwt-addons-for-elementor'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'label' => esc_html__('Title Typography', 'mwt-addons-for-elementor'),
                'name' => 'album_title_typography',
                'selector' =>
                '{{WRAPPER}} .mwt-elementor-album-title',
            ]
        );

        $this->add_control(
            'album_title_color',
            [
                'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-elementor-album-title' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'album_title_color_hover',
            [
                'label' => esc_html__('Hover Color', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-elementor-album-title a:hover' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'album_title_margin',
            [
                'label'      => __('Margin', 'mwt-addons-for-elementor'),
                'type'       => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .mwt-elementor-album-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'album_info_section',
            [
                'label' => esc_html__('Info', 'mwt-addons-for-elementor'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
                'name' => 'album_info_typography',
                'selector' =>
                '{{WRAPPER}} .mwt-elementor-album-info-list',
            ]
        );

        $this->add_control(
            'album_info_color',
            [
                'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-elementor-album-info-list' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'album_info_padding',
            [
                'label'      => __('Padding', 'mwt-addons-for-elementor'),
                'type'       => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .mwt-elementor-album-info-list>div' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'album_info_title',
            [
                'label' => esc_html__('Title', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
                'name' => 'album_info_title_typography',
                'selector' =>
                '{{WRAPPER}} .mwt-elementor-album-info-list .mwt-elementor-album-info-list-title',
            ]
        );

        $this->add_control(
            'album_info_title_color',
            [
                'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-elementor-album-info-list .mwt-elementor-album-info-list-title' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'album_info_margin',
            [
                'label'      => __('Info Block Margin', 'mwt-addons-for-elementor'),
                'type'       => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'separator' => 'before',
                'selectors'  => [
                    '{{WRAPPER}} .mwt-elementor-album-info-list' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        //soc icons style
        $this->start_controls_section(
            'album_links_icon_style',
            [
                'label'     => __('Links', 'mwt-addons-for-elementor'),
                'tab'       => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'album_links_size',
            [
                'label'       => __('Size', 'mwt-addons-for-elementor'),
                'type'        => \Elementor\Controls_Manager::SLIDER,
                'size_units'  => ['px', 'em', '%'],
                'label_block' => true,
                'selectors'   => [
                    '{{WRAPPER}} .mwt-elementor-album-link-icon' => 'font-size: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->start_controls_tabs(
            'album_links_style_tabs'
        );

        $this->start_controls_tab(
            'album_links_style_normal_tab',
            [
                'label' => esc_html__('Normal', 'mwt-addons-for-elementor'),
            ]
        );

        $this->add_control(
            'album_links_default_colors',
            [
                'label'        => __('Brands Default Colors', 'mwt-addons-for-elementor'),
                'type'         => \Elementor\Controls_Manager::SWITCHER,
                'prefix_class' => 'album-links-defaults-',
            ]
        );

        $this->add_control(
            'album_links_color',
            [
                'label'     => __('Color', 'mwt-addons-for-elementor'),
                'type'      => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-elementor-album-link-icon'  => 'color: {{VALUE}};',
                ],
                'condition' => [
                    'album_links_default_colors!' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'album_links_default_background',
            [
                'label'        => __('Brands Default Background Colors', 'mwt-addons-for-elementor'),
                'type'         => \Elementor\Controls_Manager::SWITCHER,
                'prefix_class' => 'album-links-defaults-background-',
            ]
        );

        $this->add_control(
            'album_links_background',
            [
                'label'     => __('Background Color', 'mwt-addons-for-elementor'),
                'type'      => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-elementor-album-links-wrap a .mwt-elementor-album-link-icon'  => 'background-color: {{VALUE}}',
                ],
                'condition' => [
                    'album_links_default_background!' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'album_links_default_border',
            [
                'label'        => __('Brands Default Border Colors', 'mwt-addons-for-elementor'),
                'type'         => \Elementor\Controls_Manager::SWITCHER,
                'prefix_class' => 'album-links-defaults-border-',
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'album_links_style_hover_tab',
            [
                'label' => esc_html__('Hover', 'mwt-addons-for-elementor'),
            ]
        );

        $this->add_control(
            'album_links_default_hover_colors',
            [
                'label'        => __('Brands Default Hover Colors', 'mwt-addons-for-elementor'),
                'type'         => \Elementor\Controls_Manager::SWITCHER,
                'prefix_class' => 'album-links-defaults-hover-',
            ]
        );

        $this->add_control(
            'album_links_hover_color',
            [
                'label'     => __('Hover Color', 'mwt-addons-for-elementor'),
                'type'      => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-elementor-album-links-wrap a:hover .mwt-elementor-album-link-icon'  => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'album_links_default_hover_colors!'   => 'yes',
                ],
            ]
        );

        $this->add_control(
            'album_links_default_hover_background',
            [
                'label'        => __('Brands Default Hover Background Colors', 'mwt-addons-for-elementor'),
                'type'         => \Elementor\Controls_Manager::SWITCHER,
                'prefix_class' => 'album-links-defaults-background-hover-',
            ]
        );

        $this->add_control(
            'album_links_hover_background',
            [
                'label'     => __('Hover Background Color', 'mwt-addons-for-elementor'),
                'type'      => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-elementor-album-links-wrap a:hover .mwt-elementor-album-link-icon'  => 'background-color: {{VALUE}}',
                ],
                'condition' => [
                    'album_links_default_hover_background!' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'album_links_default_border_hover',
            [
                'label'        => __('Brands Default Hover Border Colors', 'mwt-addons-for-elementor'),
                'type'         => \Elementor\Controls_Manager::SWITCHER,
                'prefix_class' => 'album-links-defaults-border-hover-',
            ]
        );

        $this->add_control(
            'album_links_hover_border',
            [
                'label'     => __('Hover Border Color', 'mwt-addons-for-elementor'),
                'type'      => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-elementor-album-links-wrap a:hover .mwt-elementor-album-link-icon'  => 'border-color: {{VALUE}}',
                ],
                'condition' => [
                    'album_links_default_border_hover!' => 'yes',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name'     => 'album_links_border',
                'selector' => '{{WRAPPER}} .mwt-elementor-album-links-wrap a i',
            ]
        );

        $this->add_responsive_control(
            'album_links_radius',
            [
                'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
                'type'       => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .mwt-elementor-album-links-wrap a i' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
                ],
            ]
        );

        $this->add_responsive_control(
            'album_links_padding',
            [
                'label'      => __('Padding', 'mwt-addons-for-elementor'),
                'type'       => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .mwt-elementor-album-links-wrap a i' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'album_links_margin',
            [
                'label'      => __('Margin', 'mwt-addons-for-elementor'),
                'type'       => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .mwt-elementor-album-links-wrap a' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();
    }

    /**
     * Render list widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display();

        $album_always_hovered = ('yes' === ($settings['album_always_hovered'])) ? 'mwt-elementor-album-opened-media' : '';

        $front_image_add_frame_block = ('yes' === ($settings['front_image_add_frame_block'])) ? 'mwt-elementor-album-with-frame-block' : '';

?>
        <div class="mwt-elementor-album <?php echo esc_attr($album_always_hovered) ?>">
            <div class="mwt-elementor-album-item-media-wrap ">
                <?php
                $image_url_back = \Elementor\Group_Control_Image_Size::get_attachment_image_src($settings['back_image']['id'], 'album_thumbnail', $settings);
                if (!$image_url_back && isset($settings['back_image']['url'])) {
                    $image_url_back = $settings['back_image']['url'];
                }
                $image_html_back = '<img class="slide-image" src="' . esc_attr($image_url_back) . '" alt="' . esc_attr('Back image', 'mwt-addons-for-elementor') . '" />';
                echo wp_kses_post($image_html_back);
                ?>
                <div class="mwt-elementor-album-item-media mwt-elementor-album-back-media">
                    <?php
                    $image_url_back = \Elementor\Group_Control_Image_Size::get_attachment_image_src($settings['back_image']['id'], 'album_thumbnail', $settings);
                    if (!$image_url_back && isset($settings['back_image']['url'])) {
                        $image_url_back = $settings['back_image']['url'];
                    }
                    $image_html_back = '<img class="slide-image" src="' . esc_attr($image_url_back) . '" alt="' . esc_attr('Back image', 'mwt-addons-for-elementor') . '" />';
                    echo wp_kses_post($image_html_back);
                    ?>
                </div>
                <div class="mwt-elementor-album-item-media mwt-elementor-album-front-media <?php echo esc_attr($front_image_add_frame_block) ?>">
                    <?php
                    $image_url_front = \Elementor\Group_Control_Image_Size::get_attachment_image_src($settings['front_image']['id'], 'album_thumbnail', $settings);
                    if (!$image_url_front && isset($settings['front_image']['url'])) {
                        $image_url_front = $settings['back_image']['url'];
                    }
                    $image_html_front = '<img class="slide-image" src="' . esc_attr($image_url_front) . '" alt="' . esc_attr('Back image', 'mwt-addons-for-elementor') . '" />';
                    echo wp_kses_post($image_html_front);
                    if ($settings['show_play_icon'] === 'yes') :
                    ?>
                        <div class="mwt-elementor-album-media-links">
                            <div class="mwt-elementor-album-link-wrap">
                                <?php if (!empty($settings['play_icon_link']['url'])) { ?>
                                    <a href="<?php echo esc_html($settings['play_icon_link']['url']) ?>">
                                    <?php } else { ?>
                                        <span>
                                        <?php } ?>
                                        <?php
                                        $play_icon = isset($settings['play_icon']) ? $settings['play_icon'] : '';
                                        if ($play_icon) : ?>
                                            <?php \Elementor\Icons_Manager::render_icon($settings['play_icon'], ['aria-hidden' => 'true', 'class' => 'mwt-elementor-album-play-icon']); ?>
                                        <?php endif; ?>
                                        <?php if (!empty($settings['play_icon_link']['url'])) { ?>
                                    </a>
                                <?php } else { ?>
                                    </span>
                                <?php } ?>
                            </div>
                        </div>
                    <?php endif; ?>
                </div>
            </div>

            <div class="mwt-elementor-album-item-content">
                <?php if ($settings['album_price']) : ?>
                    <div class="mwt-elementor-album-price">
                        <?php echo esc_html($settings['album_price']); ?>
                    </div>
                <?php endif; ?>
                <?php if ($settings['album_title']) : ?>
                    <h4 class="mwt-elementor-album-title">
                        <?php if (!empty($settings['album_title_link']['url'])) { ?>
                            <a href="<?php echo esc_html($settings['album_title_link']['url']) ?>">
                            <?php } ?>
                            <?php echo esc_html($settings['album_title']); ?>
                            <?php if (!empty($settings['album_title_link']['url'])) { ?>
                            </a>
                        <?php } ?>
                    </h4>
                <?php endif; ?>
                <div class="mwt-elementor-album-info-list">
                    <?php
                    foreach ($settings['album-info-list'] as $index => $item) {
                        $info_title = $item['info_title'];
                        $info_text = $item['info_text'];
                    ?>
                        <div><span class="mwt-elementor-album-info-list-title"><?php echo esc_html($info_title) ?></span><?php echo esc_html($info_text) ?></div>
                    <?php } ?>
                </div>
                <div class="mwt-elementor-album-links-wrap">
                    <?php
                    foreach ($settings['album-links-list'] as $index => $item) {
                        $website_link = (!empty($item['website_link']['url'])) ? $item['website_link']['url'] : '#';
                    ?>
                        <a href="<?php echo esc_url($website_link) ?>">
                            <?php
                            $link_icon = isset($item['link_icon']) ? $item['link_icon'] : '';
                            if ($link_icon) : ?>
                                <?php \Elementor\Icons_Manager::render_icon($item['link_icon'], ['aria-hidden' => 'true', 'class' => 'mwt-elementor-album-link-icon']); ?>
                            <?php endif; ?>
                        </a>
                    <?php
                    } ?>

                </div>
            </div>
        </div>
<?php
    }
}
