<?php

namespace MWT_Addons;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

/**
 * Plugin class.
 *
 * The main class that initiates and runs the addon.
 *
 * @since 1.0.0
 */
final class Plugin
{

    /**
     * Addon Version
     *
     * @since 1.0.0
     * @var string The addon version.
     */
    const VERSION = '1.0.0';

    /**
     * Minimum Elementor Version
     *
     * @since 1.0.0
     * @var string Minimum Elementor version required to run the addon.
     */
    const MINIMUM_ELEMENTOR_VERSION = '3.7.0';

    /**
     * Minimum PHP Version
     *
     * @since 1.0.0
     * @var string Minimum PHP version required to run the addon.
     */
    const MINIMUM_PHP_VERSION = '7.3';

    /**
     * Instance
     *
     * @since 1.0.0
     * @access private
     * @static
     * @var \MWT_Addons\Plugin The single instance of the class.
     */
    private static $_instance = null;

    /**
     * Instance
     *
     * Ensures only one instance of the class is loaded or can be loaded.
     *
     * @since 1.0.0
     * @access public
     * @static
     * @return \MWT_Addons\Plugin An instance of the class.
     */
    public static function instance()
    {

        if (is_null(self::$_instance)) {
            self::$_instance = new self();
        }
        return self::$_instance;
    }

    /**
     * Constructor
     *
     * Perform some compatibility checks to make sure basic requirements are meet.
     * If all compatibility checks pass, initialize the functionality.
     *
     * @since 1.0.0
     * @access public
     */
    public function __construct()
    {

        if ($this->is_compatible()) {
            add_action('elementor/init', [$this, 'init']);

            //extend existing widgets
            require_once(__DIR__ . '/extend.php');
        }
    }

    /**
     * Compatibility Checks
     *
     * Checks whether the site meets the addon requirement.
     *
     * @since 1.0.0
     * @access public
     */
    public function is_compatible()
    {

        // Check if Elementor installed and activated
        if (!did_action('elementor/loaded')) {
            add_action('admin_notices', [$this, 'admin_notice_missing_main_plugin']);
            return false;
        }

        // Check for required Elementor version
        if (!version_compare(ELEMENTOR_VERSION, self::MINIMUM_ELEMENTOR_VERSION, '>=')) {
            add_action('admin_notices', [$this, 'admin_notice_minimum_elementor_version']);
            return false;
        }

        // Check for required PHP version
        if (version_compare(PHP_VERSION, self::MINIMUM_PHP_VERSION, '<')) {
            add_action('admin_notices', [$this, 'admin_notice_minimum_php_version']);
            return false;
        }

        return true;
    }

    /**
     * Admin notice
     *
     * Warning when the site doesn't have Elementor installed or activated.
     *
     * @since 1.0.0
     * @access public
     */
    public function admin_notice_missing_main_plugin()
    {

        if (isset($_GET['activate'])) unset($_GET['activate']);

        $message = sprintf(
            /* translators: 1: Plugin name 2: Elementor */
            esc_html__('"%1$s" requires "%2$s" to be installed and activated.', 'mwt-addons-for-elementor'),
            '<strong>' . esc_html__('MWT Addons for Elementor', 'mwt-addons-for-elementor') . '</strong>',
            '<strong>' . esc_html__('Elementor', 'mwt-addons-for-elementor') . '</strong>'
        );

        printf('<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message);
    }

    /**
     * Admin notice
     *
     * Warning when the site doesn't have a minimum required Elementor version.
     *
     * @since 1.0.0
     * @access public
     */
    public function admin_notice_minimum_elementor_version()
    {

        if (isset($_GET['activate'])) unset($_GET['activate']);

        $message = sprintf(
            /* translators: 1: Plugin name 2: Elementor 3: Required Elementor version */
            esc_html__('"%1$s" requires "%2$s" version %3$s or greater.', 'mwt-addons-for-elementor'),
            '<strong>' . esc_html__('MWT Addons for Elementor', 'mwt-addons-for-elementor') . '</strong>',
            '<strong>' . esc_html__('Elementor', 'mwt-addons-for-elementor') . '</strong>',
            self::MINIMUM_ELEMENTOR_VERSION
        );

        printf('<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message);
    }

    /**
     * Admin notice
     *
     * Warning when the site doesn't have a minimum required PHP version.
     *
     * @since 1.0.0
     * @access public
     */
    public function admin_notice_minimum_php_version()
    {

        if (isset($_GET['activate'])) unset($_GET['activate']);

        $message = sprintf(
            /* translators: 1: Plugin name 2: PHP 3: Required PHP version */
            esc_html__('"%1$s" requires "%2$s" version %3$s or greater.', 'mwt-addons-for-elementor'),
            '<strong>' . esc_html__('MWT Addons for Elementor', 'mwt-addons-for-elementor') . '</strong>',
            '<strong>' . esc_html__('PHP', 'mwt-addons-for-elementor') . '</strong>',
            self::MINIMUM_PHP_VERSION
        );

        printf('<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message);
    }

    /**
     * Initialize
     *
     * Load the addons functionality only after Elementor is initialized.
     *
     * Fired by `elementor/init` action hook.
     *
     * @since 1.0.0
     * @access public
     */
    public function init()
    {
        add_action('elementor/widgets/register', [$this, 'register_widgets']);
        add_action('elementor/frontend/after_enqueue_styles', [$this, 'frontend_styles']);
        add_action('elementor/frontend/after_enqueue_scripts', [$this, 'frontend_scripts']);
        add_action('elementor/elements/categories_registered', [$this, 'mwt_elementor_categories']);
    }

    public function mwt_elementor_categories($elements_manager)
    {
        // If the category is empty, it'll be automatically hidden by Elementor.
        $elements_manager->add_category(
            'mwt',
            array(
                'title' => __('MWT addons', 'mwt-addons-for-elementor'),
                'icon'   => 'fa fa-plug',
                'active' => false,
            )
        );
    }

    /**
     * Register Widgets
     *
     * Load widgets files and register new Elementor widgets.
     *
     * Fired by `elementor/widgets/register` action hook.
     *
     * @param \Elementor\Widgets_Manager $widgets_manager Elementor widgets manager.
     */
    public function register_widgets($widgets_manager)
    {
        require_once(__DIR__ . '/widgets/main-slider.php');
        require_once(__DIR__ . '/widgets/gallery-carousel.php');
        require_once(__DIR__ . '/widgets/gallery-grid.php');
        require_once(__DIR__ . '/widgets/team-slider.php');
        require_once(__DIR__ . '/widgets/team-carousel.php');
        require_once(__DIR__ . '/widgets/mwt-posts.php');
        require_once(__DIR__ . '/widgets/testimonials-carousel.php');
        require_once(__DIR__ . '/widgets/products-carousel.php');
        require_once(__DIR__ . '/widgets/team-box.php');
        require_once(__DIR__ . '/widgets/banner.php');
        require_once(__DIR__ . '/widgets/before-after.php');
        require_once(__DIR__ . '/widgets/pricing-box.php');
        require_once(__DIR__ . '/widgets/counter-up.php');
        require_once(__DIR__ . '/widgets/events-carousel.php');
        require_once(__DIR__ . '/widgets/album.php');


        $widgets_manager->register(new \Elementor_MWT_Main_Slider());
        $widgets_manager->register(new \Elementor_MWT_Gallery_Carousel());
        $widgets_manager->register(new \Elementor_MWT_Gallery_Grid());
        $widgets_manager->register(new \Elementor_MWT_Team_Slider_Widget());
        $widgets_manager->register(new \Elementor_MWT_Team_Carousel());
        $widgets_manager->register(new \Elementor_MWT_Posts());
        $widgets_manager->register(new \Elementor_MWT_Testimonials_Carousel());
        $widgets_manager->register(new \Elementor_MWT_Products_Carousel());
        $widgets_manager->register(new \Elementor_MWT_Team_Box());
        $widgets_manager->register(new \Elementor_MWT_Banner());
        $widgets_manager->register(new \Elementor_MWT_Before_After());
        $widgets_manager->register(new \Elementor_MWT_Pricing_Box());
        $widgets_manager->register(new \Elementor_MWT_Counter_Up());
        $widgets_manager->register(new \Elementor_MWT_Events_Carousel());
        $widgets_manager->register(new \Elementor_MWT_Album());
    }


    public function frontend_styles()
    {
        wp_register_style('main-slider-style', plugins_url('../assets/css/main-slider.css', __FILE__));
        wp_register_style('team-slider-style', plugins_url('../assets/css/team-slider.css', __FILE__));
        wp_register_style('owl-carousel-style', plugins_url('../assets/css/owl-carousel.css', __FILE__));
        wp_register_style('gallery-carousel-style', plugins_url('../assets/css/gallery-carousel.css', __FILE__));
        wp_register_style('gallery-grid-style', plugins_url('../assets/css/gallery-grid.css', __FILE__));
        wp_register_style('posts-style', plugins_url('../assets/css/posts.css', __FILE__));
        wp_register_style('testimonials-carousel-style', plugins_url('../assets/css/testimonials-carousel.css', __FILE__));
        wp_register_style('team-carousel-style', plugins_url('../assets/css/team-carousel.css', __FILE__));
        wp_register_style('products-carousel-style', plugins_url('../assets/css/products-carousel.css', __FILE__));
        wp_register_style('team-box-style', plugins_url('../assets/css/team-box.css', __FILE__));
        wp_register_style('banner-style', plugins_url('../assets/css/banner.css', __FILE__));
        wp_register_style('before-after-style', plugins_url('../assets/css/before-after.css', __FILE__));
        wp_register_style('pricing-box-style', plugins_url('../assets/css/pricing-box.css', __FILE__));
        wp_register_style('counter-up-style', plugins_url('../assets/css/counter-up.css', __FILE__));
        wp_register_style('events-carousel-style', plugins_url('../assets/css/events-carousel.css', __FILE__));
        wp_register_style('album-style', plugins_url('../assets/css/album.css', __FILE__));


        wp_enqueue_style('main-slider-style');
        wp_enqueue_style('team-slider-style');
        wp_enqueue_style('owl-carousel-style');
        wp_enqueue_style('gallery-carousel-style');
        wp_enqueue_style('gallery-grid-style');
        wp_enqueue_style('posts-style');
        wp_enqueue_style('testimonials-carousel-style');
        wp_enqueue_style('team-carousel-style');
        wp_enqueue_style('products-carousel-style');
        wp_enqueue_style('team-box-style');
        wp_enqueue_style('banner-style');
        wp_enqueue_style('before-after-style');
        wp_enqueue_style('pricing-box-style');
        wp_enqueue_style('counter-up-style');
        wp_enqueue_style('events-carousel-style');
        wp_enqueue_style('album-style');
    }

    public function frontend_scripts()
    {
        wp_register_script('main-slider-script', plugins_url('../assets/js/main-slider.js', __FILE__), ['jquery']);
        wp_register_script('team-slider-script', plugins_url('../assets/js/team-slider.js', __FILE__), ['jquery']);
        wp_register_script('gallery-carousel-script', plugins_url('../assets/js/gallery-carousel.js', __FILE__), ['jquery']);
        wp_register_script('gallery-grid-script', plugins_url('../assets/js/gallery-grid.js', __FILE__), ['jquery']);
        wp_register_script('posts-script', plugins_url('../assets/js/posts.js', __FILE__), ['jquery']);
        wp_register_script('testimonials-carousel-script', plugins_url('../assets/js/testimonials-carousel.js', __FILE__), ['jquery']);
        wp_register_script('team-carousel-script', plugins_url('../assets/js/team-carousel.js', __FILE__), ['jquery']);
        wp_register_script('products-carousel-script', plugins_url('../assets/js/products-carousel.js', __FILE__), ['jquery']);
        wp_register_script('before-after-script', plugins_url('../assets/js/before-after.js', __FILE__), ['jquery']);
        wp_register_script('counter-up-script', plugins_url('../assets/js/counter-up.js', __FILE__), ['jquery']);
        wp_register_script('events-carousel-script', plugins_url('../assets/js/events-carousel.js', __FILE__), ['jquery']);

        wp_enqueue_script('main-slider-script');
        wp_enqueue_script('team-slider-script');
        wp_enqueue_script('gallery-carousel-script');
        wp_enqueue_script('gallery-grid-script');
        wp_enqueue_script('posts-script');
        wp_enqueue_script('testimonials-carousel-script');
        wp_enqueue_script('jquery-ui-slider');
        wp_enqueue_script('team-carousel-script');
        wp_enqueue_script('products-carousel-script');
        wp_enqueue_script('before-after-script');
        wp_enqueue_script('counter-up-script');
        wp_enqueue_script('events-carousel-script');
    }
}
